#if HAVE_NETDB_H
#include <stdio.h>
#include <stdlib.h>
#include <zlib.h>

#define STD_ERR 0
#define DEBUG_GZIP_MOST 0
#define N_PRINT 1

#define HEAD_CRC     0x02 /* bit 1 set: header CRC present */
#define EXTRA_FIELD  0x04 /* bit 2 set: extra field present */
#define ORIG_NAME    0x08 /* bit 3 set: original file name present */
#define COMMENT      0x10 /* bit 4 set: file comment present */
#define RESERVED     0xE0 /* bits 5..7: reserved */

static int check_gzip_header(const char *buffer, int buffer_length)
{
	unsigned int method;	/* method byte */
	unsigned int flags;	/* flags byte */
	unsigned int len;
	static const char gz_magic[2] = { 0x1f, 0x8b };
	const char *cur_pos = buffer;
	const char *tail_pos = buffer + buffer_length;
	char n2ch_mes[100];

	if (cur_pos + 4 >= tail_pos){
#if STD_ERR
		fprintf(stderr, "Too short for gzip header.\n");
#endif
		return -1;
	}

  /* Check the gzip magic header */
	for (len = 0; len < 2; len++){
		if (*cur_pos++ != gz_magic[len]){
#if STD_ERR
			fprintf(stderr, "No matching magic number.\n");
#endif
			return -1;
		}
	}

	method = *cur_pos++;
	flags = *cur_pos++;

#if DEBUG_GZIP_MOST
  fprintf(stderr, "flags: 0x%x\n", flags);
#endif
#if N_PRINT
  sprintf(n2ch_mes, "flags: 0x%x", flags);
#endif

	if (method != Z_DEFLATED || (flags & RESERVED) != 0)
		return -1;	/* Invalid header */

  /* Discard time, xflags and OS code: */
	cur_pos += 6;
	if (cur_pos >= tail_pos){
#if STD_ERR
		fprintf(stderr, "Too short for gzip header.\n");
#endif
		return -1;
	}

	if ((flags & EXTRA_FIELD) != 0){
		if (cur_pos + 2 >= tail_pos){
#if STD_ERR
			fprintf(stderr, "Too short for gzip header.\n");
#endif
			return -1; 
		}
#if DEBUG_GZIP_MOST
		fprintf(stderr, "skipping the extra field\n", cur_pos);
#endif
      /* skip the extra field */
		len = (unsigned int)*cur_pos++;
		len += ((unsigned int)*cur_pos++) << 8;
		cur_pos += len;

		if (cur_pos >= tail_pos){
#if STD_ERR
			fprintf(stderr, "Too short for gzip header.\n");
#endif
			return -1;
		}
	}

	if ((flags & ORIG_NAME) != 0){
#if DEBUG_GZIP_MOST
		fprintf(stderr, "ORIG_NAME = \"%s\"\n", cur_pos);
#endif
      /* skip the original file name */
		while (*cur_pos++ != '\0')
			if (cur_pos >= tail_pos){
#if STD_ERR
				fprintf(stderr, "Too short for gzip header.\n");
#endif
				return -1;
			}
	}

	if ((flags & COMMENT) != 0){
#if DEBUG_GZIP_MOST
		fprintf(stderr, "COMMENT = \"%s\"\n", cur_pos);
#endif
      /* skip the .gz file comment */
		while (*cur_pos++ != '\0')
			if (cur_pos >= tail_pos){
#if STD_ERR
				fprintf(stderr, "Too short for gzip header.\n");
#endif
				return -1;
			}
	}

	if ((flags & HEAD_CRC) != 0){
#if DEBUG_GZIP_MOST
		fprintf(stderr, "skipping CRC\n");
#endif
      /* skip the header crc */
		cur_pos += 2;
		if (cur_pos >= tail_pos){
#if STD_ERR
			fprintf(stderr, "Too short for gzip header.\n");
#endif
			return -1;
		}
	}

	return cur_pos - buffer;
}
int gzip_decompress(unsigned char *recv,int len,unsigned char *outbuf,int outlen)
{
	FILE *fp;

	z_stream z;
	int sep,status;
	char n2ch_mes[100];


	sep = check_gzip_header(recv,len);
#if DEBUG_GZIP_MOST
  fprintf(stderr, "header_sep: %d\n", sep);
#endif
#if N_PRINT
  sprintf(n2ch_mes, "header_sep: %d", sep);
#endif
	fp = fopen("test.gz","w");
	fwrite(recv,len,sizeof(char),fp);
	fclose(fp);

	z.zalloc = (alloc_func)Z_NULL;
	z.zfree = (free_func)Z_NULL;
	z.opaque = (voidpf)Z_NULL;

	z.next_in = recv + sep;
	z.avail_in = len - sep;

	if(inflateInit2(&z,-MAX_WBITS) != Z_OK){
#if STD_ERR
		printf("ZLIB init false:%s\n",(z.msg) ? z.msg : "??????");
#endif
#if N_PRINT
		sprintf(n2ch_mes,"ZLIB init false:%s",(z.msg) ? z.msg : "??????");
#endif
		return 0;
	}

	for(;;){
		z.next_out = outbuf;
		z.avail_out = outlen;

		status = inflate(&z,Z_SYNC_FLUSH);

		if(status == Z_STREAM_END) break;
		if(status != Z_OK){
#if STD_ERR
			printf("inflate:%s\n",(z.msg) ? z.msg : "??????");
#endif
#if N_PRINT
			sprintf(n2ch_mes,"inflate:%s",z.msg);
#endif
			return 0;
		}
	}

	if(inflateEnd(&z) != Z_OK){
#if STD_ERR
		printf("ZLIB end false:%s\n",(z.msg) ? z.msg : "??????");
#endif
#if N_PRINT
		sprintf(n2ch_mes,"ZLIB end false:%s",(z.msg) ? z.msg : "??????");
#endif
		return 0;
	}
#if DEBUG_GZIP_MOST
  fprintf(stderr, "size: %d\n",z.total_out);
#endif
#if N_PRINT
  sprintf(n2ch_mes, "size: %d",z.total_out);
#endif

	return z.total_out;
}
#endif
