// $Id$

//=============================================================================
/**
 *  @file  datetime.hpp
 *
 *  @author  Fukasawa Mitsuo
 *
 *    Copyright (C) 2001-2006 BEE Co.,Ltd. All rights reserved.
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License
 * as published by the Free Software Foundation; either version 2
 * of the License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.
 */
//=============================================================================

#ifndef BEE_DATETIME_H
#define BEE_DATETIME_H

#ifdef _MSC_VER
#pragma warning( disable: 4244 )
#endif

#include "bee/beeconf.hpp"
#include "boost/date_time/posix_time/posix_time.hpp"
using namespace boost::gregorian;
using namespace boost::posix_time;


namespace bee
{

//-----------------------------------------------------------------------------
// bee time class.
//-----------------------------------------------------------------------------
class BEE_Export datetime_base
{
    ptime tv_;

public:
    datetime_base(bool now = false) : tv_(date(1400,1,1))
    {
        if (now)
        {
            tv_ = second_clock::local_time();
            //tv_ = microsec_clock::local_time();
        }
    }
    datetime_base(const struct tm& tv) : tv_(date(1400,1,1)) { this->set(tv); }
    datetime_base(const std::string& tv)    : tv_(date(1400,1,1)) { this->set(tv); }
    datetime_base(const datetime_base& rhs) : tv_(rhs.tv_) {}
    ~datetime_base() {}

    datetime_base& operator=(const datetime_base& rhs)
    {
    	datetime_base tmp(rhs);
        std::swap(tv_, tmp.tv_);
        return *this;
    }

    datetime_base& operator=(const std::string& rhs)
    {
        this->set(rhs);
        return *this;
    }

    void toString(std::string& rs);
    std::string toString();
    std::string toName();

    inline void set(const struct tm& tv)
    {
        tv_ = datetime_base::create_time(const_cast<struct tm *>(&tv));
    }

    inline void set(const time_t& tv)
    {
        tv_ = datetime_base::create_time(localtime(&tv));
    }

    void set(const std::string& rhs);

    friend bool operator==(const datetime_base& lhs, const datetime_base& rhs);
    friend bool operator<(const datetime_base& lhs, const datetime_base& rhs);

    static unsigned int lapse(time_t curtime);
    static int  makeName(time_t curtime, std::string& tvname);
    static int  makeName(std::string& tvname);

    static ptime create_time(struct tm * current);
    static void  to_localtime(ptime pt, struct tm& retm, long& millisec);

    //
    // time of byte format
    //
    struct btime
    {
        bee::byte btv_[8]; // 0-1: year, 2:month, 3:day, 4:hour,
                           // 5:minutes, 6:second, 7:10msec

        btime() { memset(btv_, 0, sizeof(btv_)); }
        btime(const struct tm& rhs) { this->set(rhs); }
        btime(const btime& rhs)     { std::copy(&rhs.btv_[0], &rhs.btv_[8], btv_); }

        btime operator=(const btime& rhs)
        {
        	btime temp(rhs);
            std::copy(&temp.btv_[0], &temp.btv_[8], btv_);
            return *this;
        }

        void set(const struct tm& tv)
        {
            btv_[0] = (tv.tm_year + 1900) / 100;
            btv_[1] = (tv.tm_year + 1900) % 100;
            btv_[2] = tv.tm_mon + 1; btv_[3] = tv.tm_mday;
            btv_[4] = tv.tm_hour; btv_[5] = tv.tm_min;
            btv_[6] = tv.tm_sec;  btv_[7] = 0;
        }

        void set(const time_t& tv)
        {
            struct tm * when = localtime(&tv);
            set(*when);
            btv_[7] = 0;
        }
    };
};

// Operator function
inline bool operator==(const datetime_base& lhs, const datetime_base& rhs)
{
    return (lhs.tv_ == rhs.tv_);
}

inline bool operator<(const datetime_base& lhs, const datetime_base& rhs)
{
    return (lhs.tv_ < rhs.tv_);
}

inline bool operator==(const datetime_base::btime& lhs, const datetime_base::btime& rhs)
{
    if ((lhs.btv_[0] == rhs.btv_[0]) || (lhs.btv_[1] == rhs.btv_[1]) ||
        (lhs.btv_[2] == rhs.btv_[2]) || (lhs.btv_[3] == rhs.btv_[3]) ||
        (lhs.btv_[4] == rhs.btv_[4]) || (lhs.btv_[5] == rhs.btv_[5]) ||
        (lhs.btv_[6] == rhs.btv_[6]))
    {
        return true;
    }
    return false;
}

inline bool operator<(const datetime_base::btime& lhs, const datetime_base::btime& rhs)
{
    UINT lt = (lhs.btv_[0] << 24) + (lhs.btv_[1] << 16) +
              (lhs.btv_[2] << 8) + lhs.btv_[3];
    UINT rt = (rhs.btv_[0] << 24) + (rhs.btv_[1] << 16) +
              (rhs.btv_[2] << 8) + rhs.btv_[3];
    if (lt < rt)
        return true;
    else if (lt > rt)
        return false;

    lt = (lhs.btv_[4] << 16) + (lhs.btv_[5] << 8) + lhs.btv_[6];
    rt = (rhs.btv_[4] << 16) + (rhs.btv_[5] << 8) + rhs.btv_[6];
    if (lt < rt)
        return true;
    else if (lt > rt)
        return false;
    return false;
}


} // *namespace* - bee

#endif  // BEE_DATETIME_H
