// $Id$

//=============================================================================
/**
 *  @file  sequential_map.hpp
 *
 *  @author Fukasawa Mitsuo
 *
 *    Copyright (C) 2003-2006 BEE Co.,Ltd. All rights reserved.
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License
 * as published by the Free Software Foundation; either version 2
 * of the License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.
 */
//=============================================================================


#ifndef BEE_SEQUENTIAL_MAP_H
#define BEE_SEQUENTIAL_MAP_H

#include "bee/nameable.hpp"

namespace bee
{
//
// vector and map table
//
template<typename T, typename K, typename C, const K& (T::*getkey_func)() const >
struct sequential_map : std::vector< typename boost::shared_ptr<T> >
{
    typedef K            key_type;
    typedef T            value_type;
    typedef C            compare_type;
    // typedef value_type * pointer;
    typedef boost::shared_ptr<T> pointer;   
    typedef std::vector< typename boost::shared_ptr<T> > base_type;

    typedef std::map<key_type, int, compare_type> keymap_type;
    typedef typename std::map<key_type, int, compare_type>::iterator
            keymap_iterator;
    typedef typename std::map<key_type, int, compare_type>::const_iterator
            const_keymap_iterator;
    typedef typename std::iterator_traits< 
                typename std::map<key_type, int, compare_type>::iterator
                >::reference
            reference;

    sequential_map() {}
    sequential_map(const sequential_map& rhs)
        : base_type(rhs), map_(rhs.map_) {}
    ~sequential_map() {}

    sequential_map& operator=(const sequential_map& rhs)
    {
        if (this == &rhs)
            return *this;
        std::copy(rhs.begin(), rhs.end(), this->begin());
        map_ = rhs.map_;
        return *this;
    }

    value_type * find(const key_type& key) const
    {
        const_keymap_iterator iter = map_.find(key);
        if (iter == map_.end())
            return NULL;
        int idx = iter->second;
        //bee::assert(idx < (int)this->size());
        return this->base_type::at(idx).get();
    }

    value_type * get(int idx)
    {
        return ((size_t)idx >= this->size()) ? NULL : (*this)[idx].get();
    }

    int add(const pointer& elm)
    {
        this->push_back(elm);
        value_type * vp = elm.get();
        map_.insert(std::make_pair((vp->* getkey_func)(), this->size() - 1));
        return 0;
    }

    bool remove(const key_type& key)
    {
        keymap_iterator mapiter = map_.find(key);
        if (mapiter == map_.end())
            return false;
        unsigned int idx = mapiter->second;
        //bee::assert(idx < this->size());
        map_.erase(mapiter);
        if (idx >= this->size())
            return false;
       (*this)[idx].reset();

        return true;
    }

    value_type * remove(int idx)
    {
        if ((size_t)idx >= this->size())
            return NULL;
        value_type * result = (*this)[idx].get();
        this->remove((result->*getkey_func)());
        return result;
    }

protected:
    keymap_type map_;
};

//
// Nameable sequential map
//
typedef sequential_map<bee::nameable, std::string, std::less<std::string>,
            &bee::nameable::name_s> attributes_map;

} // *namespace* - bee

#endif  /* BEE_SEQUENTIAL_MAP_H */
