// $Id$

//=============================================================================
/**
 *  @file    trace_fileio.cpp
 *
 *  @author Fukasawa Mitsuo
 *
 *    Copyright (C) 2004-2006 BEE Co.,Ltd. All rights reserved.
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License
 * as published by the Free Software Foundation; either version 2
 * of the License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.
 */
//=============================================================================

#define BEE_BUILD_DLL

#include "bee/trace.hpp"
#include "bee/fileio.hpp"
#include "boost/date_time/posix_time/posix_time.hpp"
using namespace boost::gregorian;
using namespace boost::posix_time;
namespace fs = boost::filesystem;

#define ACE_MAXLOGMSGLEN  4096

namespace bee
{
#define TO_COUT 1

#ifdef TO_COUT
static std::ostream  * _fio = &std::cout;
#else
static boost::filesystem::fstream _fio_body;
static std::fstream  * _fio = &_fio_body;
#endif

//
// Output level of tracing 
//
static int _level = trace::L_LOW;

// initialize trace (static)
void trace::init(int lv, void * fname)
{
    _level = lv;
#ifndef TO_COUT
    _fio->open((fname != NULL) ? (const char *)fname : "bee-trace.log"); 
#endif
}

int trace::level()
{
    return _level;
}

inline std::string logging_time()
{
	ptime tv = microsec_clock::local_time();
	time_duration td = microseconds(500);
	td += tv.time_of_day();
    std::string buf = to_simple_string(td);
    return buf.substr(0, 12);
}

//
// Trace class
//
trace::trace(const BCHAR * fname, int lv)
    : function_(fname), file_(NULL), linenum_(0), level_(lv)
{
    if (lv < level())
    {
        *_fio << logging_time() << " " << function_ << std::endl;
    }
}

trace::trace(const BCHAR * funcname, const BCHAR * filename, int lnum, int lv)
    : function_(funcname), file_(filename), linenum_(lnum), level_(lv)
{
    if (lv < level())
    {
        *_fio << logging_time() << " " << file_ << "(" << linenum_ << "):" 
             << function_ << std::endl;
    }
}

trace::~trace()
{
    //if (lv < level())
    //{
    //    *_fio << logging_time() << " End of " << function_ << str::endl;
    //}
}

// Trace error message
void trace::error(const BCHAR * fmt, ...)
{
    va_list arglist;
    va_start(arglist, fmt);

    BCHAR buffer[ACE_MAXLOGMSGLEN];
    vsnprintf(buffer, ACE_MAXLOGMSGLEN-1, fmt, arglist);
    *_fio << logging_time() << " " << function_ << ": " << buffer << std::endl;

    va_end(arglist);
}

void trace::error(const boost::format& fmt)
{
    *_fio << logging_time() << " " << function_ << ": " << boost::str(fmt)
          << std::endl;
}

// Trace degug message
void trace::debug(const BCHAR * fmt, ...)
{
    va_list arglist;
    va_start(arglist, fmt);

    BCHAR buffer[ACE_MAXLOGMSGLEN];
    vsnprintf(buffer, ACE_MAXLOGMSGLEN-1, fmt, arglist);
    *_fio << logging_time() << " " << function_ << ": " << buffer << std::endl;

    va_end(arglist);
}

void trace::debug(const boost::format& fmt)
{
    *_fio << logging_time() << " " << function_ << ": " 
          << boost::str(fmt) << std::endl;
}

// Trace info message
void trace::notice(const BCHAR * fmt, ...)
{
    va_list arglist;
    va_start(arglist, fmt);

    BCHAR buffer[ACE_MAXLOGMSGLEN];
    vsnprintf(buffer, ACE_MAXLOGMSGLEN-1, fmt, arglist);
    *_fio << logging_time() << " " << function_ << ": " << buffer << std::endl;

    va_end(arglist);
}

void trace::notice(const boost::format& fmt)
{
    *_fio << logging_time() << " " << function_ << ": " 
          << boost::str(fmt) << std::endl;
}

}
  // *namespace* - bee

