// $Id$
//
// Copyright (C) 2006 BEE Co.,Ltd.
//
// Author : Fukasawa Mitsuo
//
// This program is free software; you can redistribute it and/or modify
// it under the terms of the GNU General Public License version 2 as
// published by the Free Software Foundation.
//
// This program is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
// GNU General Public License for more details.
//
// You should have received a copy of the GNU General Public License
// along with this program; if not, write to the Free Software
// Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA 02110-1301 USA
//

#include "bee/booster.hpp"
#include "lexicon.hpp"
#include "xml.hxx"

#include <boost/program_options.hpp>
using namespace boost;
namespace po = boost::program_options;
namespace fs = boost::filesystem;
using std::cerr;
using std::endl;

extern void init_lexicon(int argc, char* argv[]);
extern void fini_lexicon();
extern pls::lexicon * make_lexicon(const std::string& fname);

// grobal variables
std::string opt_infile("");
std::string opt_outfile("");
std::string opt_outdir("");
fs::path _outfile;
bool opt_tree = false;
int  opt_trace = 4;

bool getopt_tree()
{
    return opt_tree;
}

int getopt_trace()
{
    return opt_trace;
}

const char * getopt_outfile()
{
    return _outfile.string().c_str();
}

const char * getopt_infile()
{
    return opt_infile.c_str();
}
//
// parse command options
//
int parse_options(int argc, char * argv[])
{
    try
    {
        //po::options_description desc("Allowed options");
        po::options_description desc("Option");
        //    ("help", "produce help message")
        desc.add_options()
            ("help,h", "Help")
            ("trace,t", po::value<int>(&opt_trace)->default_value(4), 
                  "Trace Level(0..7)")
            ("output-directory,D", po::value<std::string>(&opt_outdir),
                  "Output Directory")
            ("tree,T", "Enable Tree")
            ("output-file,o", po::value<std::string>(&opt_outfile), "Output File(.txt)")
            ("input-file,i", po::value<std::string>(&opt_infile), "Input File(.grxml)")
        ;

        po::positional_options_description p;
        p.add("input-file", -1);
        
        po::variables_map vm;
        po::store(po::command_line_parser(argc, argv).
                  options(desc).positional(p).run(), vm);
        po::notify(vm);
   
        if (argc < 2 || vm.count("help")) 
        {
            std::cout << "Usage: " << argv[0] << " [options] input-file\n";
            std::cout << desc;
            return -1;
        }

        if (vm.count("output-directory"))
        {
            opt_outdir = vm["output-directory"].as<std::string>();
            if (! fs::exists(opt_outdir))
            {
                std::cerr << "Directory not found: " << opt_outdir << "\n";
                return -1;
            }
            if (! fs::is_directory(opt_outdir))
            {
                std::cerr << "\"" << opt_outdir << "\" is not directry.\n";
                return -1;
            }
        }
        if (vm.count("output-file"))
        {
            opt_outfile = vm["output-file"].as<std::string>();
            if (fs::exists(opt_outfile) && fs::is_directory(opt_outfile))
            {
                std::cerr << "\"" << opt_outfile << "\" is directry.\n";
                return -1;
            }
            //else if (fs::exists(opt_outfile))
            //{
            //    std::cerr << "Warning: over write \"" << opt_outfile << "\".\n";
            //}
        }
        if (vm.count("tree"))
        {
            opt_tree = true; 
        }

        // check input file
        if (vm.count("input-file"))
        {
            opt_infile = vm["input-file"].as<std::string>();
        }
        if (! fs::exists(opt_infile))
        {
            std::cerr << "\"" << opt_infile << "\" is not found.\n";
            return -1;
        }
        if (fs::is_directory(opt_infile))
        {
            std::cerr << "\"" << opt_infile << "\" is directry.\n";
            return -1;
        }
        // output file
        if (opt_outfile.empty())
        {
            size_t p = opt_infile.rfind('.');
            if (p == std::string::npos)
                opt_outfile = opt_infile;
            else
                opt_outfile = opt_infile.substr(0, p);
            opt_outfile += ".txt";
        }
        if (opt_outdir.size() > 0)
        {
            _outfile = opt_outdir;
            _outfile /= opt_outfile;
        }
        else
        {
            _outfile = opt_outfile;
        }
        std::cout << "output file is \"" << opt_outfile << "\"\n";
    }
    catch (std::exception& e)
    {
        std::cout << e.what() << "\n";
        return -1;
    }    
    return 0;
}

//
// Main procedure
//
int main(int argc, char* argv[])
{
    // get command option
    //if (parse_options(argc, argv) < 0)
    //{
    //    return 1;
    //}
   
    try
    {
        init_lexicon(argc, argv);
        pls::lexicon * lexicon = make_lexicon(argv[1]);
        if (lexicon != NULL)
        {      
            // Let's print what we've got.
            lexicon->dump(std::cerr);
        }
    }
    catch (const pls::exception& e)
    {
        std::cerr << e.what() << std::endl;
        fini_lexicon();
        return 1;
    }
    catch (const xml_schema::parser::exception& e)
    {
        std::cerr << e << std::endl;
        fini_lexicon();
        return 1;
    }
    catch (const std::ios_base::failure&)
    {
        std::cerr << "io failure" << std::endl;
        fini_lexicon();
        return 1;
    }
    catch (const std::exception& e)
    {
        std::cerr << e.what() << std::endl;
        fini_lexicon();
        return 1;
    }
    fini_lexicon();
    return 0;
}


