<?php
//  $Revision: 1.6.2.1 $                                                                  //
//  --------------------------------------------------------------------------  //
//  XooNIps Xoops modules for Neuroinformatics Platforms                        //
//  Copyright (C) 2005-2007 RIKEN, Japan. All rights reserved.                  //
//  http://sourceforge.jp/projects/xoonips/                                     //
//  --------------------------------------------------------------------------  //
//  This program is free software; you can redistribute it and/or               //
//  modify it under the terms of the GNU General Public License                 //
//  as published by the Free Software Foundation; either version 2              //
//  of the License, or (at your option) any later version.                      //
//                                                                              //
//  This program is distributed in the hope that it will be useful,             //
//  but WITHOUT ANY WARRANTY; without even the implied warranty of              //
//  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the               //
//  GNU General Public License for more details.                                //
//                                                                              //
//  You should have received a copy of the GNU General Public License           //
//  along with this program; if not, write to the Free Software                 //
//  Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA  02111-1307, USA. //
//  --------------------------------------------------------------------------  //

include_once "importobject.php";

/**
 * summary
 * 
 * - parse given xml(<FILE> part)
 * - retrieve attachment file from specified url and store in temporary directory
 * - retrieve attachment file from local file system
 * - move attachment files from temporary directory or local file system to upload_dir
 * - insert file information into database
 * - if file_type_name is 'preview', it creates thumbnail image automatically if <thumbnail> is not given in XML.
 * 
 */

class XooNIpsAttachment extends XooNIpsImportObject{

    /**
     * 
     * file path where file content is stored
     * 
     */
    var $filepath = null;
    
    /**
     * 
     * file path where the file is placed in
     * 
     */
    var $upload_dir = null;
    
    /**
     * 
     * file path where the attachments to be imported are placed in
     * 
     */
    var $attachment_dir = null;
    
    /**
     *
     *
     *
     */
    var $maxlen_xoonips_file = null;
    
    /**
     * 
     * @param attachment_dir path of the attachments to be imported has been placed in
     * @param upload_dir path of the attachments to be stored in
     * 
     * attachment_dir is base directory of file_name attribute
     * 
     */
    function XooNIpsAttachment( $attachment_dir, $upload_dir ){
        $this -> attachment_dir = $attachment_dir;
        $this -> upload_dir = $upload_dir;
        
        $this -> maxlen_xoonips_file = $this -> getColumnMaxLen( 'xoonips_file' );

        parent::XoopsObject();
        $this->initVar('file_id', XOBJ_DTYPE_INT, null, false);
        $this->initVar('item_id', XOBJ_DTYPE_INT, null, false);
        $this->initVar('original_file_name', XOBJ_DTYPE_TXTBOX, null, true);
        $this->initVar('file_name', XOBJ_DTYPE_TXTBOX, null, true);
        $this->initVar('file_size', XOBJ_DTYPE_INT, null, false);
        $this->initVar('thumbnail_file', XOBJ_DTYPE_OTHER, null, true, 65535 );
        $this->initVar('caption', XOBJ_DTYPE_TXTBOX, null, true);
        $this->initVar('mime_type', XOBJ_DTYPE_TXTBOX, null, false);
        $this->initVar('file_type_id', XOBJ_DTYPE_INT, null, true);
        $this->initVar('sess_id', XOBJ_DTYPE_TXTBOX, null, true);
        $this->initVar('search_module_name', XOBJ_DTYPE_TXTBOX, null, true);
        $this->initVar('search_module_version', XOBJ_DTYPE_TXTBOX, null, true);
        $this->initVar('header', XOBJ_DTYPE_TXTBOX, null, true);
        $this->initVar('timestamp', XOBJ_DTYPE_INT, null, true);
        
        parent::setNew();
    }
    
    function startElement($parser, $name, $attribs){
        global $xoopsDB, $xoopsUser;
        
        parent::startElement($parser, $name, $attribs);
        
        //echo "\nS ".implode( '/', $this -> tagstack )."\tattributes=";print_r($attribs);
        switch( implode( '/', $this -> tagstack ) ){
        case 'FILE/CAPTION':
            break; 
        case 'FILE/THUMBNAIL':
            break;
        case 'FILE':
            //
            // required attributes
            //
            if( !isset( $attribs[ 'FILE_NAME' ] ) && !isset( $attribs[ 'URL' ] ) ){
                $this -> setErrors( E_XOONIPS_ATTR_NOT_FOUND, 'FILE_NAME and URL are not found.' . $this -> getParserErrorAt( $parser ) );
            }
            
            if( isset( $attribs[ 'FILE_NAME' ] ) && isset( $attribs[ 'URL' ] ) ){
                $this -> setErrors( E_XOONIPS_ATTR_REDUNDANT, 'FILE_NAME and URL must be used exclusively.' . $this -> getParserErrorAt( $parser ) );
            }
            
            $assoc = array();
            foreach( array( 'ITEM_ID', 'FILE_SIZE', 'FILE_TYPE_NAME', 'ORIGINAL_FILE_NAME', 'MIME_TYPE' ) as $name ){
                if( isset( $attribs[ $name ] ) ){
                    $assoc[ strtolower( $name ) ] = $attribs[ $name ];
                }
            }
            
            $result = $this -> stripSurplusString( $this -> maxlen_xoonips_file, $assoc );
            foreach( $result as $key => $val ){
                $this -> setErrors( E_XOONIPS_DATA_TOO_LONG, "${key} is too long :" . $assoc[ $key ] . $this -> getParserErrorAt( $parser ) );
                //
                //update $assoc by stripped string
                //
                $assoc[ $key ] = $val;
            }
            
            //
            // test numerical attributes
            //
            foreach( array( 'ITEM_ID', 'FILE_SIZE' ) as $name ){
                if( !isset( $assoc[ strtolower( $name ) ] ) ) $this -> setErrors( E_XOONIPS_ATTR_NOT_FOUND, "${name} is not found". $this -> getParserErrorAt( $parser ) );
                else if( empty( $assoc[ strtolower( $name ) ] ) ) $this -> setErrors( E_XOONIPS_ATTR_INVALID_VALUE, "${name} is empty". $this -> getParserErrorAt( $parser ) );
                else if( !ctype_digit( $assoc[ strtolower( $name ) ] ) ) $this -> setErrors( E_XOONIPS_ATTR_INVALID_VALUE, "${name} is not integer(" . $assoc[ strtolower( $name ) ] . ")". $this -> getParserErrorAt( $parser ) );
                else $this -> setVar( strtolower( $name ), intval( $assoc[ strtolower( $name ) ] ) );
            }

            //
            // test literal attributes
            //
            foreach( array( 'FILE_TYPE_NAME', 'ORIGINAL_FILE_NAME', 'MIME_TYPE' ) as $name ){
                if( !isset( $assoc[ strtolower( $name ) ] ) ) $this -> setErrors( E_XOONIPS_ATTR_NOT_FOUND, "${name} is not found". $this -> getParserErrorAt( $parser ) );
                else if( empty( $assoc[ strtolower( $name ) ] ) ) $this -> setErrors( E_XOONIPS_ATTR_INVALID_VALUE, "${name} is empty". $this -> getParserErrorAt( $parser ) );
                else $this -> setVar( strtolower( $name ), encodeMeta2Server( $assoc[ strtolower( $name ) ] ) );
            }
            
            // 
            // retrieve attachment file content
            // 
            // - attribute 'file_name' have relative path based on zip archive where attachment file is placed in
            // - attribute 'url' have URL(http:// or file://) where attachment file is placed in
            // - only administrator can acces URL using 'file://'.
            // 
            if( isset( $attribs[ 'FILE_NAME' ] ) ){
                if( is_null( $this -> attachment_dir ) ) die( 'invalid attachment_dir is given(${attachment_dir})' . $this -> getErrorAt( __LINE__, __FILE__, __FUNCTION__ ) );
                $this -> get_file_content( $this -> attachment_dir . "/" . $attribs[ 'FILE_NAME' ] );
            }else if( isset( $attribs[ 'URL' ] ) ){
                if( strncmp( 'file://', $attribs[ 'URL' ], 7 ) != 0 && strncmp( 'http://', $attribs[ 'URL' ], 7 ) != 0 ){
                    $this -> setErrors( E_XOONIPS_ATTR_INVALID_VALUE, "illegal url(${attribs[ 'URL' ]})". $this -> getParserErrorAt( $parser ) );
                    break;
                }
                if( strncmp( 'file://', $attribs[ 'URL' ], 7 ) == 0 ){
                    if( $xoopsUser -> isAdmin() || isset( $_SESSION['xoonips_old_uid'] ) ){
                        $this -> get_url_content( $attribs[ 'URL' ] );
                    }else{
                        $this -> setErrors( E_XOONIPS_NOT_PERMITTED_ACCESS, "You don't have permission to open url(${attribs[ 'URL' ]})". $this -> getParserErrorAt( $parser ) );
                    }
                }else{
                    $this -> get_url_content( $attribs[ 'URL' ] );
                }
            }
            
            // 
            // retrieve file_type_id by file_type_name
            // 
            if( isset( $assoc[ 'file_type_name'] ) ){
                $sql = "SELECT file_type_id FROM " . $xoopsDB -> prefix( "xoonips_file_type" ) . " WHERE name=" . $xoopsDB -> quoteString( $assoc['file_type_name'] );
                $result = $xoopsDB -> query( $sql );
                if( !$result ){
                    $this -> setErrors( E_XOONIPS_DB_QUERY, $xoopsDB -> error() . $this -> getParserErrorAt( $parser ) );
                }else if( $xoopsDB -> getRowsNum( $result ) > 0 ){
                    list( $id ) = $xoopsDB->fetchRow( $result );
                    $this -> setVar( 'file_type_id', $id );
                }else{
                    $this -> setErrors( E_XOONIPS_ATTR_NOT_FOUND, "file_type_id is not found(" . $assoc['file_type_name'] . ")" . $this -> getParserErrorAt( $parser ) );
                }
            }
            break;
        default:
            $this -> setErrors( E_XOONIPS_PARSER, "unknown element '$name'" . $this -> getErrorAt( __LINE__, __FILE__, __FUNCTION__ ) );
            break;
        }
    }
    
    function endElement($parser, $name) { 
        global $xoopsDB;
//        echo "\nE ".implode( '/', $this -> tagstack );
        switch( implode( '/', $this -> tagstack ) ){
        case 'FILE/CAPTION':
            $this -> setVar( 'caption', encodeMeta2Server( end( $this -> cdata ) ) );
            break; 
        case 'FILE/THUMBNAIL':
            $this -> setVar( 'thumbnail_file', base64_decode( end( $this -> cdata ) ) );
            break;
        case 'FILE':
            // 
            // create thumbnail image if file_type is preview and <thumbnail> is not given
            // 
            $result = $xoopsDB -> query( "select * from " . $xoopsDB -> prefix( "xoonips_file_type" )
                                         . " where name = 'preview' AND file_type_id=" . $this -> getVar( 'file_type_id' ) );
            if( $result && $xoopsDB -> getRowsNum( $result ) == 1 ){
                $thumbnail = $this -> getVar( 'thumbnail_file' );
                if( empty( $thumbnail ) ){
                    $this -> createThumbnail();
                }
            }else{
            }
            break;
        default:
            $this -> setErrors( E_XOONIPS_PARSER, "unknown element '$name'" . $this -> getErrorAt( __LINE__, __FILE__, __FUNCTION__ ) );
            break;
        }
        parent::endElement($parser, $name);
    }
    
    function characterData($parser, $data) {
        parent::characterData($parser, $data);
//        echo "\nC ".implode( '/', $this -> tagstack )."\t${data}";
        switch( implode( '/', $this -> tagstack ) ){
        case 'FILE/CAPTION':
            break; 
        case 'FILE/THUMBNAIL':
            break;
        case 'FILE':
            break;
        }    
    }
    
    /**
     * 
     * get a file content using HTTP
     * (content is stored into a local file of path '$this->filepath')
     * 
     * @param url where to get from
     * 
     * @return true if succeed, false if failed
     * 
     */
    function get_url_content( $url ){
        if( !ini_get( 'allow_url_fopen' ) ) {
            die( 'allow_url_fopen must be true' );
        }
        
        if( empty( $this -> filepath ) ){
            $this -> filepath = tempnam( $this -> upload_dir, "XNP" );
        }
        
        $src = fopen( $url, "rb" );
        if( !$src ){
            $this -> setErrors( E_XOONIPS_OPEN_FILE, "can't open file for read(${url})" . $this -> getErrorAt( __LINE__, __FILE__, __FUNCTION__ ) );
            return false;
        }
        
        $dst = fopen( $this -> filepath, "wb" );
        if( !$dst ){
            $this -> setErrors( E_XOONIPS_OPEN_FILE, "can't open file for write(" . $this -> filepath . ")" . $this -> getErrorAt( __LINE__, __FILE__, __FUNCTION__ ) );
            return false;
        }
        
        while( !feof( $src ) ){
            $result = fread( $src, 65536 );
            if( FALSE === $result ){
                $this -> setErrors( E_XOONIPS_OPEN_FILE, "read error of " . $ur . $this -> getErrorAt( __LINE__, __FILE__, __FUNCTION__ ) );
                fclose( $src );
                fclose( $dst );
                @unlink( $dst );
                return false;
            }
            if( FALSE === fwrite( $dst, $result ) ){
                $this -> setErrors( E_XOONIPS_OPEN_FILE, "write error of " . $this -> filepath . $this -> getErrorAt( __LINE__, __FILE__, __FUNCTION__ ) );
                fclose( $src );
                fclose( $dst );
                @unlink( $dst );
                return false;
            }
        }
        fclose( $src );
        fclose( $dst );
        return true;
    }
    
    /**
     * 
     * get a file content from local file system
     * (check file existence only)
     * 
     * @param suorce file path where to get from
     * 
     * @return true if succeed, false if failed
     * 
     */
    function get_file_content( $source ){
        $result = file_exists( $source );
        if( $result ){
            $this -> filepath = $source;
            return true;
        }
        $this -> setErrors( E_XOONIPS_OPEN_FILE, "can't open file(${source})". $this -> getErrorAt( __LINE__, __FILE__, __FUNCTION__ ) );
        return false;
    }
    
    /** 
     * 
     * return file path of file content.
     * return false if file can't be retrieved.
     * 
     */
    function getFilePath(){
        return $this -> filepath;
    }

    /**
     * insert/update object.
     * Note: It set new file_id in insert and update. 
     *
     * @param file $file
     */
    function insert()
    {
        global $xoopsDB;
        
        $sql = "INSERT INTO " . $xoopsDB->prefix('xoonips_file') . 
            " ( item_id, original_file_name, mime_type, file_size, thumbnail_file, caption, sess_id, file_type_id, search_module_name, search_module_version, header, timestamp ) ".
            " values ( "
            . $this -> getVar( 'item_id' ) . ", "
            . $xoopsDB -> quoteString( $this -> getVar( 'original_file_name' ) ) . ", "
            . $xoopsDB -> quoteString( $this -> getVar( 'mime_type' ) ) . ", "
            . $this -> getVar( 'file_size' ) . ", "
            . ( is_null( $this -> getVar( 'thumbnail_file' ) ) ? 'null' : $xoopsDB -> quoteString( $this -> getVar( 'thumbnail_file' ) ) ) . ", "
            . ( is_null( $this -> getVar( 'caption' ) ) ? 'null' : $xoopsDB -> quoteString( $this -> getVar( 'caption' ) ) ) . ", "
            . 'null,' //sess_id of uploaded completely file must be null 
            . $this -> getVar( 'file_type_id' ) . ", "
            . ( is_null( $this -> getVar( 'search_module_name' ) ) ? 'null' : $xoopsDB -> quoteString( $this -> getVar( 'search_module_name' ) ) ) . ", "
            . ( is_null( $this -> getVar( 'search_module_version' ) ) ? 'null' : $xoopsDB -> quoteString( $this -> getVar( 'search_module_version' ) ) ) . ", "
            . ( is_null( $this -> getVar( 'header' ) ) ? 'null' : $xoopsDB -> quoteString( $this -> getVar( 'header' ) ) ) . ", "
            . ( is_null( $this -> getVar( 'timestamp' ) ) ? 'null' : $this -> getVar( 'timestamp' ) )
            . " )";
        
        $result = $xoopsDB->query($sql);
        if (!$result) {
            $this -> setErrors( E_XOONIPS_DB_QUERY, $xoopsDB -> error() . $this -> getErrorAt( __LINE__, __FILE__, __FUNCTION__ ) );
            return false;
        }else{
            $inserted_file_id = $xoopsDB -> getInsertId();

            //
            // copy file( temporary name -> file_id )
            //
            if ( !copy( $this -> filepath, $this -> upload_dir . "/" . $inserted_file_id ) ){
                $this -> setErrors( E_XOONIPS_ERROR, "rename " . $this -> filepath . " to " . $this -> upload_dir . "/" . $inserted_file_id . " failed. " . $this -> getErrorAt( __LINE__, __FILE__, __FUNCTION__ ) );
                return false;
            }
            
            // 
            // create search index of file if supported
            // 
            $modules = xnpGetFileSearchModules();
            list( $result, $errorMessage ) = xnpExtractText( $inserted_file_id, $modules );
        }
        if ($this->isNew()) {
            $this -> assignVar( 'file_id', $inserted_file_id );
            $this -> unsetNew();
        }
        return true;
    }
    
    /**
     * delete an item from the database
     * 
     * @param object $item reference to the item to delete
     * @return false if failed.
     */
    function delete(){
        global $xoopsDB;
        
        // remove old attachment information from xoonips_file table. real files will be removed in xnpCleanup().
        // don't remove real files now. they should be remain if import failed.
        $sql = "UPDATE " . $xoopsDB->prefix("xoonips_file") . " set item_id = NULL, sess_id = NULL where item_id=" . $this -> getVar( 'item_id' ) . " AND file_id=" . $this -> getVar( 'file_id' );
        $result = $xoopsDB->query($sql);
        if (!$result) {
            $this -> setErrors( E_XOONIPS_DB_QUERY, $xoopsDB -> error() . $this -> getErrorAt( __LINE__, __FILE__, __FUNCTION__ ) );
            return false;
        }
        return true;
    }
    
    function cleanVars(){
        $maxlen = $this -> getColumnMaxLen( "xoonips_file" );
        $result = true;// return value
        
        $vars = array( );
        foreach( array( 'file_id', 'item_id', 'original_file_name', 'file_name', 'file_size', 'thumbnail_file', 
                        'caption', 'mime_type', 'file_type_id', 'sess_id', 'search_module_name', 'search_module_version',
                        'header', 'timestamp' ) as $name ){
            $vars[ $name ] = $this -> getVar( $name );
        }
        $assoc = $this -> stripSurplusString( $maxlen, $vars, _CHARSET );
        foreach( $assoc as $name => $val ){
            $result = false;
            $this -> cleanVars[ $name ] = $val;
            $this -> setErrors( E_XOONIPS_DATA_TOO_LONG, "${name} is too long :" . $assoc[ $name ] . $this -> getErrorAt( __LINE__, __FILE__, __FUNCTION__ ) );
        }
        
        return $result;
    }
    
    function toString($htmlspecialchars = false){
        $text = array();
        foreach( array( 'file_id', 'item_id', 'original_file_name', 'file_name', 'file_size',/* 'thumbnail_file', */
                        'caption', 'mime_type', 'file_type_id', 'sess_id', 'search_module_name', 'search_module_version',
                        'header', 'timestamp' ) as $name ){
            $text[] = "detail.file." . $name . " " . ( $htmlspecialchars ? htmlspecialchars( $this -> getVar( $name ) ) : $this -> getVar( $name ) );
        }
        
        return implode( "\n", $text );
    }
   
    /** make thumbnail image from own properties(file_name, mime_type, original_file_name).
     * thumbnail image is stored to 'thumbnai_file' property in png format
     * @return bool true if succeed. false if some errors(see also getErrors())
     * @see xnpCreateThumbnail
     */
    function createThumbnail( ){
        $file = array( 'tmp_name' => $this -> filepath,
                       'name' => $this -> getVar( 'original_file_name' ),
                       'type' => $this -> getVar( 'mime_type' ) );
        list( $thumbnail, $error ) = xnpCreateThumbnail( $file );
        if( !$error ){
            // 
            // succeed
            // 
            $this -> setVar( 'thumbnail_file', $thumbnail );
            return true;
        }else{
            $this -> setErrors( E_XOONIPS_ERROR, $error . "(" . $this -> getVar( 'original_file_name' ) . ")" . $this -> getErrorAt( __LINE__, __FILE__, __FUNCTION__ ) );
            return false;
        }
    }
}
?>
