<?php
//  $Revision: 1.17.2.2 $                                                                  //
//  --------------------------------------------------------------------------  //
//  XooNIps Xoops modules for Neuroinformatics Platforms                        //
//  Copyright (C) 2005 RIKEN, Japan. All rights reserved.                       //
//  http://sourceforge.jp/projects/xoonips/                                     //
//  --------------------------------------------------------------------------  //
//  This program is free software; you can redistribute it and/or               //
//  modify it under the terms of the GNU General Public License                 //
//  as published by the Free Software Foundation; either version 2              //
//  of the License, or (at your option) any later version.                      //
//                                                                              //
//  This program is distributed in the hope that it will be useful,             //
//  but WITHOUT ANY WARRANTY; without even the implied warranty of              //
//  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the               //
//  GNU General Public License for more details.                                //
//                                                                              //
//  You should have received a copy of the GNU General Public License           //
//  along with this program; if not, write to the Free Software                 //
//  Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA  02111-1307, USA. //
//  --------------------------------------------------------------------------  //

include_once "importobject.php";
include_once XOOPS_ROOT_PATH . "/modules/xoonips/condefs.php";
include_once XOOPS_ROOT_PATH . "/modules/xoonips/include/encode.php";

class XooNIpsItem extends XooNIpsImportObject
{
    /**
     * 
     * temporary variable for open_level attribute of <index> 
     * 
     * open_level= public|group|private|null
     * null if not specified.
     * 
     */
    var $open_level = null;
    
    function XooNIpsItem(){
        parent::XooNIpsImportObject();
        
        $this->initVar('item_id', XOBJ_DTYPE_INT, null, false);
        $this->initVar('pseudo_id', XOBJ_DTYPE_INT, null, true);
        $this->initVar('titles', XOBJ_DTYPE_ARRAY, serialize( array() ), true);
        $this->initVar('keywords', XOBJ_DTYPE_ARRAY, serialize( array() ), true);
        $this->initVar('uid', XOBJ_DTYPE_TXTBOX, null, true);
        $this->initVar('creation_date', XOBJ_DTYPE_INT, null, true);
        $this->initVar('description', XOBJ_DTYPE_TXTAREA, null, true);
        $this->initVar('last_update_date', XOBJ_DTYPE_INT, null, false);
        $this->initVar('publication_year', XOBJ_DTYPE_INT, null, true);
        $this->initVar('publication_month', XOBJ_DTYPE_INT, null, true);
        $this->initVar('publication_mday', XOBJ_DTYPE_INT, null, true);
        $this->initVar('related_to', XOBJ_DTYPE_ARRAY, serialize( array() ), true);
        $this->initVar('item_type_id', XOBJ_DTYPE_INT, null, false);
        $this->initVar('lang', XOBJ_DTYPE_TXTBOX, 'eng', true);
        $this->initVar('doi', XOBJ_DTYPE_TXTBOX, null, true);
        $this->initVar('url', XOBJ_DTYPE_TXTBOX, null, true);
        $this->initVar('indexes', XOBJ_DTYPE_ARRAY, serialize( array() ), true);
        $this -> tag_path_count = array();
    }

    /**
     * 
     * return true if title, uid, itemtype are matched
     * 
     * 
     * @param $item
     * @return true if matched
     */
    function equals( $item ){
        if( $this -> getVar( 'uid' ) != $item -> getVar( 'uid' ) ) return false;
        if( $this -> getVar( 'item_type_id' ) != $item -> getVar( 'item_type_id' ) ) return false;
        
        $this_title = $this -> getVar( 'titles' );
        $item_title = $item -> getVar( 'titles' );
        if( count( $this_title ) != count( $item_title ) ) return false;
        
        foreach( $item_title as $title ){
            if( FALSE === array_search( $title, $this_title ) ) return false;
        }
        return true;
    }
    
    function startElement($parser, $name, $attribs){
        global $xoopsDB;
        
        parent::startElement($parser, $name, $attribs);
//        echo "\nS ".implode( '/', $this -> tagstack )."\tattributes=";print_r($attribs);
        $path = implode( '/', $this -> tagstack );
        if( count( $this -> tagstack ) >= 2 && $this -> tagstack[0] == 'ITEM' 
            && ( $this -> tagstack[1] == 'BASIC' || $this -> tagstack[1] == 'DETAIL' && count( $this -> tagstack ) == 2 )
            && isset( $this -> tag_path_count[ $path ] ) ){
            $this -> tag_path_count[ $path ][ 'count' ]++;
            if ( $this -> tag_path_count[ $path ][ 'multiple' ] == false && $this -> tag_path_count[ $path ][ 'count' ] == 2 ){
                $this -> setErrors( E_XOONIPS_TAG_REDUNDANT, "multiple " . $this -> tag_path_count[ $path ][ 'name' ] . " tags not allowed" . $this -> getParserErrorAt( $parser ) );
            }
        }
        switch( $path ){
        case 'ITEM':
            $this -> tag_path_count[ 'ITEM/BASIC'                   ] = array( 'name' => 'BASIC'            , 'required' => true , 'multiple' => false, 'count' => 0 );
            $this -> tag_path_count[ 'ITEM/DETAIL'                  ] = array( 'name' => 'DETAIL'           , 'required' => true , 'multiple' => false, 'count' => 0 );
            break;
        case 'ITEM/BASIC':
            if( !isset( $attribs['ID'] ) ) $this -> setErrors( E_XOONIPS_ATTR_NOT_FOUND, "ID is not declared". $this -> getParserErrorAt( $parser ) );
            if( empty( $attribs['ID'] ) ) $this -> setErrors( E_XOONIPS_ATTR_INVALID_VALUE, "ID is empty". $this -> getParserErrorAt( $parser ) );
            else if( !ctype_digit( $attribs['ID'] ) ) $this -> setErrors( E_XOONIPS_ATTR_INVALID_VALUE, "ID is not integer(${attribs['ID']})". $this -> getParserErrorAt( $parser ) );
            else $this -> setVar( 'pseudo_id', intval( $attribs['ID'] ) );
            $this -> tag_path_count[ 'ITEM/BASIC/ITEMTYPE'          ] = array( 'name' => 'ITEMTYPE'         , 'required' => true , 'multiple' => false, 'count' => 0 );
            $this -> tag_path_count[ 'ITEM/BASIC/CONTRIBUTOR'       ] = array( 'name' => 'CONTRIBUTOR'      , 'required' => true , 'multiple' => false, 'count' => 0 );
            $this -> tag_path_count[ 'ITEM/BASIC/TITLES'            ] = array( 'name' => 'TITLES'           , 'required' => true , 'multiple' => false, 'count' => 0 );
            $this -> tag_path_count[ 'ITEM/BASIC/KEYWORDS'          ] = array( 'name' => 'KEYWORDS'         , 'required' => false, 'multiple' => false, 'count' => 0 );
            $this -> tag_path_count[ 'ITEM/BASIC/DESCRIPTION'       ] = array( 'name' => 'DESCRIPTION'      , 'required' => true , 'multiple' => false, 'count' => 0 );
            $this -> tag_path_count[ 'ITEM/BASIC/DOI'               ] = array( 'name' => 'DOI'              , 'required' => true , 'multiple' => false, 'count' => 0 );
            $this -> tag_path_count[ 'ITEM/BASIC/LAST_UPDATE_DATE'  ] = array( 'name' => 'LAST_UPDATE_DATE' , 'required' => true , 'multiple' => false, 'count' => 0 );
            $this -> tag_path_count[ 'ITEM/BASIC/CREATION_DATE'     ] = array( 'name' => 'CREATION_DATE'    , 'required' => true , 'multiple' => false, 'count' => 0 );
            $this -> tag_path_count[ 'ITEM/BASIC/PUBLICATION_YEAR'  ] = array( 'name' => 'PUBLICATION_YEAR' , 'required' => true , 'multiple' => false, 'count' => 0 );
            $this -> tag_path_count[ 'ITEM/BASIC/PUBLICATION_MONTH' ] = array( 'name' => 'PUBLICATION_MONTH', 'required' => true , 'multiple' => false, 'count' => 0 );
            $this -> tag_path_count[ 'ITEM/BASIC/PUBLICATION_MDAY'  ] = array( 'name' => 'PUBLICATION_MDAY' , 'required' => true , 'multiple' => false, 'count' => 0 );
            $this -> tag_path_count[ 'ITEM/BASIC/LANG'              ] = array( 'name' => 'LANG'             , 'required' => true , 'multiple' => false, 'count' => 0 );
            $this -> tag_path_count[ 'ITEM/BASIC/URL'               ] = array( 'name' => 'URL'              , 'required' => true , 'multiple' => false, 'count' => 0 );
            $this -> tag_path_count[ 'ITEM/BASIC/CHANGELOGS'        ] = array( 'name' => 'CHANGELOGS'       , 'required' => false, 'multiple' => false, 'count' => 0 );
            break;
        case "ITEM/BASIC/TITLES":
            $this -> tag_path_count[ 'ITEM/BASIC/TITLES/TITLE'      ] = array( 'name' => 'TITLE'            , 'required' => true , 'multiple' => true , 'count' => 0 );
            break;
        case "ITEM/BASIC/CONTRIBUTOR":
//            if( is_null( $this -> importer_uid ) ) die( "invalid importer_uid is given(${importer_uid})" . $this -> getErrorAt( __LINE__, __FILE__, __FUNCTION__ ) );
//            $this -> setVar( 'uid', $this -> importer_uid );

            if( !is_null( $this -> importer_uid ) ){
                $this -> setVar( 'uid', $this -> importer_uid );
            }else{
                if( !isset( $attribs['UNAME'] ) ) $this -> setErrors( E_XOONIPS_ATTR_NOT_FOUND, "UNAME is not declared". $this -> getParserErrorAt( $parser ) );
                else if( empty( $attribs['UNAME'] ) ) $this -> setErrors( E_XOONIPS_ATTR_INVALID_VALUE, "UNAME is empty". $this -> getParserErrorAt( $parser ) );
                else {
                    $result = $xoopsDB -> query( "select xu.uid"
                                                 ." from ".$xoopsDB->prefix("users")." as xu, ".$xoopsDB->prefix("xoonips_users")." as xnpu"
                                                 ." where xu.uid = xnpu.uid and uname=".$xoopsDB -> quoteString( $attribs['UNAME'] ) );
                    if( !$result ){
                        $this -> setErrors( E_XOONIPS_DB_QUERY, $xoopsDB -> error() . $this -> getParserErrorAt( $parser ) );
                    }else if( $xoopsDB -> getRowsNum( $result ) > 0 ){
                        list( $uid ) = $xoopsDB->fetchRow( $result );
                        $this -> setVar( 'uid', $uid );
                    }else{
                        $this -> setErrors( E_XOONIPS_USER_NOT_FOUND, "user is not found(".$attribs['UNAME'].")"
                                            . $this -> getParserErrorAt( $parser ) );
                    }
                }
            }
            break;
        case "ITEM/BASIC/RELATED_TO": 
            if( empty( $attribs['ITEM_ID'] ) )
                $this -> setErrors( E_XOONIPS_ATTR_NOT_FOUND, "ITEM_ID is not declared". $this -> getParserErrorAt( $parser ) );
            else if( !ctype_digit( $attribs['ITEM_ID'] ) )
                $this -> setErrors( E_ATTR_INVALITEM_ID_VALUE, "ITEM_ID is not integer(${attribs['ITEM_ID']})"
                                    . $this -> getParserErrorAt( $parser ) );
            else {
                $related_to = $this -> getVar( 'related_to' );
                $related_to[] = intval( $attribs['ITEM_ID'] );
                $this -> setVar( 'related_to', serialize( $related_to ) );
            }
            break;
        case "ITEM/BASIC/INDEX": 
            if( !isset( $attribs['OPEN_LEVEL'] ) ){
                $this -> open_level = null;
            }else{
                switch( $attribs['OPEN_LEVEL'] ){
                case 'private':
                case 'group':
                case 'public':
                    $this -> open_level = $attribs['OPEN_LEVEL'];
                    break;
                default:
                    $this -> open_level = null;
                    $this -> setErrors( E_XOONIPS_ATTR_INVALID_VALUE, "illegal open_level(". $attribs['OPEN_LEVEL'].")" . $this -> getParserErrorAt( $parser ) );
                    break;
                }
            }
        }
    }
    
    function endElement($parser, $name) { 
        global $xoopsDB, $xoopsUser;
        
        $path = implode( '/', $this -> tagstack );
//        echo "\nE ".implode( '/', $this -> tagstack );
        switch( $path ){
        case "ITEM/BASIC": 
            // 
            // itemmust be registered into one more more private indexes.
            // error if this item is not registered any private indexes.
            // 
            $indexobj = new XooNIpsIndex();
            foreach( $this -> getVar( 'indexes' ) as $index_id ){
                if( !$indexobj -> get( $index_id ) ){
                    die( '$indexobj->get return false' );
                }
                if( $indexobj -> getVar( 'open_level' ) == 3 ) break;
            }
            if( $indexobj -> getVar( 'open_level' ) != 3 ){
                $this -> setErrors( E_XOONIPS_NO_PRIVATE_INDEX, 'item is not registered any private indexes' . $this -> getErrorAt( __LINE__, __FILE__, __FUNCTION__ ) );
            }
            break;
        case 'ITEM/BASIC/TITLES/TITLE':
            //
            // convert CDATA to internal character encoing
            //
            $title = encodeMeta2Server( end( $this -> cdata ) );
            
            //
            // strip title string and set error if too long
            //
            $ret = $this -> stripSurplusString( $this -> getColumnMaxLen( 'xoonips_item_title' ), array( 'title' => $title ) );
            // 
            // 
            $titles = $this -> getVar( 'titles' );
            if( $ret && isset( $ret['title'] ) ){
                $titles[] = $ret['title'];
                $this -> setErrors( E_XOONIPS_DATA_TOO_LONG, 'title is too long :' . $title . $this -> getParserErrorAt( $parser ) );
            }else{
                $titles[] = $title;
            }
            $this -> setVar( 'titles', serialize( $titles ) );
            break;
        case "ITEM/BASIC/KEYWORDS/KEYWORD": 
            //
            // convert CDATA to internal character encoing
            //
            $keyword = encodeMeta2Server( end( $this -> cdata ) );

            //
            // strip string and set error if too long
            //
            $keywords = $this -> getVar( 'keywords' );
            $keywords[] = $keyword;
            $this -> setVar( 'keywords', serialize( $keywords ) );
            break;
        case "ITEM/BASIC/ITEMTYPE":
            $result = $xoopsDB -> query( "select item_type_id from ".$xoopsDB->prefix("xoonips_item_type")
                                         ." where name=".$xoopsDB -> quoteString( end( $this -> cdata ) ) );
            if( !$result ){
                $this -> setErrors( E_XOONIPS_DB_QUERY, $xoopsDB -> error() . $this -> getParserErrorAt( $parser ) );
            }else if( $xoopsDB -> getRowsNum( $result ) > 0 ){
                list( $item_type_id ) = $xoopsDB->fetchRow( $result );
                $this -> setVar( 'item_type_id', $item_type_id );
            }else{
                $this -> setErrors( E_XOONIPS_INVALID_VALUE, "unknown itemtype(".end( $this -> cdata ).")"
                                    . $this -> getParserErrorAt( $parser ) );
            }
            break;
        case "ITEM/BASIC/CONTRIBUTOR": 
            break;
        case "ITEM/BASIC/DESCRIPTION": 
            //
            // convert CDATA to internal character encoing
            //
            $description = encodeMeta2Server( end( $this -> cdata ) );

            //
            // strip string and set error if too long
            //
            $this -> setVar( 'description', $description );
            break;
        case "ITEM/BASIC/DOI": 
            $this -> setVar( 'doi', end( $this -> cdata ) );
            break;
        case "ITEM/BASIC/LAST_UPDATE_DATE": 
            $last_update_date = $this -> ISO8601toUTC( end( $this -> cdata ) );
            if( FALSE === $last_update_date ){
                $this -> setErrors( E_XOONIPS_INVALID_VALUE, "illegal date format(" . end( $this -> cdata ) . ")"
                                    . $this -> getParserErrorAt( $parser ) );
            }else{
                $this -> setVar( 'last_update_date', $last_update_date );
            }
            break;
        case "ITEM/BASIC/CREATION_DATE": 
            $creation_date = $this -> ISO8601toUTC( end( $this -> cdata ) );
            if( FALSE === $creation_date ){
                $this -> setErrors( E_XOONIPS_INVALID_VALUE, "illegal date format(" . end( $this -> cdata ) . ")"
                                    . $this -> getParserErrorAt( $parser ) );
            }else{
                $this -> setVar( 'creation_date', $creation_date );
            }
            break;
        case "ITEM/BASIC/PUBLICATION_YEAR": 
            $this -> setVar( 'publication_year', end( $this -> cdata ) );
            break;
        case "ITEM/BASIC/PUBLICATION_MONTH": 
            $this -> setVar( 'publication_month', end( $this -> cdata ) );
            break;
        case "ITEM/BASIC/PUBLICATION_MDAY": 
            $this -> setVar( 'publication_mday', end( $this -> cdata ) );
            break;
        case "ITEM/BASIC/URL": 
            $this -> setVar( 'url', end( $this -> cdata ) );
            break;
        case "ITEM/BASIC/LANG": 
            $this -> setVar( 'lang', end( $this -> cdata ) );
            break;
        case "ITEM/BASIC/RELATED_TO": 
            break;
        case "ITEM/BASIC/INDEX": 
            // 
            // error if unescaped yen is found
            // 
            $str = end( $this -> cdata );
            if( extension_loaded( 'mbstring' ) ){
                $regexp = array( ".*[^\\\\]\\\\[^\\/\\\\].*", ".*[^\\\\]\\\\$", "^\\\\[^\\/\\\\].*"  );
                if( mb_ereg_match( $regexp[0], $str )
                    || mb_ereg_match( $regexp[1], $str )
                    || mb_ereg_match( $regexp[2], $str ) ){
                    $this -> setErrors( E_XOONIPS_INVALID_VALUE, "invalid value in index. illegal use of '\\'($str)"
                                        . $this -> getErrorAt( __LINE__, __FILE__, __FUNCTION__ ) );
                    break;
                }
            }else{
                $regexp = array( "/[^\\\\]\\\\[^\\/\\\\]/", "/.*[^\\\\]\\\\$/", "/^\\\\[^\\/\\\\]/" );
                if( preg_match( $regexp[0], $str ) 
                    || preg_match( $regexp[1], $str ) 
                    || preg_match( $regexp[2], $str ) ){
                    $this -> setErrors( E_XOONIPS_INVALID_VALUE, "invalid value in index. illegal use of '\\'($str)"
                                        . $this -> getErrorAt( __LINE__, __FILE__, __FUNCTION__ ) );
                    break;
                }
            }
            // 
            // get index ID for each given base_index_id if CDATA of <index> is relative path
            // 
            if( strncmp( end( $this -> cdata ) , '/', 1 ) != 0 ){//true if relative path
                foreach( $this -> base_index_id as $index_id ){
                    // 
                    // retrieve open_level of index specified by given $base_index_id
                    // 
                    $base_index = new XooNIpsIndex( );
                    if( $base_index -> get( $index_id ) ){
                        switch( $base_index -> getVar('open_level') ){
                        case 1://public
                            $open_level = 'public';
                            break;
                        case 2://group
                            $open_level = 'group';
                            break;
                        case 3://private
                        default:
                            $open_level = 'private';
                            break;
                        }
                        // 
                        // get index ID that item is imported to
                        // 
                        $index_str = $this -> index_id2index_str( $index_id );
                        $id = $this -> index_str2index_id( end( $this -> cdata ),
                                                           extension_loaded( 'mbstring' ) ? mb_convert_encoding( $index_str, "UTF-8" ) : $index_str,
                                                           $open_level );
                        if( FALSE === $id ){
                            $this -> setErrors( E_XOONIPS_INDEX_NOT_FOUND, 'index is not found(' 
                                                . $this -> index_id2index_str( $index_id ) . '/'
                                                . end( $this -> cdata ) . ')'
                                                . $this -> getErrorAt( __LINE__, __FILE__, __FUNCTION__ ) );
                            continue; // try to next base index
                        }

                        // 
                        // does user(importer_uid) have access to index($id) ?
                        // 
                        $do_import = false;// true if item can be imported.
                        $indexobj = new XooNIpsIndex();
                        if( !$indexobj -> get( $id ) )
                            continue; // try to next base index
                        
                        $do_import = $indexobj -> getPerm( $this -> getVar( 'uid' ) );// true if item can be imported.
                        if( !$do_import ){
                            $this -> setErrors( E_XOONIPS_NOT_PERMITTED_ACCESS, 'not permitted access to index(' 
                                                . $this -> index_id2index_str( $index_id ) . "/"
                                                . end( $this -> cdata )
                                                . ') by user(uid='
                                                . $this -> getVar( 'uid' )
                                                . ')'
                                                . $this -> getErrorAt( __LINE__, __FILE__, __FUNCTION__ ) );
                        }
                            
                        // 
                        // import this item to index.
                        // 
                        if( $do_import ){
                            $indexes = $this -> getVar( 'indexes' );
                            $indexes[] = $id;
                            $this -> setVar( 'indexes', serialize( $indexes ) );
                        }
                    }
                }
            }else{
                // absolute index path
                $id = $this -> index_str2index_id( end( $this -> cdata ), "", $this -> open_level );
                if( $id !== false ){
                    // 
                    // does user(importer_uid) have access to index($id) ?
                    // 
                    $do_import = false;// true if item can be imported.
                    $indexobj = new XooNIpsIndex();
                    if( $indexobj -> get( $id ) ){
                        $do_import = $indexobj -> getPerm( $this -> getVar( 'uid' ) );// true if item can be imported.
                        if( !$do_import ){
                            $this -> setErrors( E_XOONIPS_NOT_PERMITTED_ACCESS, 'not permitted access to index(' 
                                                . end( $this -> cdata )
                                                . ') by user(uid='
                                                . $this -> getVar( 'uid' )
                                                . ')'
                                                . $this -> getErrorAt( __LINE__, __FILE__, __FUNCTION__ ) );
                        }
                        else {
                            $indexes = $this -> getVar( 'indexes' );
                            $indexes[] = $id;
                            $this -> setVar( 'indexes', serialize( $indexes ) );
                        }
                    }
                }
            }
            break;
        case "ITEM":
            // check if required tag exists
            foreach ( $this -> tag_path_count as $path => $info ){
                if ( $info['required'] && $info['count'] == 0 ){
                    $this -> setErrors( E_XOONIPS_VALUE_IS_REQUIRED, "tag(" . $info['name'] . ") is required " . $this -> getParserErrorAt( $parser ) );
                }
            }
            break;
        }
        
        parent::endElement($parser, $name);
    }
    
    function characterData($parser, $data) {
        parent::characterData($parser, $data);
//        echo "\nC ".implode( '/', $this -> tagstack )."\t${data}";
    }
    
    /**
     * 
     * change expression of ISO8601 to UTC. return false when we can't change.
     * Usage: ISO8601toUTC( "2005-08-01T12:00:00Z" );
     * Usage: ISO8601toUTC( "2005-08-01" );
     * 
     */
    function ISO8601toUTC( $str )
    {
        // $match[?]
        // $match[0]  : ($str)
        // $match[1]  : 
        // $match[2]  : 
        // $match[3]  : 
        // $match[4]  : 
        // $match[5]  : 
        // $match[6]  : 
        // $match[7]  : 
        // $match[8]  : ʬ
        // $match[9]  : 
        // $match[10] : 
        // $match[11] : 'Z'ޤ (Rexp Z|[-+][0-9]{2}:[0-9]{2})
        // $match[12] : (+|-)
        // $match[13] : ()
        // $match[14] : (ʬ)
        
        if( preg_match( '/^([0-9]{4})(-([0-9]{2})(-([0-9]{2})(T([0-9]{2}):([0-9]{2})(:([0-9]{2}))?(Z|([-+])([0-9]{2}):([0-9]{2}))?)?)?)?$/', $str, $match ) == 0 )
//        if( preg_match( '/([0-9]{4})(-([0-9]{2})(-([0-9]{2})(T([0-9]{2}):([0-9]{2})(:([0-9]{2}))?(Z|([-+])([0-9]{2}):([0-9]{2}))?)?)?)?/', $str, $match ) == 0 )
            return false;
//        print_r($match);
        
        if( !isset( $match[3] ) )  $match[3] = '01';
        if( !isset( $match[5] ) )  $match[5] = '01';
        if( !isset( $match[7] ) )  $match[7] = '00';
        if( !isset( $match[8] ) )  $match[8] = '00';
        if( !isset( $match[10] ) || $match[10] == "" ) $match[10] = '00';
        $tm = gmmktime( $match[7], $match[8], $match[10], $match[3], $match[5], $match[1] );
        if ( false === $tm || -1 == $tm && version_compare(phpversion(), "5.1.0", "<") )
            return false; // gmmktime failed.
        
        // hh:mm:ss must be in 00:00:00 - 24:00:00
        if ( $match[10] >= 60 ) return false;
        if ( $match[8] >= 60 ) return false;
        if ( $match[7] > 24 || $match[7] == 24 && ( $match[8] != 0 || $match[10] != 0 ) ) return false;
        
        // mm and dd must not overflow
        if ( gmdate( 'Ymd', gmmktime( 0, 0, 0, $match[3], $match[5], $match[1] ) ) != $match[1].$match[3].$match[5] )
            return false;
        
        //correct a time difference to GMT
        if( isset( $match[11] ) && isset( $match[12] ) && isset( $match[13] ) && isset( $match[14] ) ){
            if( $match[11] != 'Z' && $match[12] == '-' ){
                $tm = $tm + ( $match[ 13 ] * 3600 + $match[ 14 ] * 60 );
            }else if( isset( $match[12] ) && $match[11] != 'Z' && $match[12] == '+' ){
                $tm = $tm - ( $match[ 13 ] * 3600 + $match[ 14 ] * 60 );
            }
        }
        return $tm;
    }

    /**
     * 
     * get an id of index specified by given arguments.
     * if $str is relative path, $base_index_path must be given.
     * regards as private if $open_level is null
     * 
     * 
     * @param $str index path like '/Public/Parent/Child'(absolute) or 'Parent/Child'(relative) (must be UTF-8)
     * @param $uid uid of contributor(if null, use executor's uid)
     * @param $base_index_path base index path like '/Public/XXX/YYY'(absolute only). default is '/'.
     * @param $open_level open_level of index of base_index_path('private'|'group'|'public'|null). null if not specified.
     * @return index id or false if failed.
     */
    function index_str2index_id( $str, $base_index_path = "/", $open_level = 'private' ){
        global $xoopsDB, $xoopsUser;
        
        if( !is_object( $xoopsUser ) ){
            $this -> setErrors( E_XOONIPS_ERROR, 'xoopsUser is not set' . $this -> getErrorAt( __LINE__, __FILE__, __FUNCTION__ ) );
            return false;
        }
        
        if( is_null( $this -> importer_uid ) ){
            $user = $xoopsUser;
        }else{
            $user_handler =& xoops_gethandler('user');
            $user =& $user_handler->get($this -> importer_uid);
            if( $user === FALSE ){
                $this -> setErrors( E_XOONIPS_ERROR, 'User is not found(' . $this -> importer_uid . ')' . $this -> getErrorAt( __LINE__, __FILE__, __FUNCTION__ ) );
                return false;
            }
        }
        
        if( !empty( $str ) && $str[ 0 ] == '/' ){
            $index_path = $str;
        }else{
            // 
            // don't append any string if $str is empty(''). $index_path equals to $base_index_path
            // to specify index that items is imported to by relatively.
            // 
            $index_path = $base_index_path . ( empty( $str ) ? '' : '/' . $str );
        }
        
        // 
        // convert given open_level to int value
        // 
        if( is_null( $open_level ) )  $open_level = 'private';// regards as private if $open_level is null
        if( $open_level == 'public' ){
            $open_level_val = 1;
        }else if( $open_level == 'group' ){
            $open_level_val = 2;
        }else if( $open_level == 'private' ){
            $open_level_val = 3;
        }else{
            $this -> setErrors( E_XOONIPS_INVALID_VALUE, "unknown open_level '$open_level'" . $this -> getErrorAt( __LINE__, __FILE__, __FUNCTION__ ));
            return false;
        }
        
        $indexes = $this -> decomposite_index_path( $index_path );
        
        // 
        // - find last(depth is max) indexes
        // - for each indeses, we traverse ancestor index to root index
        // - returns index path that reach to root index first.
        // 
        
        // replace 'Private' to own uname
        // set 'private' to $open_level 
        // set 3 to to $open_level_val
        if( $indexes[ 0 ] == 'Private' ){
            $indexes[ 0 ] = $user -> getVar( 'uname' );
            $open_level = 'private';
            $open_level_val = 3;
        }
        
        // return index id
        return $this -> index_array2index_id( IID_ROOT, $indexes, $open_level );
    }
    
    /**
     * 
     * index path strnig to array of each indexes and strip escape char 'yen'
     * (ex:/ABC/XYZ/012 -> array( 'ABC', 'XYZ', '012' ) )
     * (ex:/ABC/XYZ/012/ -> array( 'ABC', 'XYZ', '012', '' ) )
     * 
     * @param $str index path like '/Public/Parent/Child' (must be UTF-8)
     * @return array of indexes;
     */
    function decomposite_index_path( $str ){
        $ret = array();
        
        $i = 1;
        $tok = '';
        $maxlen = extension_loaded("mbstring") ? mb_strlen( $str, 'UTF-8' ) : strlen( $str );
        while( $i < $maxlen ){
            $ch = extension_loaded("mbstring") ? mb_substr( $str, $i, 1, 'UTF-8' ) : $str[ $i ];
            if( $ch == "/" ){
                $ret[] = $tok;
                $tok = '';
            }else if( $ch == "\\" ){
                $i++;
                $ch = extension_loaded("mbstring") ? mb_substr( $str, $i, 1, 'UTF-8' ) : $str[ $i ];
                $tok .= $ch;
            }else{
                $tok .= $ch;
            }
            $i++;
        }
        $ret[] = $tok;
        return $ret;
    }

    /**
     * retrieve item's basic information.
     * 
     * @param id item ID(need to be more than 0)
     * @return true if succeed. false if failed.
     */
    function get($id){
        global $xoopsDB;
        if (intval($id) > 0) {
            $sql = 'SELECT * FROM '.$xoopsDB -> prefix('xoonips_item_basic').' WHERE item_id='.$id;
            if (!$result = $xoopsDB -> query($sql)) {
                $this -> setErrors( E_XOONIPS_DB_QUERY, $xoopsDB -> error() . $this -> getErrorAt( __LINE__, __FILE__, __FUNCTION__ ) );
                return false;
            }
            if ($xoopsDB -> getRowsNum($result) == 1) {
                $row = $xoopsDB -> fetchArray($result);
                foreach( array( 'item_id', 'uid', 'creation_date', 'description', 'last_update_date', 'publication_year', 'publication_month', 'publication_mday', 'item_type_id', 'lang', 'doi' ) as $k ){
                    $this->assignVar( $k, $row[ $k ] );
                }
            }
            
            $sql = 'SELECT * FROM '.$xoopsDB -> prefix('xoonips_item_title').' WHERE item_id=' . $id . " order by title_id";
            if (!$result = $xoopsDB -> query($sql)) {
                $this -> setErrors( E_XOONIPS_DB_QUERY, $xoopsDB -> error() . $this -> getErrorAt( __LINE__, __FILE__, __FUNCTION__ ) );
                return false;
            }
            if ($xoopsDB -> getRowsNum($result) > 0 ) {
                $titles = array();
                while( $row = $xoopsDB -> fetchArray($result) ){
                    $titles[] = $row[ 'title' ];
                }
                $this->assignVar( 'titles', serialize( $titles ) );
            }
            
            $sql = 'SELECT * FROM '.$xoopsDB -> prefix('xoonips_item_keyword').' WHERE item_id=' . $id . " order by keyword_id";
            if (!$result = $xoopsDB -> query($sql)) {
                $this -> setErrors( E_XOONIPS_DB_QUERY, $xoopsDB -> error() . $this -> getErrorAt( __LINE__, __FILE__, __FUNCTION__ ) );
                return false;
            }
            if ($xoopsDB -> getRowsNum($result) > 0 ) {
                $keywords = array();
                while( $row = $xoopsDB -> fetchArray($result) ){
                    $keywords[] = $row[ 'keyword' ];
                }
                $this->assignVar( 'keywords', serialize( $keywords ) );
            }
            
            $sql = 'SELECT * FROM '.$xoopsDB -> prefix('xoonips_related_to').' WHERE parent_id=' . $id . ' order by item_id';
            if (!$result = $xoopsDB -> query($sql)) {
                $this -> setErrors( E_XOONIPS_DB_QUERY, $xoopsDB -> error() . $this -> getErrorAt( __LINE__, __FILE__, __FUNCTION__ ) );
                return false;
            }
            if ($xoopsDB -> getRowsNum($result) > 0 ) {
                $rel = array();
                while( $row = $xoopsDB -> fetchArray($result) ){
                    $rel[] = $row[ 'item_id' ];
                }
                $this->assignVar( 'related_to', serialize( $rel ) );
            }
            
            $sql = 'SELECT * FROM '.$xoopsDB -> prefix('xoonips_index_item_link').' WHERE item_id=' . $id . ' order by index_item_link_id';
            if (!$result = $xoopsDB -> query($sql)) {
                $this -> setErrors( E_XOONIPS_DB_QUERY, $xoopsDB -> error() . $this -> getErrorAt( __LINE__, __FILE__, __FUNCTION__ ) );
                return false;
            }
            if ($xoopsDB -> getRowsNum($result) > 0 ) {
                $idx = array();
                while( $row = $xoopsDB -> fetchArray($result) ){
                    $idx[] = $row[ 'index_id' ];
                }
                $this->assignVar( 'indexes', serialize( $idx ) );
            }
        }
        return true;
    }
    
    /**
     * insert/update object
     *
     * - insert/update basic information 
     * - insert/update titles
     * - insert/update keywords
     * - insert/update related_to
     * - insert/update index_item_link(this functino don't certify these items)
     * 
     * @return false if failed, true if insert successfully or already exists and unchanged.
     *   
     */
    function insert()
    {
        global $xoopsDB;
        
        if (!$this->isDirty()) {
            return true;
        }
        
        if (!$this->cleanVars()) {
            // 
            // return false if error in cleanVars 
            // 
            return false;
        }
        
        $this->unsetDirty(); //to be clean
        // 
        // insert/update basic information
        // 
        if ($this->isNew()) {
            $sql = sprintf("INSERT INTO %s (item_type_id, uid, description, doi, last_update_date, creation_date, publication_year, publication_month, publication_mday, lang) VALUES (%u, %u, %s, %s, %u, %u, %u, %u, %u, %s)",
                           $xoopsDB -> prefix( "xoonips_item_basic" ),
                           $this -> getVar( 'item_type_id' ),
                           $this -> getVar( 'uid' ),
                           $xoopsDB -> quoteString( $this -> getVar( 'description', 'none' ) ),
                           $xoopsDB -> quoteString( $this -> getVar( 'doi' ) ),
                           $this -> getVar( 'last_update_date' ),
                           $this -> getVar( 'creation_date' ),
                           $this -> getVar( 'publication_year' ),
                           $this -> getVar( 'publication_month' ),
                           $this -> getVar( 'publication_mday' ),
                           $xoopsDB -> quoteString( $this -> getVar( 'lang' ) ) );
        } else {
            $sql = sprintf("UPDATE %s SET item_type_id = %u, uid = %u, description = %s, doi = %s, last_update_date = %u, creation_date = %u, publication_year = %u, publication_month = %u, publication_mday = %u, lang = %s WHERE item_id = %u",
                           $xoopsDB -> prefix( "xoonips_item_basic" ),
                           $this -> getVar( 'item_type_id' ),
                           $this -> getVar( 'uid' ),
                           $xoopsDB -> quoteString( $this -> getVar( 'description', 'none' ) ),
                           $xoopsDB -> quoteString( $this -> getVar( 'doi' ) ),
                           $this -> getVar( 'last_update_date' ),
                           $this -> getVar( 'creation_date' ),
                           $this -> getVar( 'publication_year' ),
                           $this -> getVar( 'publication_month' ),
                           $this -> getVar( 'publication_mday' ),
                           $xoopsDB -> quoteString( $this -> getVar( 'lang' ) ),
                           $this -> getVar( 'item_id' ) );
        }
        $result = $xoopsDB->query($sql);
        if (!$result) {
            $this -> setErrors( E_XOONIPS_DB_QUERY, $xoopsDB -> error() . $this -> getErrorAt( __LINE__, __FILE__, __FUNCTION__ ) );
            return false;
        }
        
        // 
        // set this item id
        // 
        if ($this->isNew()) {
            $this_item_id = $xoopsDB->getInsertId();
        }else{
            $this_item_id = $this -> getVar( 'item_id' );
        }

        // 
        // insert/update titles
        // 
        if( count( $this -> getVar( 'titles' ) ) > 0 ){
            $title_id = 0;
            $values = array( );//array of string "( <item_id>, <title_id>, <quoted title string> )"
            foreach( $this -> getVar( 'titles' ) as $title ){
                $values[] = sprintf( "( %u, %u, %s )", $this_item_id, $title_id, $xoopsDB -> quoteString( $title ) );
                $title_id++;
            }
            if ($this->isNew()) {
                $sql = sprintf( "INSERT INTO %s (item_id, title_id, title) VALUES " . implode( ", ", $values ),
                                $xoopsDB -> prefix( "xoonips_item_title" ) );
            } else {
                $sql = sprintf( "REPLACE %s (item_id, title_id, title) VALUES " . implode( ", ", $values ),
                                $xoopsDB -> prefix( "xoonips_item_title" ) );
            }
            $result = $xoopsDB->query($sql);
            if (!$result) {
                $this -> setErrors( E_XOONIPS_DB_QUERY, $xoopsDB -> error() . $this -> getErrorAt( __LINE__, __FILE__, __FUNCTION__ ) );
                return false;
            }
        }
        // 
        // insert/update keywords
        // 
        if (!$this->isNew()) {
            $sql = sprintf( "DELETE FROM %s WHERE item_id=$this_item_id", $xoopsDB -> prefix( "xoonips_item_keyword" ) );
            $result = $xoopsDB->query($sql);
            if (!$result) {
                $this -> setErrors( E_XOONIPS_DB_QUERY, $xoopsDB -> error() . $this -> getErrorAt( __LINE__, __FILE__, __FUNCTION__ ) );
                return false;
            }
        }
        if( count( $this -> getVar( 'keywords' ) ) > 0 ){
            $keyword_id = 0;
            $values = array( );//array of string "( <item_id>, <keyword_id>, <quoted keyword string> )"
            foreach( $this -> getVar( 'keywords' ) as $keyword ){
                $values[] = sprintf( "( %u, %u, %s )", $this_item_id, $keyword_id, $xoopsDB -> quoteString( $keyword ) );
                $keyword_id++;
            }
            $sql = sprintf( "INSERT INTO %s (item_id, keyword_id, keyword) VALUES " . implode( ", ", $values ),
                            $xoopsDB -> prefix( "xoonips_item_keyword" ) );
            $result = $xoopsDB->query($sql);
            if (!$result) {
                $this -> setErrors( E_XOONIPS_DB_QUERY, $xoopsDB -> error() . $this -> getErrorAt( __LINE__, __FILE__, __FUNCTION__ ) );
                return false;
            }
        }
        
        // 
        // insert/update related_to
        // 
        if (!$this->isNew()) {
            $sql = sprintf( "DELETE FROM %s WHERE parent_id=%u", $xoopsDB -> prefix( "xoonips_related_to" ), $this_item_id );
            $result = $xoopsDB->query($sql);
            if (!$result) {
                $this -> setErrors( E_XOONIPS_DB_QUERY, $xoopsDB -> error() . $this -> getErrorAt( __LINE__, __FILE__, __FUNCTION__ ) );
                return false;
            }
        }
        if( count( $this -> getVar( 'related_to' ) ) > 0 ){
            $values = array( );//array of string "( <parent_id(this item)>, <related item_id> )"
            foreach( array_unique( $this -> getVar( 'related_to' ) ) as $related_to ){
                $values[] = sprintf( "( %u, %u )", $this_item_id, $related_to );
            }
            $sql = sprintf( "INSERT IGNORE INTO %s (parent_id, item_id) VALUES " . implode( ", ", $values ),
                            $xoopsDB -> prefix( "xoonips_related_to" ) );
            $result = $xoopsDB->query($sql);
            if (!$result) {
                $this -> setErrors( E_XOONIPS_DB_QUERY, $xoopsDB -> error() . $this -> getErrorAt( __LINE__, __FILE__, __FUNCTION__ ) );
                return false;
            }
        }
        
        // 
        // private_index flag 
        // true if item is registered to private index(es)
        // 
        $private_index_flag = false;

        // 
        // insert/update item_index_link( item is not certified in this process )
        // 
        if (!$this->isNew()) {
            $sql = sprintf( "DELETE FROM %s WHERE item_id=%u", $xoopsDB -> prefix( "xoonips_index_item_link" ), $this_item_id );
            $result = $xoopsDB->query($sql);
            if (!$result) {
                $this -> setErrors( E_XOONIPS_DB_QUERY, $xoopsDB -> error() . $this -> getErrorAt( __LINE__, __FILE__, __FUNCTION__ ) );
                return false;
            }
        }
        if( count( $this -> getVar( 'indexes' ) ) > 0 ){
            $values = array( );//array of string "( <index_id>, <item_id> )"
            foreach( array_unique( $this -> getVar( 'indexes' ) ) as $index_id ){
                $values[] = sprintf( "( %u, %u )", $index_id, $this_item_id );
            }
            $sql = sprintf( "INSERT IGNORE INTO %s (index_id, item_id) VALUES " . implode( ", ", $values ),
                            $xoopsDB -> prefix( "xoonips_index_item_link" ) );
            $result = $xoopsDB->query($sql);
            if (!$result) {
                $this -> setErrors( E_XOONIPS_DB_QUERY, $xoopsDB -> error() . $this -> getErrorAt( __LINE__, __FILE__, __FUNCTION__ ) );
                return false;
            }
        }
        
        // 
        // update variables(item_id, url) with item ID of the inserted record.
        // 
        if ($this->isNew()) {
            //use assinVar instead of setVar so that isDirty flag holds false
            $this -> assignVar('item_id', $this_item_id);
            $this -> assignVar('url', XOOPS_URL . "/modules/xoonips/detail.php?item_id=" . $this -> getVar( 'item_id' ) );
            $this -> unsetNew();
        }
        return true;
    }
    
    /**
     * delete an item from the database
     * 
     * @return false if failed, true if delete successfully
     *   
     */
    function delete(){
        global $xoopsDB;
        
        $sql = sprintf( "DELETE FROM %s WHERE item_id = %u", $xoopsDB -> prefix( "xoonips_item_basic" ), $this -> getVar( 'item_id' ) );
        $result = $xoopsDB->query( $sql );
        if (!$result) {
            $this -> setErrors( E_XOONIPS_DB_QUERY, $xoopsDB -> error() . $this -> getErrorAt( __LINE__, __FILE__, __FUNCTION__ ) );
            return false;
        }
        return true;
    }


    /**
     * - insert item to database
     * - register item to indexes
     * @param certify_auto true if item imported to /Public index must be certified automatically.(default=false)
     * @return boolean true if succeed. false if failed in importing(see getErrors).
     */
    function import( $certify_auto = false ){
        global $xoopsDB, $xoopsUser;
        $xnpsid = $_SESSION['XNPSID'];
        
        $is_new = $this->isNew(); //hold result of isNew()
        $is_public_item = false;// true if this item is registered to public index
        $old_certify_state = false;// true if this item is certified in one or more public indexes before importing.
        
        if( !$is_new ){// set $old_certify_state
            $old_item = new XooNIpsItem();
            $old_item -> get( $this -> getVar( 'item_id' ) );
            foreach( $old_item -> getVar( 'indexes' ) as $index_id ){
                $state = null;
                if( xnp_get_certify_state( $xnpsid, $index_id, $this -> getVar( 'item_id' ), $state ) == RES_OK ){
                    if( $state == CERTIFIED ){
                        $old_certify_state = true;
                        break;
                    }
                }
            }
        }
        
        if( $this -> insert( ) ){
            $item_id = $this -> getVar( 'item_id' );
            
            //
            // register this to indexes
            // (here, only update certify_state if it's needed. because item has been registered to index in insert(). )
            //
            if( count( $this -> getVar( 'indexes' ) ) > 0 ){ 
                $values = array();
                $indexobj = new XooNIpsIndex();
                foreach( $this -> getVar( 'indexes' ) as $index_id ){
                    $indexobj -> get( $index_id );
                    
                    if( !$is_public_item && $indexobj -> getVar( 'open_level' ) == OL_PUBLIC ) $is_public_item = true;
                    
                    $certify_state = NOT_CERTIFIED; //default( not certified )
                    $index_gid = $indexobj -> getVar( 'gid' );
                    if( $xoopsUser -> isAdmin() || isset( $_SESSION[ 'xoonips_old_uid' ] ) ){//for administrator or su
                        // 
                        // certify item if xoopsUser is administrator
                        // 
                        if( $indexobj -> getVar( 'open_level' ) == OL_PUBLIC )
                            $certify_state = ( $certify_auto ? CERTIFIED : CERTIFY_REQUIRED );
                        else if( $indexobj -> getVar( 'open_level' ) == OL_GROUP_ONLY )
                            $certify_state = CERTIFY_REQUIRED;
                        else if( $indexobj -> getVar( 'open_level' ) == OL_PRIVATE )
                            $certify_state = NOT_CERTIFIED;
                        else{
                            $this -> setErrors( E_XOONIPS_INVALID_VALUE, "illegal open_level '" . $indexobj -> getVar( 'open_level' ) . "'" . $this -> getErrorAt( __LINE__, __FILE__, __FUNCTION__ ));
                            return false;
                        }
                    }else if( !empty( $index_gid )
                              && xnp_is_group_admin( $xnpsid, $index_gid, $xoopsUser -> getVar( 'uid' ) ) ){ // for group admin
                        // 
                        // certify item if xoopsUser is group administrator
                        // 
                        if( $indexobj -> getVar( 'open_level' ) == OL_PUBLIC )
                            $certify_state = CERTIFY_REQUIRED;
                        else if( $indexobj -> getVar( 'open_level' ) == OL_GROUP_ONLY )
                            $certify_state = CERTIFY_REQUIRED;
                        else if( $indexobj -> getVar( 'open_level' ) == OL_PRIVATE )
                            $certify_state = NOT_CERTIFIED;
                        else{
                            $this -> setErrors( E_XOONIPS_INVALID_VALUE, "illegal open_level '" . $indexobj -> getVar( 'open_level' ) . "'" . $this -> getErrorAt( __LINE__, __FILE__, __FUNCTION__ ));
                            return false;
                        }
                    }else{
                        // 
                        // request certify to administrator or group administrator
                        // 
                        if( $indexobj -> getVar( 'open_level' ) == OL_PUBLIC )
                            $certify_state = CERTIFY_REQUIRED;
                        else if( $indexobj -> getVar( 'open_level' ) == OL_GROUP_ONLY )
                            $certify_state = CERTIFY_REQUIRED;
                        else if( $indexobj -> getVar( 'open_level' ) == OL_PRIVATE )
                            $certify_state = NOT_CERTIFIED;
                        else{
                            $this -> setErrors( E_XOONIPS_INVALID_VALUE, "illegal open_level '" . $indexobj -> getVar( 'open_level' ) . "'" . $this -> getErrorAt( __LINE__, __FILE__, __FUNCTION__ ) );
                            return false;
                        }
                    }
                    // 
                    // update this link only that open_level is public 
                    // 
                    $sql = 'UPDATE ' . $xoopsDB -> prefix( 'xoonips_index_item_link' )
                        . ' SET certify_state = ' . $certify_state
                        . ' WHERE index_id=' . $index_id
                        . ' AND item_id=' . $item_id;
                    $result = $xoopsDB -> query( $sql );
                    if (!$result) {
                        $this -> setErrors( E_XOONIPS_DB_QUERY, $xoopsDB -> error() . $this -> getErrorAt( __LINE__, __FILE__, __FUNCTION__ ) );
                        $this -> delete( );
                        return false;
                    }
                }
            }
            
            // updating xoonips_item_status to show public items to OAI-PMH harvester if following items
            //  - new item in public index and certified automatically
            //  - updated item in public index and certified automatically
            //  - updated item that has been certified before importing
            if( $is_new && $is_public_item && $certify_auto
                || !$is_new && $is_public_item && $certify_auto
                || !$is_new && $is_public_item && $old_certify_state == CERTIFIED ){
                if( insertMetadataEventAuto( $this -> getVar( 'item_id' ), $is_new ) != RES_OK ){
                    if( $certify_auto ){
                        $this -> setErrors( E_XOONIPS_ERROR, "Item is not shown to OAI-PMH harvester(because of failure in updating item status)"
                                            . $this -> getErrorAt( __LINE__, __FILE__, __FUNCTION__ ) );
                    }else{
                        $this -> setErrors( E_XOONIPS_ERROR, "Item is not hidden to OAI-PMH harvester(because of failure in updating item status)"
                                            . $this -> getErrorAt( __LINE__, __FILE__, __FUNCTION__ ) );
                    }
                    return false;
                }
            }
            return true;
        }else{
            return false;
        }
    }

    /**
     * 
     * return string of item
     * @param htmlspecialchars escape html special chars if true.
     * @return string repeat of 'basic.<fieldname> <value>\n' 
     * 
     */
    function toString($htmlspecialchars=false){
        $text = parent::toString($htmlspecialchars);
        $item_id = $this -> getVar( 'item_id' );
        if( !empty( $item_id ) ){
            $text .= "basic.id " . $this -> getVar( 'item_id' ) . "\n";//	ƥID	/item/basic/@id
        }
        foreach( $this -> getVar( 'titles' ) as $title ){
            $text .= "basic.title " . ( $htmlspecialchars ? htmlspecialchars( $title ) : $title ) . "\n"; //	ȥ	/item/basic/titles/title
        }

        $text .= "basic.contributor " . $this -> getVar( 'uid' ) //	ԤUID	/item/basic/contributor
            . "\nbasic.itemtype " . $this -> getVar( 'item_type_id' ); //	ƥॿID	/item/basic/itemtype

        foreach( $this -> getVar( 'keywords' ) as $keyword ){
            $text .= "\nbasic.keyword " . ( $htmlspecialchars ? htmlspecialchars( $keyword ) : $keyword ); //	ȥ	/item/basic/keywords/keyword
        }
        
        //		/item/basic/description
        $comment = '';
        if( extension_loaded( 'mbstring' ) ){
            $comment .= mb_ereg_replace( '\n', '\n', mb_ereg_replace( '\\\\', '\\\\', $this -> getVar( 'description', 'none' ) ) );
        }else{
            $comment .= str_replace( array( "\\", "\n" ), array( "\\\\", "\\n" ), $this -> getVar( 'description', 'none' ) );
        }
        $text .= "\nbasic.description " . ( $htmlspecialchars ? htmlspecialchars( $comment ) : $comment );
        
        $text .= "\nbasic.doi " . $this -> getVar( 'doi' ) //	ID	/item/basic/doi
            . "\nbasic.last_update_date " . $this -> getVar( 'last_update_date' ) //	ǽɽ	/item/basic/last_update_date
            . "\nbasic.creation_date " . $this -> getVar( 'creation_date' ) //	ɽ	/item/basic/creation_date
            . "\nbasic.publication_year " . $this -> getVar( 'publication_year' ) //	ȯǯ	/item/basic/publication_year
            . "\nbasic.publication_month " . $this -> getVar( 'publication_month' ) //	ȯԷ	/item/basic/publication_month
            . "\nbasic.publication_mday " . $this -> getVar( 'publication_mday' ) //	ȯ	/item/basic/publication_mday
            . "\nbasic.lang " . $this -> getVar( 'lang' ); //		/item/basic/lang

        $url = $this -> getVar( 'item_id' );
        if( !empty( $url ) ){
            $text .= "\nbasic.url " . $this -> getVar( 'url' ); //	URL of /item/basic/url 
        }
        
        foreach( $this -> getVar( 'related_to' ) as $related_to ){
            $text .= "\nbasic.related_to " . $related_to; //	ϢƥID()	/item/basic/related_to/@item_id
        }
        foreach( $this -> getVar( 'indexes' ) as $id ){
            $text .= "\nbasic.index " . $this -> index_id2index_str( $id ); //indexes that item is imported to.
        }
        
        return $text;
    }
    
    /**
     * 
     * return index path to the index specified by $index_id like a '/userA/books', '/Public/Paper/Science'
     * @param usernameToPrivate use '/Private' instead of '/username'
     */
    function index_id2index_str( $index_id, $usernameToPrivate = false ){
        global $xoopsDB;
        $path = array();
        
        do{
            $sql = "select t.title as title, parent_index_id, open_level from " . $xoopsDB -> prefix( "xoonips_index" ) . ", "
                . $xoopsDB -> prefix( "xoonips_item_basic" ) . " as b, "
                . $xoopsDB -> prefix( "xoonips_item_title" ) . " as t "
                . "where index_id=b.item_id and b.item_id = t.item_id and t.title_id=0 and index_id=${index_id}";
            if (!$result = $xoopsDB -> query($sql)) {
                $this -> setErrors( E_XOONIPS_DB_QUERY, $xoopsDB -> error() . $this -> getErrorAt( __LINE__, __FILE__, __FUNCTION__ ) );
                return null;
            }
            if( $xoopsDB -> getRowsNum( $result ) == 1 ){
                $row = $xoopsDB -> fetchArray( $result );
                if( $usernameToPrivate && $row['parent_index_id'] == IID_ROOT && OL_PRIVATE == $row['open_level'] ){
                    $path[] = XNP_PRIVATE_INDEX_TITLE;
                }
                else if( extension_loaded( 'mbstring' ) ){
                    $path[] = mb_ereg_replace( '/', '\\/', mb_ereg_replace( '\\\\', '\\\\', $row[ 'title' ] ) );
                }else{
                    $path[] = str_replace( array( "\\", "/" ), array( "\\\\", "\\/" ), $row[ 'title' ] );
                }
                $index_id = $row[ 'parent_index_id' ];
            }
        }while( $row[ 'parent_index_id' ] != 0 );
        array_pop( $path );//remove '/Root'
        return '/' . implode( '/', array_reverse( $path ) );
    }


    /**
     * clean values of all variables of the object for storage.
     * also add slashes whereever needed
     *
     * @return bool true if successful
     * @access public
     */
    function cleanVars(){
        $retval = true; // return value that has true if no error.
        
        $lengths = $this -> getColumnMaxLen( 'xoonips_item_title' );
        foreach( $this -> getVar( 'titles' ) as $title ){
            //
            // strip title string and set error if too long
            //
            $ret = $this -> stripSurplusString( $lengths, array( 'title' => $title ) );
            if( $ret && isset( $ret['title'] ) ){
                $this -> setErrors( E_XOONIPS_DATA_TOO_LONG, 'title is too long :' . $title . $this -> getErrorAt( __LINE__, __FILE__, __FUNCTION__ ) );
                $this -> cleanVars[ 'title' ] = $ret[ 'title' ];
                $retval = false;
            }else{
                $this -> cleanVars[ 'title' ] = $title;
            }
        }
        
        $lengths = $this -> getColumnMaxLen( 'xoonips_item_keyword' );
        foreach( $this -> getVar( 'keywords' ) as $keyword ){
            //
            // strip keyword string and set error if too long
            //
            $ret = $this -> stripSurplusString( $lengths, array( 'keyword' => $keyword ) );
            if( $ret && isset( $ret['keyword'] ) ){
                $this -> setErrors( E_XOONIPS_DATA_TOO_LONG, 'keyword is too long :' . $keyword . $this -> getErrorAt( __LINE__, __FILE__, __FUNCTION__ ) );
                $this -> cleanVars[ 'keyword' ] = $ret[ 'keyword' ];
                $retval = false;
            }else{
                $this -> cleanVars[ 'keyword' ] = $keyword;
            }
        }
        
        $lengths = $this -> getColumnMaxLen( 'xoonips_related_to' );
        foreach( $this -> getVar( 'related_to' ) as $related_to ){
            //
            // strip related_to string and set error if too long
            //
            $ret = $this -> stripSurplusString( $lengths, array( 'item_id' => $related_to ) );
            if( $ret && isset( $ret['related_to'] ) ){
                $this -> setErrors( E_XOONIPS_DATA_TOO_LONG, 'related_to is too long :' . $related_to . $this -> getErrorAt( __LINE__, __FILE__, __FUNCTION__ ) );
                $this -> cleanVars[ 'related_to' ] = $ret[ 'related_to' ];
                $retval = false;
            }else{
                $this -> cleanVars[ 'related_to' ] = $related_to;
            }
        }
        
        $lengths = $this -> getColumnMaxLen( 'xoonips_item_basic' );
        $basic = array( 'item_id' => $this -> getVar('item_id'),
                        'uid' => $this -> getVar( 'uid' ),
                        'item_type_id' => $this -> getVar( 'item_type_id' ),
                        'description' => $this -> getVar( 'description', 'none' ),
                        'doi' => $this -> getVar( 'doi' ),
                        'last_update_date' => $this -> getVar( 'last_update_date' ),
                        'creation_date' => $this -> getVar( 'creation_date' ),
                        'publication_year' => $this -> getVar( 'publication_year' ),
                        'publication_month' => $this -> getVar( 'publication_month' ),
                        'publication_mday' => $this -> getVar( 'publication_mday' ),
                        'lang' => $this -> getVar( 'lang' ) );
        $ret = $this -> stripSurplusString( $lengths, $basic );
        foreach( $basic as $k => $v ){
            if( $ret && isset( $ret[ $k ] ) ){
                $this -> setErrors( E_XOONIPS_DATA_TOO_LONG, $k . ' is too long :' . $v . $this -> getErrorAt( __LINE__, __FILE__, __FUNCTION__ ) );
                $this -> cleanVars[ $k ] = $ret[ $k ];
                $retval = false;
            }else{
                $this -> cleanVars[ $k ] = $v;
            }
        }
        return $retval;
    }

    
    /**
     * 
     * get index_id of given an array of index hierarchy
     * (ex: index_array2index_id <id of root index>, array( 'foo', 'bar' ), 'group' ); you can get group index id of /foo/bar )
     * 
     * @param index_id parent index id
     * @param indexes array of index hierarchy
     * @param open_level open_level of index of target index(public|group|private)
     * @return index id or false if index is not found.
     * 
     */
    function index_array2index_id( $index_id, $indexes, $open_level='private' ){
        global $xoopsDB;

        if( is_null( $open_level ) )  $open_level = 'private';// regards as private if $open_level is null
        if( $open_level == 'public' ){
            $open_level_val = 1;
        }else if( $open_level == 'group' ){
            $open_level_val = 2;
        }else if( $open_level == 'private' ){
            $open_level_val = 3;
        }
    
        $sql = "SELECT i.item_id FROM "
            . $xoopsDB -> prefix( "xoonips_item_basic" ) . " AS i, "
            . $xoopsDB -> prefix( "xoonips_item_title" ) . " AS t, "
            . $xoopsDB -> prefix( "xoonips_index" ) . " AS x "
            . "WHERE title_id = 0 AND x.parent_index_id = $index_id AND x.index_id = t.item_id AND t.item_id = i.item_id AND t.title = " . $xoopsDB -> quoteString( encodeMeta2Server( reset( $indexes ) ) )
            . " AND open_level = ${open_level_val}";
        $result = $xoopsDB -> query( $sql );
        if( !$result ){
            $this -> setErrors( E_XOONIPS_DB_QUERY, $xoopsDB -> error() . $this -> getErrorAt( __LINE__, __FILE__, __FUNCTION__ ) );
            return false;
        }else if( $xoopsDB->getRowsNum( $result ) == 0 ){
            $this -> setErrors( E_XOONIPS_ERROR, 'index(' . reset( $indexes ) . ") is not found in index(id=$index_id)" . $this -> getErrorAt( __LINE__, __FILE__, __FUNCTION__ ) );
            return false;
        }
        list( $next_index_id ) = $xoopsDB->fetchRow( $result );
        array_shift( $indexes );
        if( count( $indexes ) >= 1 ){
            return $this -> index_array2index_id( $next_index_id, $indexes, $open_level );
        }else{
            return $next_index_id;
        }
    }
}
?>
