/* $Id: getput.c,v 1.2 2003/01/22 22:17:31 rtakano Exp $
 * davcp: WebDAV copy
 * Copyright (c) 2002 TAKANO Ryousei <rtakano@sourceforge.jp>
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 */
#include <sys/stat.h>
#include <unistd.h>
#include <stdlib.h>
#include <string.h>
#include <fcntl.h>
#include <dirent.h>
#include <errno.h>

#include <ne_session.h>
#include <ne_basic.h>

#ifdef HAVE_CONFIG_H
#include "config.h"
#endif
#include "davcp.h"

extern int opt_verbose;


static void error_exit(ne_session* session)
{
  fprintf(stderr, "davcp: Failed: %s\n", ne_get_error(session));
  exit(1);
}


/* get file/collection
 * @session: ne_session structure
 * @local: local file name
 * @remote: remote file name
 */
int get_file(ne_session* session, const char* local, const char* remote)
{
  int fd;
  int ret;

  if (opt_verbose) {
    printf("get  : %s -> %s\n", remote, local);
  }

  /* create 'local' file. */
  fd = open(local, O_WRONLY|O_CREAT|O_TRUNC, 0644);
  if (fd == -1) {
    perror("open");
    return -1;
  }

  ret = ne_get(session, remote, fd);
  close(fd);
  if (ret != NE_OK) {
    error_exit(session);
  }
  return 0;
}

int get_collection(ne_session* session, const char* local, const char* remote)
{
  struct resource* reslist = NULL;
  struct resource* current;
  struct resource* next;
  char* col;
  int ret;

  ret = mkdir(local, 0755);
  if (ret != 0 && errno != EEXIST) {
    perror("mkdir");
    return -1;
  }

  col = ne_concat(remote, "/", NULL);
  ret = fetch_resource_list(session, col, NE_DEPTH_ONE, 0, &reslist);
  if (ret != NE_OK) {
    error_exit(session);
  }
  for (current = reslist; current != NULL; current = next) {
    next = current->next;
    switch (current->type) {
    case resr_normal:
      {
	char* lpath;
	char* p;

	p = strrchr(current->uri, '/');
	lpath = ne_concat(local, p, NULL);
	ret = get_file(session, lpath, current->uri);
	free(lpath);
	if (ret == -1) goto out;
	break;
      }
    case resr_collection:
      {
	char* lpath;
	char* p;

	if (ne_path_has_trailing_slash(current->uri)) {
	  current->uri[strlen(current->uri) - 1] = '\0';
	}
	p = strrchr(current->uri, '/');
	lpath = ne_concat(local, p, NULL);
	ret = get_collection(session, lpath, current->uri);
	free(lpath);
	if (ret == -1) goto out;
	break;
      }
    default:
      ret = -1;
      goto out;
    }
  }

 out:
  free_resource_list(reslist);
  free(col);
  return ret;
}


/* put file/collection
 * @session: ne_session structure
 * @local: local file name
 * @remote: remote file name
 */
int put_file(ne_session* session, const char* local, const char* remote)
{
  int fd;
  int ret;

  if (opt_verbose) {
    printf("put  : %s -> %s\n", local, remote);
  }

  fd = open(local, O_RDONLY);
  if (fd == -1) {
    perror("open");
    return -1;
  }

  ret = ne_put(session, remote, fd);
  close(fd);
  if (ret != NE_OK) {
    error_exit(session);
  }
  return 0;
}

int put_collection(ne_session* session, const char* local, const char* remote)
{
  DIR* dirp;
  struct dirent* dp;
  char* col;
  time_t time;
  int ret;

  dirp = opendir(local);
  if (dirp == NULL) {
    perror("opendir");
    return -1;
  }

  /* TODO: a very ad-hoc mkcol code. */
  col = ne_concat(remote, "/", NULL);
  ret = ne_getmodtime(session, col, &time);
  if (ret == NE_ERROR) {
    if (strncmp(ne_get_error(session), "404", 3) == 0) { /* 404: not found */
      if (opt_verbose) {
	printf("mkcol: %s\n", col);
      }
      ret = ne_mkcol(session, col);
      if (ret != NE_OK) {
	fprintf(stderr, "davcp: can not mkcol: %s.\n", col);
	ret = -1;
	goto out;
      }
    }
  }

  ret = 0;
  while (ret != -1 && (dp = readdir(dirp)) != NULL) {
    char* lpath;
    char* rpath;

    if (dp->d_ino == 0) {
      continue;
    }
    if (strcmp(dp->d_name, ".") == 0 || strcmp(dp->d_name, "..") == 0) {
      continue;
    }

    lpath = ne_concat(local, "/", dp->d_name, NULL);
    rpath = ne_concat(remote, "/", dp->d_name, NULL);
    if (dp->d_type == DT_DIR) {
      ret = put_collection(session, lpath, rpath);
    } else {
      ret = put_file(session, lpath, rpath);
    }
    free(rpath);
    free(lpath);
  }

 out:
  free(col);
  closedir(dirp);
  return ret;
}
