// IMKit: An inputmethod adaptation library for Qtopia environment
// Copyright (C) 2002-2004  YamaKen <yamaken@bp.iij4u.or.jp>
//
// This program is free software; you can redistribute it and/or modify
// it under the terms of the GNU General Public License as published by
// the Free Software Foundation; either version 2 of the License, or
// (at your option) any later version.
//
// This program is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
// GNU General Public License for more details.
//
// You should have received a copy of the GNU General Public License
// along with this program; if not, write to the Free Software
// Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA

// $Name: IMKIT_0_4_4 $
// $Id: keyfilter.h,v 1.12 2004/03/07 07:07:43 yamaken Exp $

#ifndef IMKIT_KEYFILTER_H
#define IMKIT_KEYFILTER_H

#include <map>
#include <string>
#include <qobject.h>
#include <qevent.h>
#include "util.h"


#define IMKIT_NON_UNICODE_CHAR 0xffff

#define IMKIT_BTNMAPPER_SHIFT_MODIFIER(modifier) (modifier << 16)
#define IMKIT_BTNMAPPER_MOD_SHIFT \
  IMKIT_BTNMAPPER_SHIFT_MODIFIER(Qt::ShiftButton)
#define IMKIT_BTNMAPPER_MOD_CTRL \
  IMKIT_BTNMAPPER_SHIFT_MODIFIER(Qt::ControlButton)
#define IMKIT_BTNMAPPER_MOD_ALT \
  IMKIT_BTNMAPPER_SHIFT_MODIFIER(Qt::AltButton)

#define IMKIT_ASCII_MOD_SHIFT(_ascii) (_ascii - ' ')
#define IMKIT_ASCII_MOD_CTRL(_ascii) (_ascii - '`')

#define IMKIT_REGISTER_CMD0(_map, _receiver_type, _receiver, _method_name) \
  ((_map)[#_method_name] = new TemplateCommand<_receiver_type>(_receiver, \
                                 &_receiver_type::cmd_##_method_name, \
                                 #_method_name))

#define IMKIT_REGISTER_CMD1(_map, _receiver_type, _receiver, _method_name) \
  ((_map)[#_method_name] = new TemplateCommandWithChar<_receiver_type>( \
                                 _receiver, \
                                 &_receiver_type::cmd_##_method_name, \
                                 #_method_name))

class Command;

typedef map<string, Command *> CommandMap;

void imkit_delete_command_map(CommandMap *command_map);
unsigned char imkit_strip_modifier(unsigned char chr);
unsigned char imkit_strip_control(unsigned char chr);

class KeyEventConverter : public QObject {
  Q_OBJECT

public:
  static QChar extract_qchar(const QKeyEvent &e);

  KeyEventConverter(void);
  virtual ~KeyEventConverter(void);
  QKeyEvent *create(int unicode, int keycode, int modifiers,
                    bool is_press, bool autorepeat);

signals:
  void created(QKeyEvent *e);
  void converted(int unicode, int keycode, int modifiers,
                 bool is_press, bool autorepeat);

public slots:
  virtual void create_by_signal(int unicode, int keycode, int modifiers,
                                bool is_press, bool autorepeat);
  virtual void convert(QKeyEvent *e);
};


// CascadeKeyFilter
//
// - QKeyEvent eoutput(e)顢output(e)θƤ
//   Фdelete e
// - If you output(e) that e is newly generated, caller of
//   output(e) must delete e.
// 
// - on_input(e)ǥϤդe.accept()
//   ʹߤϢʤCascadeKeyFiltere̵뤵
// - Invoke e.accept() if on_input(e) processed e
//   successfully. This guarrantees e is ignored on succeeding
//   CascadeKeyFilters.

class CascadeKeyFilter : public virtual Activatable {
protected:
  CascadeKeyFilter *_prev, *_next;

protected:
  static void insert(CascadeKeyFilter *to_be_prev,
                     CascadeKeyFilter *to_be_next,
                     CascadeKeyFilter *other);
  static void remove(CascadeKeyFilter *begin, CascadeKeyFilter *end);

  virtual void on_input(QKeyEvent &e);  //using reference to prevent delete e
  virtual void output(QKeyEvent *e);

public:
  CascadeKeyFilter(void);
  CascadeKeyFilter(const CascadeKeyFilter &orig);
  virtual ~CascadeKeyFilter(void);
  virtual CascadeKeyFilter *prev(void);
  virtual CascadeKeyFilter *next(void);
  virtual void insert_before(CascadeKeyFilter *other);
  virtual void insert_after(CascadeKeyFilter *other);
  virtual CascadeKeyFilter &operator<<(CascadeKeyFilter &other);
  virtual void remove(void);
  virtual void remove_till(CascadeKeyFilter *end);
  virtual void input(QKeyEvent *e);
};


class DumbPipe : public CascadeKeyFilter, public StdActivatable {
};


class CharInjector : public QObject, public CascadeKeyFilter,
                     public StdActivatable
{

  Q_OBJECT

public:
  CharInjector(void);
  virtual ~CharInjector(void);

public slots:
  virtual void inject_char(QChar chr);
  virtual void inject_str(const QString &str);
  virtual void commit(const QString &str);
};


class Command {
public:
  virtual ~Command(void);
  virtual const char *name(void) const = 0;
  virtual void execute(QKeyEvent &e) = 0;  //using reference to prevent delete e
};


template <class Receiver>
class TemplateCommandBase : public Command {
protected:
  Receiver *cmd_receiver;
  const char *_name;

public:
  TemplateCommandBase(Receiver *cmd_receiver_init, const char *name_init);
  virtual const char *name(void) const { return _name; }
};


template <class Receiver>
class TemplateCommand : public TemplateCommandBase<Receiver> {
public:
  typedef void (Receiver:: *CommandMethod)(void);

protected:
  CommandMethod cmd_method;
  
public:
  TemplateCommand(Receiver *cmd_receiver_init, CommandMethod cmd_method_init,
                  const char *name_init);
  virtual void execute(QKeyEvent &e);
};


template <class Receiver, class Arg>
class TemplateCommandWithPreArg : public TemplateCommandBase<Receiver> {
public:
  typedef void (Receiver:: *CommandMethod)(Arg);

protected:
  CommandMethod cmd_method;
  Arg arg;
  
public:
  TemplateCommandWithPreArg(Receiver *cmd_receiver_init,
                            CommandMethod cmd_method_init,
                            const char *name_init, Arg arg_init);
  virtual void execute(QKeyEvent &e);
};


template <class Receiver>
class TemplateCommandWithChar : public TemplateCommandBase<Receiver> {
public:
  typedef void (Receiver:: *CommandMethod)(QChar chr);

protected:
  CommandMethod cmd_method;

public:
  TemplateCommandWithChar(Receiver *cmd_receiver_init,
                          CommandMethod cmd_method_init,
                          const char *name_init);
  virtual void execute(QKeyEvent &e);
};


template <class Receiver>
class TemplateCommandWithKeyEvent : public TemplateCommandBase<Receiver> {
public:
  typedef void (Receiver:: *CommandMethod)(QKeyEvent &e);

protected:
  CommandMethod cmd_method;

public:
  TemplateCommandWithKeyEvent(Receiver *cmd_receiver_init,
                              CommandMethod cmd_method_init,
                              const char *name_init);
  virtual void execute(QKeyEvent &e);
};


class DoNothingCommand : public Command {
protected:
  const char *_name;

public:
  inline DoNothingCommand(const char *name_init = 0) : _name(name_init) {}
  virtual ~DoNothingCommand(void);
  virtual const char *name(void) const;
  virtual void execute(QKeyEvent &e);
};


template <class CommandType = Command>
class RawMapper : public CascadeKeyFilter, public StdActivatable {
public:
  typedef CommandType command_t;

protected:
  command_t *command;

protected:
  virtual void on_input(QKeyEvent &e);

public:
  inline RawMapper(command_t *command_init) : command(command_init) {}
  virtual ~RawMapper(void);
};


template <class KeyType, class CommandType = Command>
class KeyMapper : public CascadeKeyFilter, public StdActivatable {
public:
  typedef KeyType mapkey_t;
  typedef CommandType command_t;

protected:
  virtual void on_input(QKeyEvent &e) = 0;

public:
  virtual ~KeyMapper(void);
};


template <class CommandType = Command>
class RawCharMapper : public KeyMapper<QChar, CommandType> {
protected:
  command_t *press_command, *release_command;

protected:
  virtual void on_input(QKeyEvent &e);

public:
  RawCharMapper(command_t *press_command_init,
                command_t *release_command_init);
};


template <class CommandType = Command>
class GraphCharMapper : public KeyMapper<QChar, CommandType> {
protected:
  command_t *command;

protected:
  virtual void on_input(QKeyEvent &e);

public:
  inline GraphCharMapper(command_t *command_init) : command(command_init) {}
};


template <class KeyType, class CommandType = Command>
class StaticKeyMapper : public KeyMapper<KeyType, CommandType> {
public:
  typedef map<string, command_t *> commandmap_t;
  typedef map<mapkey_t, command_t *> keymap_t;
  struct keymapdef_t {
    mapkey_t key;
    const char *command_name;
  };

protected:
  keymap_t *keymap;

protected:
  static keymap_t *construct_keymap(keymapdef_t *keymap_defs,
                                    commandmap_t &command_map);

  virtual void on_input(QKeyEvent &e);
  virtual mapkey_t extract_key(QKeyEvent &e) const = 0;

public:
  StaticKeyMapper(void);
  StaticKeyMapper(keymapdef_t *keymap_defs, commandmap_t &command_map);
  StaticKeyMapper(const StaticKeyMapper &orig);
  StaticKeyMapper &operator=(const StaticKeyMapper &orig);
  virtual ~StaticKeyMapper(void);
  virtual void add_map(mapkey_t key, command_t *command);
};


template <class CommandType = Command>
class QCharKeyMapper : public StaticKeyMapper<QChar, CommandType> {
protected:
  virtual mapkey_t extract_key(QKeyEvent &e) const;

public:
  QCharKeyMapper(keymapdef_t *keymap_defs, commandmap_t &command_map);
};


template <class CommandType = Command>
class OneShotKeyMapper : public QCharKeyMapper<CommandType> {
protected:
  virtual void on_input(QKeyEvent &e);

public:
  OneShotKeyMapper(keymapdef_t *keymap_defs, commandmap_t &command_map);
};


template <class CommandType = Command>
class ButtonMapper : public StaticKeyMapper<int, CommandType> {
protected:
  virtual mapkey_t extract_key(QKeyEvent &e) const;

public:
  ButtonMapper(keymapdef_t *keymap_defs, commandmap_t &command_map);
};

//////////////////////////////////////////////////////////////////////////////

template <class CommandType>
RawMapper<CommandType>::~RawMapper(void) {
}

template <class CommandType>
void
RawMapper<CommandType>::on_input(QKeyEvent &e) {
  command->execute(e);
}

template <class KeyType, class CommandType>
KeyMapper<KeyType, CommandType>::~KeyMapper(void) {
}

template <class CommandType>
RawCharMapper<CommandType>::RawCharMapper(command_t *press_command_init,
                                          command_t *release_command_init)
  : press_command(press_command_init), release_command(release_command_init)
{
}

template <class CommandType>
void
RawCharMapper<CommandType>::on_input(QKeyEvent &e) {
  bool is_keypress;

  is_keypress = (e.type() == QEvent::KeyPress);
  if (is_keypress) {
    press_command->execute(e);
  } else {
    release_command->execute(e);
  }
  e.accept();
}

template <class CommandType>
void
GraphCharMapper<CommandType>::on_input(QKeyEvent &e) {
  mapkey_t key;
  bool is_keypress, is_graph;

  key = KeyEventConverter::extract_qchar(e);
  is_keypress = (e.type() == QEvent::KeyPress);
  is_graph = (key.isPrint() && !key.isSpace()
              && key != QChar(IMKIT_NON_UNICODE_CHAR));

  if (is_graph) {
    if (is_keypress) {
      command->execute(e);
    }
    e.accept();
  }
}

template <class KeyType, class CommandType>
StaticKeyMapper<KeyType, CommandType>::keymap_t *
StaticKeyMapper<KeyType, CommandType>::
construct_keymap(keymapdef_t *keymap_defs, commandmap_t &command_map)
{
  keymap_t *_keymap;
  keymapdef_t *pair;
  command_t *command;
  
  _keymap = new keymap_t;
  for (pair = keymap_defs; *pair->command_name; pair++) {
    command = command_map[pair->command_name];
    (*_keymap)[pair->key] = command;
  }
  
  return _keymap;
}

template <class KeyType, class CommandType>
void
StaticKeyMapper<KeyType, CommandType>::on_input(QKeyEvent &e) {
  mapkey_t key;
  command_t *command;
  bool is_keypress, mapped;

  key = extract_key(e);
  is_keypress = (e.type() == QEvent::KeyPress);
  mapped = (keymap->find(key) != keymap->end());

  if (mapped) {
    if (is_keypress) {
      command = (*keymap)[key];
      command->execute(e);
    }
    e.accept();
  }
}

template <class KeyType, class CommandType>
StaticKeyMapper<KeyType, CommandType>::StaticKeyMapper(void)
  : keymap(NULL)
{
  keymap = new keymap_t;
  activate();
}

template <class KeyType, class CommandType>
StaticKeyMapper<KeyType, CommandType>::
StaticKeyMapper(keymapdef_t *keymap_defs, commandmap_t &command_map)
  : keymap(NULL)
{
  keymap = construct_keymap(keymap_defs, command_map);
  activate();
}

template <class KeyType, class CommandType>
StaticKeyMapper<KeyType, CommandType>::
StaticKeyMapper(const StaticKeyMapper &orig)
  : CascadeKeyFilter(orig), StdActivatable(), keymap(NULL)
{
  operator=(orig);
  activate();
}

template <class KeyType, class CommandType>
StaticKeyMapper<KeyType, CommandType> &
StaticKeyMapper<KeyType, CommandType>::operator=(const StaticKeyMapper &orig) {
  if (keymap) {
    delete keymap;
  }
  keymap = new keymap_t(*orig.keymap);

  return *this;
}

template <class KeyType, class CommandType>
StaticKeyMapper<KeyType, CommandType>::~StaticKeyMapper(void) {
  if (keymap) {
    delete keymap;
  }
}

template <class KeyType, class CommandType>
void
StaticKeyMapper<KeyType, CommandType>::add_map(mapkey_t key, command_t *command) {
  if (keymap) {
    (*keymap)[key] = command;
  }
}


template <class CommandType>
QCharKeyMapper<CommandType>::mapkey_t
QCharKeyMapper<CommandType>::extract_key(QKeyEvent &e) const {
  return KeyEventConverter::extract_qchar(e);
}

template <class CommandType>
QCharKeyMapper<CommandType>::QCharKeyMapper(keymapdef_t *keymap_defs,
                                            commandmap_t &command_map)
  : StaticKeyMapper<QChar, CommandType>(keymap_defs, command_map)
{
}

template <class CommandType>
void
OneShotKeyMapper<CommandType>::on_input(QKeyEvent &e) {
  bool is_keypress;

  is_keypress = (e.type() == QEvent::KeyPress);
  QCharKeyMapper<CommandType>::on_input(e);
  if (e.isAccepted() && is_keypress) {
    deactivate();
  } else {
    e.accept();
  }
}

template <class CommandType>
OneShotKeyMapper<CommandType>::OneShotKeyMapper(keymapdef_t *keymap_defs,
                                   commandmap_t &command_map)
  : QCharKeyMapper<CommandType>(keymap_defs, command_map)
{
}

template <class CommandType>
ButtonMapper<CommandType>::mapkey_t
ButtonMapper<CommandType>::extract_key(QKeyEvent &e) const {
  mapkey_t merged_key;

  merged_key = (IMKIT_BTNMAPPER_SHIFT_MODIFIER(e.state()) | e.key());

  return merged_key;
}

template <class CommandType>
ButtonMapper<CommandType>::ButtonMapper(keymapdef_t *keymap_defs,
                                        commandmap_t &command_map)
  : StaticKeyMapper<int, CommandType>(keymap_defs, command_map)
{
}

template <class Receiver>
TemplateCommandBase<Receiver>::TemplateCommandBase(Receiver *cmd_receiver_init,
                                                   const char *name_init)
  : cmd_receiver(cmd_receiver_init), _name(name_init)
{
}


template <class Receiver>
TemplateCommand<Receiver>::TemplateCommand(Receiver *cmd_receiver_init,
                                           CommandMethod cmd_method_init,
                                           const char *name_init)
  : TemplateCommandBase<Receiver>(cmd_receiver_init, name_init),
    cmd_method(cmd_method_init)
{
}

template <class Receiver>
void
TemplateCommand<Receiver>::execute(QKeyEvent &e) {
  (cmd_receiver->*cmd_method)();
}

template <class Receiver, class Arg>
TemplateCommandWithPreArg<Receiver, Arg>::
TemplateCommandWithPreArg(Receiver *cmd_receiver_init,
                          CommandMethod cmd_method_init,
                          const char *name_init, Arg arg_init)
  : TemplateCommandBase<Receiver>(cmd_receiver_init, name_init),
    cmd_method(cmd_method_init),
    arg(arg_init)
{
}

template <class Receiver, class Arg>
void
TemplateCommandWithPreArg<Receiver, Arg>::execute(QKeyEvent &e) {
  (cmd_receiver->*cmd_method)(arg);
}

template <class Receiver>
TemplateCommandWithChar<Receiver>::
TemplateCommandWithChar(Receiver *cmd_receiver_init,
                        CommandMethod cmd_method_init,
                        const char *name_init)
  : TemplateCommandBase<Receiver>(cmd_receiver_init, name_init),
    cmd_method(cmd_method_init)
{
}

template <class Receiver>
void
TemplateCommandWithChar<Receiver>::execute(QKeyEvent &e) {
  QChar chr;
  
  chr = KeyEventConverter::extract_qchar(e);
  (cmd_receiver->*cmd_method)(chr);
}

template <class Receiver>
TemplateCommandWithKeyEvent<Receiver>::
TemplateCommandWithKeyEvent(Receiver *cmd_receiver_init,
                            CommandMethod cmd_method_init,
                            const char *name_init)
  : TemplateCommandBase<Receiver>(cmd_receiver_init, name_init),
    cmd_method(cmd_method_init)
{
}

template <class Receiver>
void
TemplateCommandWithKeyEvent<Receiver>::execute(QKeyEvent &e) {
  (cmd_receiver->*cmd_method)(e);
}

#endif  // IMKIT_KEYFILTER_H
