import java.awt.event.*;

import java.util.*;


/*
 * 作成日: 2004/01/19
 */

/**
 * 地図の読み込み、ポリゴンの結合、塗り分け、属性配置位置の計算、再描画など、
 * バックグラウンドで行う作業を管理するクラスです。
 * @author Kumano Tatsuo
 */
class BackgroundThread extends Thread {
    private Map maps; // 地図
    private MapPanel panel; // パネル
    private boolean isChanged; // 地図の状態が変化したかどうか

    /**
     * バックグラウンドで行う処理を初期化します。
     * @param maps 地図
     * @param panel パネル
     */
    BackgroundThread(Map maps, MapPanel panel) {
        this.maps = maps;
        this.panel = panel;
        isChanged = true;

        panel.addMouseListener(new MouseAdapter() {
                public void mouseReleased(MouseEvent e) {
                    isChanged = true;
                }
            });
        panel.addMouseWheelListener(new MouseWheelListener() {
                public void mouseWheelMoved(MouseWheelEvent e) {
                    isChanged = true;
                }
            });
        panel.addComponentListener(new ComponentAdapter() {
                public void componentResized(ComponentEvent e) {
                    isChanged = true;
                }
            });
    }

    /**
     * スレッドが開始されるときに呼び出されます。
     */
    public void run() {
        try {
            while (true) {
                if (panel.isIdle() && isChanged) {
                    System.out.println("計算中です。");
                    panel.setIsBusy(true);
                    loadMapPaintTyomeJoinTyome();
                    System.out.println("計算が終了しました。");
                    panel.setIsBusy(false);
                    isChanged = false;
                }

                Thread.sleep(200);
            }
        } catch (InterruptedException e) {
            e.printStackTrace(System.err);
        }
    }

    /**
     * 地図の読み込み、ポリゴンの結合、塗り分け、属性配置位置の計算を行います。
     */
    void loadMapPaintTyomeJoinTyome() {
        try {
            long startTime = System.currentTimeMillis();
            LoadMap loadMap = new LoadMap();
            loadMap.loadMap(maps, (MapPanel) panel);

            if ((System.currentTimeMillis() - startTime) > 200) {
                System.out.println("地図の読み込み："
                    + (System.currentTimeMillis() - startTime) + " ms");
            }

            startTime = System.currentTimeMillis();

            PaintTyome paintTyome = new PaintTyome();
            paintTyome.paintTyome(maps);

            if ((System.currentTimeMillis() - startTime) > 200) {
                System.out.println("塗り分け："
                    + (System.currentTimeMillis() - startTime) + " ms");
            }

            startTime = System.currentTimeMillis();

            JoinPolygon joinTyome = new JoinPolygon();
            joinTyome.joinPolygon(maps);

            JoinTatemono joinTatemono = new JoinTatemono();
            joinTatemono.joinTatemono(maps);

            if ((System.currentTimeMillis() - startTime) > 200) {
                System.out.println("ポリゴンの結合："
                    + (System.currentTimeMillis() - startTime) + " ms");
            }

            startTime = System.currentTimeMillis();

            FixAttributeLocation fixAttributeLocation = new FixAttributeLocation();
            fixAttributeLocation.fixAttributeLocation(maps, (MapPanel) panel);

            if ((System.currentTimeMillis() - startTime) > 200) {
                System.out.println("属性描画位置の計算："
                    + (System.currentTimeMillis() - startTime) + " ms");
            }

            panel.setNeedsRepaint(true);
            panel.repaint();
        } catch (Exception exception) {
            System.err.println("Failed to load map.");
            exception.printStackTrace(System.err);
        }
    }
}
