import java.awt.*;
import java.awt.geom.*;

import java.util.*;


/*
 * 作成日: 2004/01/16
 */

/**
 * 属性を描画する座標を計算するクラスです。
 * @author Kumano Tatsuo
 */
class FixAttributeLocation {
    private boolean isChanged; // 直前の計算で属性を描画する座標が変化したかどうか

    /** 属性を描画する座標を決定します。
         * @param maps 地図
         * @param panel パネル
         * @throws Exception 例外
         */
    void fixAttributeLocation(Map maps, MapPanel panel)
        throws Exception {
        isChanged = false;

        Font tatemonoFont = new Font("MS UI Gothic", Font.PLAIN, 12);
        Font zyoutiFont = new Font("MS UI Gothic", Font.PLAIN, 15);
        Font mizuFont = new Font("ＭＳ Ｐ明朝", Font.PLAIN, 14);
        Font tyomeFont = new Font("MS UI Gothic", Font.PLAIN, 18);
        Font tyomeFont2 = new Font("MS UI Gothic", Font.PLAIN, 16);
        Font tyomeFont3 = new Font("MS UI Gothic", Font.PLAIN, 14);
        Font ekiFont = new Font("MS UI Gothic", Font.PLAIN, 20);
        Font roadFont = new Font("MS UI GOthic", Font.PLAIN, 15);
        Font tetudouFont = new Font("MS UI GOthic", Font.PLAIN, 16);
        Area usedArea = new Area(); // 属性によって使用済みの領域（仮想座標）
        Rectangle2D visibleRectangle = panel.getVisibleRectangle(); // 現在画面に表示されている範囲（仮想座標）
        Collection usedPoints = new HashSet(); // 既に配置されたポリゴンの中心座標（仮想座標）
        int loopCount = 4;
        double tatemonoPointSize = 4; // 建物に表示する点の直径
        double ekiPointSize = 8; // 駅に表示する点の直径
        double zoom = panel.getZoom();
        double x = panel.getVisibleRectangle().getX();
        double y = panel.getVisibleRectangle().getY();
        double w = panel.getVisibleRectangle().getWidth();
        double h = panel.getVisibleRectangle().getHeight();

        // 駅の属性の表示位置を計算する
        for (Iterator iter = maps.values().iterator(); iter.hasNext();) {
            MapData mapData = (MapData) iter.next();

            if (mapData.hasEki()) {
                if (panel.isVisible(mapData.getBounds())) {
                    for (Iterator iter2 = mapData.getEki().values().iterator();
                            iter2.hasNext();) {
                        PointData point = (PointData) iter2.next();
                        Rectangle2D pointRectangle = new Rectangle2D.Double(point
                                .getX() - (ekiPointSize / 2 / zoom),
                                point.getY() - (ekiPointSize / 2 / zoom),
                                ekiPointSize / zoom, ekiPointSize / zoom);
                        usedArea.add(new Area(pointRectangle));
                    }
                }
            }
        }

        for (Iterator iter = maps.values().iterator(); iter.hasNext();) {
            MapData mapData = (MapData) iter.next();

            if (mapData.hasEki()) {
                if (panel.isVisible(mapData.getBounds())) {
                    FontMetrics metrics = panel.getFontMetrics(ekiFont);
                    double attributeHeight = metrics.getHeight() / zoom;

                    for (Iterator iter2 = mapData.getEki().values().iterator();
                            iter2.hasNext();) {
                        PointData point = (PointData) iter2.next();
                        fixPointAttributeLocation(point, usedArea, usedPoints,
                            visibleRectangle, zoom, attributeHeight, metrics,
                            ekiPointSize);
                    }
                }
            }
        }

        // 建物の属性の表示位置を計算する
        long startTime = System.currentTimeMillis();

        for (Iterator iter = maps.values().iterator(); iter.hasNext();) {
            MapData mapData = (MapData) iter.next();

            if (mapData.hasTatemono()) {
                if (panel.isVisible(mapData.getBounds())) {
                    FontMetrics metrics = panel.getFontMetrics(tatemonoFont);
                    double attributeHeight = metrics.getHeight() / zoom;

                    for (Iterator iter2 = mapData.getTatemono().values()
                                                 .iterator(); iter2.hasNext();) {
                        PolygonData polygon = (PolygonData) iter2.next();
                        fixTatemonoAttributeLocation(polygon, usedArea,
                            usedPoints, visibleRectangle, zoom,
                            attributeHeight, metrics, tatemonoPointSize);
                    }
                }
            }
        }

        if ((System.currentTimeMillis() - startTime) > 200) {
            System.out.println("（建物の属性配置："
                + (System.currentTimeMillis() - startTime) + " ms）");
        }

        // 場地の属性の表示位置を計算する
        for (Iterator iter = maps.values().iterator(); iter.hasNext();) {
            MapData mapData = (MapData) iter.next();

            if (mapData.hasZyouti()) {
                if (panel.isVisible(mapData.getBounds())) {
                    FontMetrics metrics = panel.getFontMetrics(zyoutiFont);
                    double attributeHeight = metrics.getHeight() / zoom;

                    for (Iterator iter2 = mapData.getZyouti().values().iterator();
                            iter2.hasNext();) {
                        PolygonData polygon = (PolygonData) iter2.next();
                        fixPolygonAttributeLocation(polygon, usedArea,
                            usedPoints, visibleRectangle, zoom,
                            attributeHeight, metrics, tatemonoPointSize);
                    }
                }
            }
        }

        // 内水面の属性の表示位置を計算する
        for (Iterator iter = maps.values().iterator(); iter.hasNext();) {
            MapData mapData = (MapData) iter.next();

            if (mapData.hasMizu()) {
                if (panel.isVisible(mapData.getBounds())) {
                    FontMetrics metrics = panel.getFontMetrics(mizuFont);
                    double attributeHeight = metrics.getHeight() / zoom;

                    for (Iterator iter2 = mapData.getMizu().values().iterator();
                            iter2.hasNext();) {
                        PolygonData polygon = (PolygonData) iter2.next();
                        fixPolygonAttributeLocation(polygon, usedArea,
                            usedPoints, visibleRectangle, zoom,
                            attributeHeight, metrics, tatemonoPointSize);
                    }
                }
            }
        }

        // 道路の属性の表示位置を計算する
        {
            FontMetrics metrics = panel.getFontMetrics(roadFont);
            double attributeHeight = metrics.getHeight() / zoom;
            Collection fixedAttributes = new HashSet();

            for (Iterator iter = maps.values().iterator(); iter.hasNext();) {
                MapData mapData = (MapData) iter.next();

                if (mapData.hasRoadArc()) {
                    if (panel.isVisible(mapData.getBounds())) {
                        for (Iterator iter2 = mapData.getRoadArc().values()
                                                     .iterator();
                                iter2.hasNext();) {
                            ArcData arc = (ArcData) iter2.next();
                            String attribute = arc.getAttribute();

                            if (attribute != null) {
                                arc.setAttributeLocation(0, 0);

                                if (!fixedAttributes.contains(attribute)) {
                                    double attributeWidth = metrics.stringWidth(arc
                                            .getAttribute()) / zoom;
                                    Point2D currentPoint = arc.getPath()
                                                              .getCurrentPoint();
                                    Rectangle2D attributeRectangle = new Rectangle2D.Double(currentPoint
                                            .getX() - (attributeWidth / 2),
                                            currentPoint.getY()
                                            - attributeHeight, attributeWidth,
                                            attributeHeight);

                                    if (visibleRectangle.contains(
                                                attributeRectangle)
                                            && !usedArea.intersects(
                                                attributeRectangle)) {
                                        arc.setAttributeLocation(currentPoint
                                            .getX() - (attributeWidth / 2),
                                            currentPoint.getY());
                                        fixedAttributes.add(attribute);
                                        usedArea.add(new Area(
                                                attributeRectangle));
                                    }
                                }
                            }
                        }
                    }
                }
            }
        }

        // 鉄道の属性の表示位置を計算する
        {
            FontMetrics metrics = panel.getFontMetrics(tetudouFont);
            double attributeHeight = metrics.getHeight() / zoom;
            Collection fixedAttributes = new HashSet();

            for (Iterator iter = maps.values().iterator(); iter.hasNext();) {
                MapData mapData = (MapData) iter.next();

                if (mapData.hasOthers()) {
                    if (panel.isVisible(mapData.getBounds())) {
                        for (Iterator iter2 = mapData.getOthers().values()
                                                     .iterator();
                                iter2.hasNext();) {
                            ArcData arc = (ArcData) iter2.next();
                            String attribute = arc.getAttribute();

                            if (attribute != null) {
                                arc.setAttributeLocation(0, 0);

                                if (!fixedAttributes.contains(attribute)) {
                                    double attributeWidth = metrics.stringWidth(arc
                                            .getAttribute()) / zoom;
                                    PathIterator iter3 = arc.getPath()
                                                            .getPathIterator(new AffineTransform());
                                    double[] coords = new double[6];

                                    while (!iter3.isDone()) {
                                        int currentSegment = iter3
                                            .currentSegment(coords);

                                        if (currentSegment == PathIterator.SEG_LINETO) {
                                            double currentX = coords[0];
                                            double currentY = coords[1];
                                            Rectangle2D attributeRectangle = new Rectangle2D.Double(currentX
                                                    - (attributeWidth / 2),
                                                    currentY - attributeHeight,
                                                    attributeWidth,
                                                    attributeHeight);

                                            if (visibleRectangle.contains(
                                                        attributeRectangle)
                                                    && !usedArea.intersects(
                                                        attributeRectangle)) {
                                                arc.setAttributeLocation(currentX
                                                    - (attributeWidth / 2),
                                                    currentY);
                                                fixedAttributes.add(attribute);
                                                usedArea.add(new Area(
                                                        attributeRectangle));

                                                break;
                                            }
                                        }

                                        iter3.next();
                                    }
                                }
                            }
                        }
                    }
                }
            }
        }

        // 丁目の属性の表示位置を計算する
        startTime = System.currentTimeMillis();

        for (Iterator iter = maps.values().iterator(); iter.hasNext();) {
            MapData mapData = (MapData) iter.next();

            if (mapData.hasTyome()) {
                if (panel.isVisible(mapData.getBounds())) {
                    for (Iterator iter2 = mapData.getTyome().values().iterator();
                            iter2.hasNext();) {
                        PolygonData polygon = (PolygonData) iter2.next();
                        polygon.setAttributeLocation(0, 0);
                        fixTyomeAttributeLocation(polygon, tyomeFont,
                            visibleRectangle, usedArea, usedPoints, panel, true);
                        fixTyomeAttributeLocation(polygon, tyomeFont2,
                            visibleRectangle, usedArea, usedPoints, panel, true);
                        fixTyomeAttributeLocation(polygon, tyomeFont3,
                            visibleRectangle, usedArea, usedPoints, panel, false);
                    }
                }
            }
        }

        if ((System.currentTimeMillis() - startTime) > 200) {
            System.out.println("（丁目の属性配置："
                + (System.currentTimeMillis() - startTime) + " ms）");
        }

        isChanged = true;
    }

    private void fixTyomeAttributeLocation(PolygonData polygon, Font font,
        Rectangle2D visibleRectangle, Area usedArea, Collection usedPoints,
        MapPanel panel, boolean contains) {
        if (polygon.getAttribute() != null) {
            double zoom = panel.getZoom();
            FontMetrics metrics = panel.getFontMetrics(font);
            double attributeHeight = metrics.getHeight() / zoom;
            double size = 4;
            double attributeWidth = metrics.stringWidth(polygon.getAttribute()) / zoom;

            if (contains) {
                if (polygon.getArea().getBounds().getWidth() < attributeWidth) {
                    return;
                }
            }

            Rectangle2D pointRectangle = new Rectangle2D.Double(-polygon.getX()
                    - (size / zoom), -polygon.getY() - (size / zoom),
                    (size * 2) / zoom, (size * 2) / zoom);
            Rectangle2D attributeRectangle;

            // ポリゴンの中心
            attributeRectangle = new Rectangle2D.Double(polygon.getX()
                    - (attributeWidth / 2),
                    polygon.getY() - (attributeHeight / 2), attributeWidth,
                    attributeHeight);

            Point point = new Point((int) polygon.getX(), (int) polygon.getY());

            if (isTyomePutable(polygon, point, usedPoints, attributeRectangle,
                        usedArea, visibleRectangle, contains)) {
                polygon.setAttributeLocation(attributeRectangle.getX(),
                    attributeRectangle.getMaxY());
                polygon.setTyomeFont(font);
                usedArea.add(new Area(pointRectangle));
                usedArea.add(new Area(attributeRectangle));
                usedPoints.add(point);
            }

            int div = 4; // 分割数
            double dx; // 動かす幅
            double dy; // 動かす高さ
            dx = (polygon.getArea().getBounds().getWidth() - attributeWidth) / 2 / div;
            dy = (polygon.getArea().getBounds().getHeight() - attributeHeight) / 2 / div;

            for (int i = 1; i <= div; i++) {
                // ポリゴンの中心から上
                attributeRectangle = new Rectangle2D.Double(polygon.getX()
                        - (attributeWidth / 2),
                        polygon.getY() - (attributeHeight / 2) - (dy * i),
                        attributeWidth, attributeHeight);

                if (isTyomePutable(polygon, point, usedPoints,
                            attributeRectangle, usedArea, visibleRectangle,
                            contains)) {
                    polygon.setAttributeLocation(attributeRectangle.getX(),
                        attributeRectangle.getMaxY());
                    polygon.setTyomeFont(font);
                    usedArea.add(new Area(pointRectangle));
                    usedArea.add(new Area(attributeRectangle));
                    usedPoints.add(point);
                }

                // ポリゴンの中心から下
                attributeRectangle = new Rectangle2D.Double(polygon.getX()
                        - (attributeWidth / 2),
                        polygon.getY() - (attributeHeight / 2) + (dy * i),
                        attributeWidth, attributeHeight);

                if (isTyomePutable(polygon, point, usedPoints,
                            attributeRectangle, usedArea, visibleRectangle,
                            contains)) {
                    polygon.setAttributeLocation(attributeRectangle.getX(),
                        attributeRectangle.getMaxY());
                    polygon.setTyomeFont(font);
                    usedArea.add(new Area(pointRectangle));
                    usedArea.add(new Area(attributeRectangle));
                    usedPoints.add(point);
                }

                // ポリゴンの中心から右
                attributeRectangle = new Rectangle2D.Double(polygon.getX()
                        - (attributeWidth / 2) + (dx * i),
                        polygon.getY() - (attributeHeight / 2), attributeWidth,
                        attributeHeight);

                if (isTyomePutable(polygon, point, usedPoints,
                            attributeRectangle, usedArea, visibleRectangle,
                            contains)) {
                    polygon.setAttributeLocation(attributeRectangle.getX(),
                        attributeRectangle.getMaxY());
                    polygon.setTyomeFont(font);
                    usedArea.add(new Area(pointRectangle));
                    usedArea.add(new Area(attributeRectangle));
                    usedPoints.add(point);
                }

                // ポリゴンの中心から左
                attributeRectangle = new Rectangle2D.Double(polygon.getX()
                        - (attributeWidth / 2) - (dx * i),
                        polygon.getY() - (attributeHeight / 2), attributeWidth,
                        attributeHeight);

                if (isTyomePutable(polygon, point, usedPoints,
                            attributeRectangle, usedArea, visibleRectangle,
                            contains)) {
                    polygon.setAttributeLocation(attributeRectangle.getX(),
                        attributeRectangle.getMaxY());
                    polygon.setTyomeFont(font);
                    usedArea.add(new Area(pointRectangle));
                    usedArea.add(new Area(attributeRectangle));
                    usedPoints.add(point);
                }
            }
        }
    }

    /**
     * 点の属性の表示位置を決定します。
     * @param point 点
     * @param usedArea 属性によって使用済みの領域
     * @param visibleRectangle 現在表示されている領域
     * @param zoom 倍率
     * @param attributeHeight 属性の高さ
     * @param metrics フォントメトリクス
     * @param size 点の直径
     */
    private void fixPointAttributeLocation(PointData point, Area usedArea,
        Collection usedPoints, Rectangle2D visibleRectangle, double zoom,
        double attributeHeight, FontMetrics metrics, double size) {
        if (point.getAttribute() != null) {
            point.setAttributeLocation(0, 0);

            double attributeWidth = metrics.stringWidth(point.getAttribute()) / zoom;
            Rectangle2D pointRectangle = new Rectangle2D.Double(point.getX()
                    - (size / zoom), point.getY() - (size / zoom),
                    size / 2 / zoom, size / 2 / zoom);
            Rectangle2D attributeRectangle;

            // 点の右
            attributeRectangle = new Rectangle2D.Double(point.getX()
                    + (size / zoom), point.getY() - (attributeHeight / 2),
                    attributeWidth, attributeHeight);

            Point location = new Point((int) point.getX(), (int) point.getY());

            if (isPointPutable(attributeRectangle, location, usedPoints,
                        usedArea, visibleRectangle)) {
                point.setAttributeLocation(attributeRectangle.getX(),
                    attributeRectangle.getMaxY());
                usedArea.add(new Area(attributeRectangle));
                usedPoints.add(location);
            }

            // 点の左
            attributeRectangle = new Rectangle2D.Double(point.getX()
                    - attributeWidth - (size / zoom),
                    point.getY() - (attributeHeight / 2), attributeWidth,
                    attributeHeight);

            if (isPointPutable(attributeRectangle, location, usedPoints,
                        usedArea, visibleRectangle)) {
                point.setAttributeLocation(attributeRectangle.getX(),
                    attributeRectangle.getMaxY());
                usedArea.add(new Area(attributeRectangle));
                usedPoints.add(location);
            }

            // 点の右上
            attributeRectangle = new Rectangle2D.Double(point.getX()
                    + (size / zoom), point.getY() - attributeHeight,
                    attributeWidth, attributeHeight);

            if (isPointPutable(attributeRectangle, location, usedPoints,
                        usedArea, visibleRectangle)) {
                point.setAttributeLocation(attributeRectangle.getX(),
                    attributeRectangle.getMaxY());
                usedArea.add(new Area(attributeRectangle));
                usedPoints.add(location);
            }

            // 点の右下
            attributeRectangle = new Rectangle2D.Double(point.getX()
                    + (size / zoom), point.getY(), attributeWidth,
                    attributeHeight);

            if (isPointPutable(attributeRectangle, location, usedPoints,
                        usedArea, visibleRectangle)) {
                point.setAttributeLocation(attributeRectangle.getX(),
                    attributeRectangle.getMaxY());
                usedArea.add(new Area(attributeRectangle));
                usedPoints.add(location);
            }

            // 点の左上
            attributeRectangle = new Rectangle2D.Double(point.getX()
                    - attributeWidth - (size / zoom),
                    point.getY() - attributeHeight, attributeWidth,
                    attributeHeight);

            if (isPointPutable(attributeRectangle, location, usedPoints,
                        usedArea, visibleRectangle)) {
                point.setAttributeLocation(attributeRectangle.getX(),
                    attributeRectangle.getMaxY());
                usedArea.add(new Area(attributeRectangle));
                usedPoints.add(location);
            }

            // 点の左下
            attributeRectangle = new Rectangle2D.Double(point.getX()
                    - attributeWidth - (size / zoom), point.getY(),
                    attributeWidth, attributeHeight);

            if (isPointPutable(attributeRectangle, location, usedPoints,
                        usedArea, visibleRectangle)) {
                point.setAttributeLocation(attributeRectangle.getX(),
                    attributeRectangle.getMaxY());
                usedArea.add(new Area(attributeRectangle));
                usedPoints.add(location);
            }
        }
    }

    /**
     * 建物の属性の表示位置を決定します。
     * @param polygon ポリゴン
     * @param usedArea 属性によって使用済みの領域
     * @param visibleRectangle 現在表示されている領域
     * @param zoom 倍率
     * @param attributeHeight 属性の高さ
     * @param metrics フォントメトリクス
     * @param size 点の直径
     */
    private void fixTatemonoAttributeLocation(PolygonData polygon,
        Area usedArea, Collection usedPoints, Rectangle2D visibleRectangle,
        double zoom, double attributeHeight, FontMetrics metrics, double size) {
        if ((polygon.getAttribute() != null)
                && (polygon.getTatemonoCode() != PolygonData.TATEMONO_STATION)) {
            polygon.setAttributeLocation(0, 0);

            double attributeWidth = metrics.stringWidth(polygon.getAttribute()) / zoom;
            Rectangle2D pointRectangle = new Rectangle2D.Double(polygon.getX()
                    - (size / zoom), polygon.getY() - (size / zoom),
                    (size * 2) / zoom, (size * 2) / zoom);
            Rectangle2D attributeRectangle;

            // 点の右
            attributeRectangle = new Rectangle2D.Double(polygon.getX()
                    + (size / zoom), polygon.getY() - (attributeHeight / 2),
                    attributeWidth, attributeHeight);

            Point point = new Point((int) polygon.getX(), (int) polygon.getY());

            if (isPutable(point, usedPoints, attributeRectangle, usedArea,
                        visibleRectangle)) {
                polygon.setAttributeLocation(attributeRectangle.getX(),
                    attributeRectangle.getMaxY());
                usedArea.add(new Area(pointRectangle));
                usedArea.add(new Area(attributeRectangle));
                usedPoints.add(point);
            }

            // 点の左
            attributeRectangle = new Rectangle2D.Double(polygon.getX()
                    - attributeWidth - (size / zoom),
                    polygon.getY() - (attributeHeight / 2), attributeWidth,
                    attributeHeight);

            if (isPutable(point, usedPoints, attributeRectangle, usedArea,
                        visibleRectangle)) {
                polygon.setAttributeLocation(attributeRectangle.getX(),
                    attributeRectangle.getMaxY());
                usedArea.add(new Area(pointRectangle));
                usedArea.add(new Area(attributeRectangle));
                usedPoints.add(point);
            }

            // 点の右上
            attributeRectangle = new Rectangle2D.Double(polygon.getX()
                    + (size / 2 / zoom), polygon.getY() - attributeHeight,
                    attributeWidth, attributeHeight);

            if (isPutable(point, usedPoints, attributeRectangle, usedArea,
                        visibleRectangle)) {
                polygon.setAttributeLocation(attributeRectangle.getX(),
                    attributeRectangle.getMaxY());
                usedArea.add(new Area(pointRectangle));
                usedArea.add(new Area(attributeRectangle));
                usedPoints.add(point);
            }

            // 点の右下
            attributeRectangle = new Rectangle2D.Double(polygon.getX()
                    + (size / 2 / zoom), polygon.getY(), attributeWidth,
                    attributeHeight);

            if (isPutable(point, usedPoints, attributeRectangle, usedArea,
                        visibleRectangle)) {
                polygon.setAttributeLocation(attributeRectangle.getX(),
                    attributeRectangle.getMaxY());
                usedArea.add(new Area(pointRectangle));
                usedArea.add(new Area(attributeRectangle));
                usedPoints.add(point);
            }

            // 点の左上
            attributeRectangle = new Rectangle2D.Double(polygon.getX()
                    - attributeWidth - (size / 2 / zoom),
                    polygon.getY() - attributeHeight, attributeWidth,
                    attributeHeight);

            if (isPutable(point, usedPoints, attributeRectangle, usedArea,
                        visibleRectangle)) {
                polygon.setAttributeLocation(attributeRectangle.getX(),
                    attributeRectangle.getMaxY());
                usedArea.add(new Area(pointRectangle));
                usedArea.add(new Area(attributeRectangle));
                usedPoints.add(point);
            }

            // 点の左下
            attributeRectangle = new Rectangle2D.Double(polygon.getX()
                    - attributeWidth - (size / 2 / zoom), polygon.getY(),
                    attributeWidth, attributeHeight);

            if (isPutable(point, usedPoints, attributeRectangle, usedArea,
                        visibleRectangle)) {
                polygon.setAttributeLocation(attributeRectangle.getX(),
                    attributeRectangle.getMaxY());
                usedArea.add(new Area(pointRectangle));
                usedArea.add(new Area(attributeRectangle));
                usedPoints.add(point);
            }
        }
    }

    private void fixPolygonAttributeLocation(PolygonData polygon,
        Area usedArea, Collection usedPoints, Rectangle2D visibleRectangle,
        double zoom, double attributeHeight, FontMetrics metrics, double size) {
        if (polygon.getAttribute() != null) {
            polygon.setAttributeLocation(0, 0);

            double attributeWidth = metrics.stringWidth(polygon.getAttribute()) / zoom;
            Rectangle2D pointRectangle = new Rectangle2D.Double(polygon.getX()
                    - (size / zoom), polygon.getY() - (size / zoom),
                    (size * 2) / zoom, (size * 2) / zoom);
            Rectangle2D attributeRectangle;

            // ポリゴンの中心
            attributeRectangle = new Rectangle2D.Double(polygon.getX()
                    - (attributeWidth / 2),
                    polygon.getY() - (attributeHeight / 2), attributeWidth,
                    attributeHeight);

            Point point = new Point((int) polygon.getX(), (int) polygon.getY());

            if (isPutable(point, usedPoints, attributeRectangle, usedArea,
                        visibleRectangle)) {
                polygon.setAttributeLocation(attributeRectangle.getX(),
                    attributeRectangle.getMaxY());
                usedArea.add(new Area(pointRectangle));
                usedArea.add(new Area(attributeRectangle));
                usedPoints.add(point);
            }
        }
    }

    private boolean isPutable(Point point, Collection usedPoints,
        Rectangle2D attributeRectangle, Area usedArea,
        Rectangle2D visibleRectangle) {
        return !usedPoints.contains(point) && visibleRectangle.contains(point)
        && visibleRectangle.contains(attributeRectangle)
        && !usedArea.contains(point)
        && !usedArea.intersects(attributeRectangle);
    }

    private boolean isPointPutable(Rectangle2D attributeRectangle, Point point,
        Collection usedPoints, Area usedArea, Rectangle2D visibleRectangle) {
        return !usedPoints.contains(point)
        && visibleRectangle.contains(attributeRectangle)
        && !usedArea.intersects(attributeRectangle);
    }

    private boolean isTyomePutable(PolygonData polygon, Point point,
        Collection usedPoints, Rectangle2D attributeRectangle, Area usedArea,
        Rectangle2D visibleRectangle, boolean contains) {
        if (contains) {
            return !usedPoints.contains(point)
            && visibleRectangle.contains(attributeRectangle)
            && polygon.getArea().contains(attributeRectangle)
            && !usedArea.intersects(attributeRectangle);
        } else {
            return !usedPoints.contains(point)
            && visibleRectangle.contains(attributeRectangle)
            && polygon.getArea().intersects(attributeRectangle)
            && !usedArea.intersects(attributeRectangle);
        }
    }

    /**
     * 直前の計算で属性を描画する座標が変化したかどうかを取得します。
     * @return 座標が変化したかどうか
     */
    boolean isChanged() {
        return isChanged;
    }
}
