import java.awt.event.ComponentAdapter;
import java.awt.event.ComponentEvent;
import java.awt.event.MouseAdapter;
import java.awt.event.MouseEvent;
import java.awt.event.MouseWheelEvent;
import java.awt.event.MouseWheelListener;
import java.util.Map;

/**
 * 地図の読み込み、ポリゴンの結合、塗り分け、属性配置位置の計算、再描画など、
 * バックグラウンドで行う作業を管理するクラスです。
 * @author Kumano Tatsuo
 * 作成日: 2004/01/19
 */
class BackgroundThread extends Thread {
    /**
     * 地図
     */
    private Map<String, MapData> maps; // 地図

    /**
     * パネル
     */
    private MapPanel panel; // パネル

    /**
     * 地図の状態が変化したかどうか
     */
    boolean isChanged; // 地図の状態が変化したかどうか

    /**
     * バックグラウンドで行う処理を初期化します。
     * @param maps 地図
     * @param panel パネル
     */
    BackgroundThread(Map<String, MapData> maps, MapPanel panel) {
        this.maps = maps;
        this.panel = panel;
        this.isChanged = true;
        panel.addMouseListener(new MouseAdapter() {
            public void mouseReleased(MouseEvent e) {
                BackgroundThread.this.isChanged = true;
            }
        });
        panel.addMouseWheelListener(new MouseWheelListener() {
            public void mouseWheelMoved(MouseWheelEvent e) {
                BackgroundThread.this.isChanged = true;
            }
        });
        panel.addComponentListener(new ComponentAdapter() {
            public void componentResized(ComponentEvent e) {
                BackgroundThread.this.isChanged = true;
            }
        });
    }

    /**
     * スレッドが開始されるときに呼び出されます。
     */
    public void run() {
        try {
            while (true) {
                if (this.panel.isIdle() && this.isChanged) {
                    System.out.println("計算中です。");
                    this.panel.setIsBusy(true);
                    loadMapPaintTyomeJoinTyome();
                    System.out.println("計算が終了しました。");
                    this.panel.setIsBusy(false);
                    this.isChanged = false;
                }
                Thread.sleep(200);
            }
        } catch (InterruptedException e) {
            e.printStackTrace(System.err);
        }
    }

    /**
     * 地図の読み込み、ポリゴンの結合、塗り分け、属性配置位置の計算を行います。
     */
    void loadMapPaintTyomeJoinTyome() {
        try {
            long startTime = System.currentTimeMillis();
            LoadMap loadMap = new LoadMap();
            loadMap.loadMap(this.maps, (MapPanel) this.panel);
            if ((System.currentTimeMillis() - startTime) > 200) {
                System.out.println("地図の読み込み：" + (System.currentTimeMillis() - startTime) + " ms");
            }
            startTime = System.currentTimeMillis();
            PaintTyome paintTyome = new PaintTyome();
            paintTyome.paintTyome(this.maps);
            if ((System.currentTimeMillis() - startTime) > 200) {
                System.out.println("塗り分け：" + (System.currentTimeMillis() - startTime) + " ms");
            }
            startTime = System.currentTimeMillis();
            JoinPolygon joinTyome = new JoinPolygon();
            joinTyome.joinPolygon(this.maps);
            JoinTatemono joinTatemono = new JoinTatemono();
            joinTatemono.joinTatemono(this.maps);
            if ((System.currentTimeMillis() - startTime) > 200) {
                System.out.println("ポリゴンの結合：" + (System.currentTimeMillis() - startTime) + " ms");
            }
            startTime = System.currentTimeMillis();
            FixAttributeLocation fixAttributeLocation = new FixAttributeLocation();
            fixAttributeLocation.fixAttributeLocation(this.maps, (MapPanel) this.panel);
            if ((System.currentTimeMillis() - startTime) > 200) {
                System.out.println("属性描画位置の計算：" + (System.currentTimeMillis() - startTime) + " ms");
            }
            this.panel.setNeedsRepaint(true);
            this.panel.repaint();
        } catch (Exception exception) {
            System.err.println("Failed to load map.");
            exception.printStackTrace(System.err);
        }
    }
}
