package map;

import java.awt.BasicStroke;
import java.awt.Color;
import java.awt.Dimension;
import java.awt.Font;
import java.awt.FontMetrics;
import java.awt.Graphics;
import java.awt.Graphics2D;
import java.awt.Image;
import java.awt.Rectangle;
import java.awt.RenderingHints;
import java.awt.Shape;
import java.awt.event.ActionEvent;
import java.awt.event.ActionListener;
import java.awt.event.ComponentAdapter;
import java.awt.event.ComponentEvent;
import java.awt.event.InputEvent;
import java.awt.event.MouseAdapter;
import java.awt.event.MouseEvent;
import java.awt.event.MouseMotionListener;
import java.awt.event.MouseWheelEvent;
import java.awt.event.MouseWheelListener;
import java.awt.geom.AffineTransform;
import java.awt.geom.Ellipse2D;
import java.awt.geom.Point2D;
import java.awt.geom.Rectangle2D;
import java.awt.image.BufferedImage;
import java.awt.print.PageFormat;
import java.awt.print.Printable;
import java.awt.print.PrinterException;
import java.io.File;
import java.io.FileNotFoundException;
import java.io.IOException;
import java.io.UnsupportedEncodingException;
import java.util.Collection;
import java.util.Map;
import java.util.Stack;

import javax.imageio.ImageIO;
import javax.print.PrintException;
import javax.swing.JLabel;
import javax.swing.JPanel;

import map.Const.Zoom;

import org.apache.batik.svggen.SVGGraphics2DIOException;

import psout.PSOut;
import search.Search;
import svgout.Paintable;
import svgout.SVGOut;

/**
 * 地図をパネルに描画するクラスです。
 * @author Kumano Tatsuo
 */
public class MapPanel extends JPanel implements Printable {
	/**
	 * アンチエイリアス表示するかどうか
	 */
	boolean isAntialias;

	/**
	 * 文字の大きさ
	 */
	private double fontZoom;

	/**
	 * 裏描画用のイメージ
	 */
	private Image image;

	/**
	 * 地図が変化したかどうか
	 */
	boolean isChanged;

	/**
	 * 直前のマウスカーソル座標
	 */
	double lastMouseX; // 直前のマウスカーソル座標

	/**
	 * 直前のマウスカーソル座標
	 */
	double lastMouseY; // 直前のマウスカーソル座標

	/**
	 * 背景スレッドに再計算を要求するためのアクションリスナ
	 */
	private ActionListener listener;

	/**
	 * 地図の設定
	 */
	MapPreferences mapPreferences; // 地図の設定

	/**
	 * 地図
	 */
	private final Map<String, MapData> maps;

	/**
	 * x座標の最大値
	 */
	private double maxX;

	/**
	 * y座標の最大値
	 */
	private double maxY;

	/**
	 * x座標の最小値
	 */
	private double minX;

	/**
	 * y座標の最小値
	 */
	private double minY;

	/**
	 * 表示倍率の最小値
	 */
	double minZoom;

	/**
	 * オフセット（実座標）
	 */
	double offsetX; // オフセット(実座標)

	/**
	 * オフセット（実座標）
	 */
	double offsetY; // オフセット(実座標)

	/**
	 * 都道府県の一覧
	 */
	private final Collection<Prefecture> prefectures;

	/**
	 * 彩度の差分
	 */
	private float saturationDifference;

	/**
	 * 地図を検索するためのデータ構造
	 */
	private Search search;

	/**
	 * パネルの幅と高さ
	 */
	private Dimension size;

	/**
	 * 表示倍率
	 */
	double zoom; // 表示倍率

	/**
	 * ステータスバーに表示するメッセージ
	 */
	private final Stack<String> messages;

	/**
	 * ステータスバー
	 */
	private JLabel statusBar;

	/**
	 * 画面中央の都道府県市区町村名
	 * @since 4.06
	 */
	private String centerPrefectureCity;

	/**
	 * 画面中央の町丁目名
	 * @since 4.06
	 */
	private String centerTyome;

	/**
	 * @param statusBar ステータスバー
	 */
	public void setStatusBar(final JLabel statusBar) {
		this.statusBar = statusBar;
	}

	/**
	 * ステータスバーに表示するメッセージをスタックに積みます。
	 * @param message メッセージ
	 */
	public void addMessage(final String message) {
		this.messages.push(message);
		this.statusBar.setText(this.getMessage());
	}

	/**
	 * @return メッセージ
	 */
	public String getMessage() {
		if (this.messages.size() > 0) {
			return this.messages.peek();
		} else {
			return this.centerPrefectureCity + this.centerTyome + " ";
		}
	}

	/**
	 * ステータスバーに表示されるメッセージをスタックから取り出して捨てます。
	 */
	public void removeMessage() {
		if (this.messages.size() > 0) {
			this.messages.pop();
		}
		this.statusBar.setText(this.getMessage());
	}

	/**
	 * 地図を表示するパネルを初期化します。
	 * @param maps 地図
	 */
	public MapPanel(final Map<String, MapData> maps) {
		this.mapPreferences = new DefaultMapPreferences();
		this.setBackground(this.mapPreferences.getBackGroundColor());
		this.offsetX = 0;
		this.offsetY = 0;
		this.zoom = 1;
		this.isChanged = true;
		this.fontZoom = 1;
		this.saturationDifference = 0;
		this.lastMouseX = this.offsetX;
		this.lastMouseY = this.offsetY;
		this.maps = maps;
		this.isAntialias = true;
		this.prefectures = Prefectures.loadPrefectures(this.mapPreferences, this);
		this.messages = new Stack<String>();
		this.centerPrefectureCity = "";
		this.centerTyome = "";
		this.addMouseListener(new MouseAdapter() {
			@Override
			public void mousePressed(final MouseEvent e) {
				MapPanel.this.lastMouseX = e.getX();
				MapPanel.this.lastMouseY = e.getY();
				MapPanel.this.isAntialias = false;
			}

			@Override
			public void mouseReleased(final MouseEvent e) {
				MapPanel.this.isAntialias = true;
			}

			@Override
			/**
			 * ダブルクリックで移動、拡大、縮小をします。
			 * @since 4.05
			 */
			public void mouseClicked(final MouseEvent e) {
				if (e.getClickCount() > 1) {
					if (e.getButton() == MouseEvent.BUTTON1) {
						final Point2D point = MapPanel.this.toVirtualLocation(new Point2D.Double(e.getX(), e.getY()));
						MapPanel.this.moveTo(point.getX(), point.getY());
						if (MapPanel.this.getZoom() < Const.Zoom.LOAD_CITIES) {
							MapPanel.this.zoomCities();
						} else if (MapPanel.this.getZoom() < Const.Zoom.LOAD_FINE_CITIES) {
							MapPanel.this.zoomFineCities();
						} else if (MapPanel.this.getZoom() < Const.Zoom.LOAD_2500) {
							MapPanel.this.zoomWide();
						} else if (MapPanel.this.getZoom() < Const.Zoom.LOAD_GYOUSEI) {
							MapPanel.this.zoomMiddle();
						} else if (MapPanel.this.getZoom() < Const.Zoom.LOAD_ALL) {
							MapPanel.this.zoomDetail();
						}
					} else {
						if (MapPanel.this.getZoom() > Const.Zoom.LOAD_ALL) {
							MapPanel.this.zoomDetail();
						} else if (MapPanel.this.getZoom() > Const.Zoom.LOAD_GYOUSEI) {
							MapPanel.this.zoomMiddle();
						} else if (MapPanel.this.getZoom() > Const.Zoom.LOAD_2500) {
							MapPanel.this.zoomWide();
						} else if (MapPanel.this.getZoom() > Const.Zoom.LOAD_FINE_CITIES) {
							MapPanel.this.zoomFineCities();
						} else if (MapPanel.this.getZoom() > Const.Zoom.LOAD_CITIES) {
							MapPanel.this.zoomCities();
						} else {
							MapPanel.this.zoomWhole();
						}
					}
				}
			}

		});
		this.addMouseMotionListener(new MouseMotionListener() {
			public void mouseDragged(final MouseEvent e) {
				MapPanel.this.offsetX -= (e.getX() - MapPanel.this.lastMouseX);
				MapPanel.this.offsetY -= (e.getY() - MapPanel.this.lastMouseY);
				MapPanel.this.lastMouseX = e.getX();
				MapPanel.this.lastMouseY = e.getY();
				MapPanel.this.isChanged = true;
				MapPanel.this.repaint();
			}

			public void mouseMoved(final MouseEvent e) {
				if (e.getModifiersEx() == (InputEvent.SHIFT_DOWN_MASK | InputEvent.CTRL_DOWN_MASK | InputEvent.ALT_DOWN_MASK)) {
					final Point2D point = MapPanel.this.toVirtualLocation(new Point2D.Double(e.getX(), e.getY()));
					System.out.println("DEBUG: x = " + point.getX() + ", y = " + point.getY());
				}
			}
		});
		this.addMouseWheelListener(new MouseWheelListener() {
			public void mouseWheelMoved(final MouseWheelEvent e) {
				final int wheelRotation = e.getWheelRotation();
				final int centerX = e.getX();
				final int centerY = e.getY();
				MapPanel.this.doWheelRotation(wheelRotation, centerX, centerY);
				MapPanel.this.isChanged = true;
				MapPanel.this.repaint();
			}
		});
		this.addComponentListener(new ComponentAdapter() {
			@Override
			public void componentResized(final ComponentEvent e) {
				MapPanel.this.isChanged = true;
			}
		});
	}

	/**
	 * 座標の最大値と最小値を計算します。
	 */
	public void calcMinMaxXY() {
		this.minX = Double.POSITIVE_INFINITY;
		this.minY = Double.POSITIVE_INFINITY;
		this.maxX = Double.NEGATIVE_INFINITY;
		this.maxY = Double.NEGATIVE_INFINITY;
		if (this.maps != null) {
			if (this.maps.isEmpty()) {
				for (final Prefecture prefecture : this.prefectures) {
					final Rectangle2D boudns = prefecture.getBounds();
					this.minX = Math.min(this.minX, boudns.getMinX());
					this.minY = Math.min(this.minY, boudns.getMinY());
					this.maxX = Math.max(this.maxX, boudns.getMaxX());
					this.maxY = Math.max(this.maxY, boudns.getMaxY());
				}
			} else {
				for (final MapData mapData : this.maps.values()) {
					final Rectangle bounds = mapData.getBounds().getBounds();
					if (bounds.getMinX() < this.minX) {
						this.minX = bounds.getMinX();
					}
					if (bounds.getMinY() < this.minY) {
						this.minY = bounds.getMinY();
					}
					if (this.maxX < bounds.getMaxX()) {
						this.maxX = bounds.getMaxX();
					}
					if (this.maxY < bounds.getMaxY()) {
						this.maxY = bounds.getMaxY();
					}
				}
			}
		}
	}

	/**
	 * 文字を小さくします。
	 */
	public void decreaseFontSize() {
		this.fontZoom *= 0.9;
		this.mapPreferences.setFontZoom(this.fontZoom);
		if (this.listener != null) {
			this.listener.actionPerformed(new ActionEvent(this, this.hashCode(), "decrement font size"));
		}
		this.isChanged = true;
		this.repaint();
	}

	/**
	 * 彩度を減らします。
	 */
	public void decreaseSaturation() {
		this.saturationDifference -= 0.05;
		this.isChanged = true;
		this.repaint();
	}

	/**
	 * マウスホイールの回転をシミュレートします。
	 * @param wheelRotation マウスホイールの回転方向、1で手前、-1で向こう
	 * @param centerX マウスポインタのx座標（実座標）
	 * @param centerY マウスポインタのy座標（実座標）
	 */
	void doWheelRotation(final int wheelRotation, final int centerX, final int centerY) {
		final double newZoom = Math.min(Zoom.MAX_VALUE, Math.max(this.minZoom, wheelRotation > 0 ? this.zoom
				* Zoom.RATE : this.zoom / Zoom.RATE));
		this.zoom(newZoom, centerX, centerY);
	}

	/**
	 * 地図を描画します。
	 * SVGGraphics2D#transform(AffineTransform)の誤差が大きくて使いものにならないので、AffineTransform#createTransformedShape(Shape)を使うようにしました。
	 * @param g グラフィクスコンテキスト
	 * @param isTransform 描画対象全体を座標変換するかどうか。
	 * @throws IOException 
	 */
	public void drawMap(final Graphics2D g, final boolean isTransform) throws IOException {
		this.addMessage("地図を描画しています。");
		final FixAttributeLocation fixAttributeLocation = new FixAttributeLocation();
		synchronized (this.maps) {
			fixAttributeLocation.fixAttributeLocation(this.maps, this.prefectures, this);
		}
		final int width = (this.size == null) ? this.getWidth() : this.size.width;
		final int height = (this.size == null) ? this.getHeight() : this.size.height;
		g.setColor(this.mapPreferences.getMizuPreferences().getFillColor());
		g.fillRect(0, 0, width, height);
		final AffineTransform transform = new AffineTransform();
		transform.translate(-this.offsetX, -this.offsetY);
		transform.scale(this.zoom, this.zoom);
		if (isTransform) {
			g.transform(transform);
		}
		final double x = this.offsetX / this.zoom;
		final double y = this.offsetY / this.zoom;
		final double w = width / this.zoom;
		final double h = height / this.zoom;
		final Point2D center = this.toVirtualLocation(new Point2D.Double(width / 2, height / 2));
		this.centerPrefectureCity = "";
		this.centerTyome = "";
		try {
			// 都道府県を描画する
			for (final Prefecture prefecture : this.prefectures) {
				this.setFixedStroke(g, this.mapPreferences.getPrefecturePreferences().getWidth(), isTransform);
				if (prefecture.getBounds().intersects(x, y, w, h)) {
					final Shape shape = prefecture.hasFine() ? prefecture.getFineShape() : prefecture.getShape();
					if (shape.intersects(x, y, w, h)) {
						final Color color = prefecture.getColor();
						if (this.saturationDifference == 0) {
							g.setColor(color);
						} else {
							final float[] hsb = Color.RGBtoHSB(color.getRed(), color.getGreen(), color.getBlue(),
									new float[] { 0, 0, 0 });
							hsb[1] = Math.min(1, Math.max(0, hsb[1] + this.saturationDifference));
							g.setColor(Color.getHSBColor(hsb[0], hsb[1], hsb[2]));
						}
						this.fill(g, shape, isTransform, transform);
						g.setColor(Color.BLACK);
						this.draw(g, shape, isTransform, transform);
					}
				}
			}
			if (this.zoom >= Const.Zoom.LOAD_CITIES) {
				// 市区町村を描画する
				for (final Prefecture prefecture : this.prefectures) {
					if (prefecture.hasCities()) {
						this.setFixedStroke(g, this.mapPreferences.getCityPreferences().getWidth(), isTransform);
						for (final City city : prefecture.getCities()) {
							final Shape shape = city.hasFineShape() ? city.getFineShape() : city.getShape();
							if (shape.intersects(x, y, w, h)) {
								if (city.getURL() == null) {
									g.setColor(Color.LIGHT_GRAY);
									this.fill(g, shape, isTransform, transform);
								}
								g.setColor(Color.BLACK);
								this.draw(g, shape, isTransform, transform);
								if (city.hasFineShape() && shape.contains(center)) {
									this.centerPrefectureCity = prefecture.getLabel() + city.getLabel();
								}
							}
						}
						this.setFixedStroke(g, this.mapPreferences.getPrefecturePreferences().getWidth(), isTransform);
						this.draw(g, prefecture.hasFine() ? prefecture.getFineShape() : prefecture.getShape(),
								isTransform, transform);
					}
				}
			}
			synchronized (this.maps) {
				if (this.maps != null & this.zoom >= Const.Zoom.LOAD_2500) {
					g.setStroke(new BasicStroke(1f));
					for (final MapData mapData : this.maps.values()) {
						if (mapData.getBounds().intersects(x, y, w, h)) {
							// 海を描画する
							if (mapData.hasTyome()) {
								g.setColor(this.mapPreferences.getMizuPreferences().getFillColor());
								this.fill(g, mapData.getBounds(), isTransform, transform);
								this.draw(g, mapData.getBounds(), isTransform, transform);
							}
						}
					}
					for (final MapData mapData : this.maps.values()) {
						if (mapData.getBounds().intersects(x, y, w, h)) {
							// 丁目を描画する
							if (mapData.hasTyome()) {
								for (final PolygonData polygon : mapData.getTyome().values()) {
									if (polygon.getArea() != null) {
										if (polygon.getClassificationCode() == PolygonData.CLASSIFICATION_TYOME) {
											final Color color = this.mapPreferences.getTyomeFillColor(polygon
													.getTyomeColorIndex());
											if (this.saturationDifference == 0) {
												g.setColor(color);
											} else {
												final float[] hsb = Color.RGBtoHSB(color.getRed(), color.getGreen(),
														color.getBlue(), new float[] { 0, 0, 0 });
												hsb[1] = Math.min(1, Math.max(0, hsb[1] + this.saturationDifference));
												g.setColor(Color.getHSBColor(hsb[0], hsb[1], hsb[2]));
											}
											this.fill(g, polygon.getArea(), isTransform, transform);
											// since 4.06
											if (polygon.getArea().contains(center)) {
												this.centerTyome = polygon.getAttribute() + "付近";
											}
										}
									}
								}
							}
						}
					}
					for (final MapData mapData : this.maps.values()) {
						if (mapData.getBounds().intersects(x, y, w, h)) {
							// 場地を描画する
							if (mapData.hasZyouti()) {
								for (final PolygonData polygon : mapData.getZyouti().values()) {
									if (polygon.getClassificationCode() == PolygonData.CLASSIFICATION_RAILROAD) {
										g.setColor(this.mapPreferences.getZyoutiPreferences().getFillColor());
										this.fill(g, polygon.getArea(), isTransform, transform);
									} else if (polygon.getClassificationCode() == PolygonData.CLASSIFICATION_PARK) {
										g.setColor(this.mapPreferences.getParkPreferences().getFillColor());
										this.fill(g, polygon.getArea(), isTransform, transform);
									} else if (polygon.getClassificationCode() == PolygonData.CLASSIFICATION_SCHOOL) {
										g.setColor(this.mapPreferences.getZyoutiPreferences().getFillColor());
										this.fill(g, polygon.getArea(), isTransform, transform);
									} else if (polygon.getClassificationCode() == PolygonData.CLASSIFICATION_TEMPLE) {
										g.setColor(this.mapPreferences.getZyoutiPreferences().getFillColor());
										this.fill(g, polygon.getArea(), isTransform, transform);
									} else if (polygon.getClassificationCode() == PolygonData.CLASSIFICATION_GRAVEYARD) {
										g.setColor(this.mapPreferences.getZyoutiPreferences().getFillColor());
										this.fill(g, polygon.getArea(), isTransform, transform);
									} else if (polygon.getClassificationCode() == PolygonData.CLASSIFICATION_OTHER) {
										g.setColor(this.mapPreferences.getZyoutiPreferences().getFillColor());
										this.fill(g, polygon.getArea(), isTransform, transform);
									} else {
										System.out.println(this.getClass().getName() + ": unknown classification code "
												+ polygon.getClassificationCode());
									}
								}
							}
							// 内水面を描画する
							if (mapData.hasMizu()) {
								for (final PolygonData polygon : mapData.getMizu().values()) {
									if (polygon.getArea() != null) {
										g.setColor(this.mapPreferences.getMizuPreferences().getFillColor());
										this.fill(g, polygon.getArea(), isTransform, transform);
										this.draw(g, polygon.getArea(), isTransform, transform);
									}
								}
							}
						}
					}
					for (final MapData mapData : this.maps.values()) {
						if (mapData.getBounds().intersects(x, y, w, h)) {
							// 建物を描画する
							if (mapData.hasTatemono()) {
								for (final PolygonData polygon : mapData.getTatemono().values()) {
									if (polygon.getArea() != null) {
										g.setColor(this.mapPreferences.getTatemonoPreferences().getFillColor());
										this.fill(g, polygon.getArea(), isTransform, transform);
									}
								}
							}
						}
					}
					this.setVariableStroke(g, this.mapPreferences.getZyoutiPreferences().getWidth(), isTransform);
					for (final MapData mapData : this.maps.values()) {
						if (mapData.getBounds().intersects(x, y, w, h)) {
							// 場地界を描画する
							if (mapData.hasZyouti()) {
								for (final ArcData arc : mapData.getOthers().values()) {
									if (arc.getClassification() != ArcData.TYPE_RAILWAY) {
										if (arc.getTag() == ArcData.TAG_NORMAL) {
											g.setColor(this.mapPreferences.getZyoutiPreferences().getBorderColor());
											this.draw(g, arc.getPath(), isTransform, transform);
										}
									}
								}
							}
							// 内水面界を描画する
							if (mapData.hasMizuArc()) {
								for (final ArcData arc : mapData.getMizuArc().values()) {
									if (arc.getTag() == ArcData.TAG_NORMAL) {
										if (arc.getClassification() == ArcData.TYPE_MIZU_INSIDE) {
											g.setColor(this.mapPreferences.getMizuPreferences().getBorderColor());
											this.draw(g, arc.getPath(), isTransform, transform);
										}
									}
								}
							}
							// 建物界を描画する
							if (mapData.hasTatemonoArc()) {
								for (final ArcData arc : mapData.getTatemonoArc().values()) {
									if (arc.getTag() == ArcData.TAG_NORMAL) {
										g.setColor(this.mapPreferences.getTatemonoPreferences().getBorderColor());
										this.draw(g, arc.getPath(), isTransform, transform);
									}
								}
							}
						}
					}
					for (final MapData mapData : this.maps.values()) {
						if (mapData.getBounds().intersects(x, y, w, h)) {
							// 道路の輪郭を描画する
							if (mapData.hasRoadArc() && mapData.hasTyome()) {
								for (final ArcData arc : mapData.getRoadArc().values()) {
									if (arc.getRoadType() == ArcData.ROAD_HIGHWAY) {
										this.setVariableFatStroke(g, this.mapPreferences.getHighwayPreferences()
												.getWidth(), isTransform);
										g.setColor(this.mapPreferences.getHighwayPreferences().getBorderColor());
										this.draw(g, arc.getPath(), isTransform, transform);
									} else if (arc.getRoadType() == ArcData.ROAD_KOKUDO) {
										this.setVariableFatStroke(g, this.mapPreferences.getKokudoPreferences()
												.getWidth(), isTransform);
										g.setColor(this.mapPreferences.getKokudoPreferences().getBorderColor());
										this.draw(g, arc.getPath(), isTransform, transform);
									} else if (arc.getRoadType() == ArcData.ROAD_KENDO) {
										this.setVariableFatStroke(g, this.mapPreferences.getKendoPreferences()
												.getWidth(), isTransform);
										g.setColor(this.mapPreferences.getKendoPreferences().getBorderColor());
										this.draw(g, arc.getPath(), isTransform, transform);
									} else if (arc.getRoadType() == ArcData.ROAD_CHIHODO) {
										this.setVariableFatStroke(g, this.mapPreferences.getChihodoPreferences()
												.getWidth(), isTransform);
										g.setColor(this.mapPreferences.getChihodoPreferences().getBorderColor());
										this.draw(g, arc.getPath(), isTransform, transform);
									} else if (arc.getRoadType() == ArcData.ROAD_MAJOR) {
										this.setVariableFatStroke(g, this.mapPreferences.getMajorRoadPreferences()
												.getWidth(), isTransform);
										g.setColor(this.mapPreferences.getMajorRoadPreferences().getBorderColor());
										this.draw(g, arc.getPath(), isTransform, transform);
									} else {
										this.setVariableFatStroke(g, this.mapPreferences.getNormalRoadPreferences()
												.getWidth(), isTransform);
										g.setColor(this.mapPreferences.getNormalRoadPreferences().getBorderColor());
										this.draw(g, arc.getPath(), isTransform, transform);
									}
								}
							}
						}
					}
					for (final MapData mapData : this.maps.values()) {
						if (mapData.getBounds().intersects(x, y, w, h)) {
							// 道路の塗りつぶし部を描画する
							if (mapData.hasRoadArc() && mapData.hasTyome()) {
								// 一般の道路を描画する
								for (final ArcData arc : mapData.getRoadArc().values()) {
									if (arc.getRoadType() == ArcData.ROAD_NORMAL) {
										g.setColor(this.mapPreferences.getNormalRoadPreferences().getFillColor());
										this.setVariableStroke(g, this.mapPreferences.getNormalRoadPreferences()
												.getWidth(), isTransform);
										this.draw(g, arc.getPath(), isTransform, transform);
									} else if (arc.getRoadType() == ArcData.ROAD_MAJOR) {
										g.setColor(this.mapPreferences.getMajorRoadPreferences().getFillColor());
										this.setVariableStroke(g, this.mapPreferences.getMajorRoadPreferences()
												.getWidth(), isTransform);
										this.draw(g, arc.getPath(), isTransform, transform);
									}
								}
								// 主要地方道、県道を描画する
								for (final ArcData arc : mapData.getRoadArc().values()) {
									if (arc.getRoadType() == ArcData.ROAD_KENDO) {
										this.setVariableStroke(g, this.mapPreferences.getKendoPreferences().getWidth(),
												isTransform);
										g.setColor(this.mapPreferences.getKendoPreferences().getFillColor());
										this.draw(g, arc.getPath(), isTransform, transform);
									} else if (arc.getRoadType() == ArcData.ROAD_CHIHODO) {
										this.setVariableStroke(g, this.mapPreferences.getChihodoPreferences()
												.getWidth(), isTransform);
										g.setColor(this.mapPreferences.getChihodoPreferences().getFillColor());
										this.draw(g, arc.getPath(), isTransform, transform);
									}
								}
								// 国道を描画する
								this.setVariableStroke(g, this.mapPreferences.getKokudoPreferences().getWidth(),
										isTransform);
								g.setColor(this.mapPreferences.getKokudoPreferences().getFillColor());
								for (final ArcData arc : mapData.getRoadArc().values()) {
									if (arc.getRoadType() == ArcData.ROAD_KOKUDO) {
										this.draw(g, arc.getPath(), isTransform, transform);
									}
								}
								// 高速道路を描画する
								this.setVariableStroke(g, this.mapPreferences.getHighwayPreferences().getWidth(),
										isTransform);
								g.setColor(this.mapPreferences.getHighwayPreferences().getFillColor());
								for (final ArcData arc : mapData.getRoadArc().values()) {
									if (arc.getRoadType() == ArcData.ROAD_HIGHWAY) {
										this.draw(g, arc.getPath(), isTransform, transform);
									}
								}
							}
						}
					}
					for (final MapData mapData : this.maps.values()) {
						if (mapData.getBounds().intersects(x, y, w, h)) {
							// 行政界を描画する
							if (mapData.hasGyousei() && mapData.hasTyome()) {
								for (final ArcData arc : mapData.getGyousei().values()) {
									if (arc.getTag() == ArcData.TAG_NORMAL) {
										if ((arc.getClassification() == ArcData.TYPE_GYOUSEI_PREFECTURE)
												|| (arc.getClassification() == ArcData.TYPE_GYOUSEI_CITY)
												|| (arc.getClassification() == ArcData.TYPE_GYOUSEI_VILLAGE)) {
											g.setColor(this.mapPreferences.getSi_tyoPreferences().getBorderColor());
											this.setVariableStroke(g, this.mapPreferences.getSi_tyoPreferences()
													.getWidth(), isTransform);
										} else {
											g.setColor(this.mapPreferences.getTyomePreferences().getBorderColor());
											this.setVariableStroke(g, this.mapPreferences.getTyomePreferences()
													.getWidth(), isTransform);
										}
										this.draw(g, arc.getPath(), isTransform, transform);
									}
								}
							}
						}
					}
					final float jrShinkansenRailwayWidth = this.mapPreferences.getJRShinkansenPreferences().getWidth();
					final float otherRailwayWidth = this.mapPreferences.getRailwayPreferences().getWidth();
					for (final MapData mapData : this.maps.values()) {
						if (mapData.getBounds().intersects(x, y, w, h)) {
							// 鉄道を描画する
							if (mapData.hasOthers() && mapData.hasTyome()) {
								this.setVariableStroke(g, this.mapPreferences.getJRPreferences().getWidth() + 4,
										isTransform);
								for (final ArcData arc : mapData.getOthers().values()) {
									if (arc.getClassification() == ArcData.TYPE_RAILWAY) {
										if (arc.getRailwayType() == ArcData.RAILWAY_JR) {
											g.setColor(this.mapPreferences.getJRPreferences().getBorderColor());
											this.draw(g, arc.getPath(), isTransform, transform);
										} else if (arc.getRailwayType() == ArcData.RAILWAY_JR_SHINKANSEN) {
											g.setColor(this.mapPreferences.getJRShinkansenPreferences()
													.getBorderColor());
											this.draw(g, arc.getPath(), isTransform, transform);
										}
									}
								}
								for (final ArcData arc : mapData.getOthers().values()) {
									if (arc.getClassification() == ArcData.TYPE_RAILWAY) {
										if (arc.getRailwayType() == ArcData.RAILWAY_JR) {
											if (isTransform) {
												g
														.setStroke(new BasicStroke(this.mapPreferences
																.getJRPreferences().getWidth(), BasicStroke.CAP_BUTT,
																BasicStroke.JOIN_MITER, 10,
																new float[] { this.mapPreferences.getJRPreferences()
																		.getWidth() * 5 }, 0));
											} else {
												g.setStroke(new BasicStroke((float) (this.mapPreferences
														.getJRPreferences().getWidth() * this.zoom),
														BasicStroke.CAP_BUTT, BasicStroke.JOIN_MITER,
														(float) (10 * this.zoom),
														new float[] { (float) (this.mapPreferences.getJRPreferences()
																.getWidth() * 5 * this.zoom) }, 0));
											}
											g.setColor(this.mapPreferences.getJRPreferences().getFillColor());
											this.draw(g, arc.getPath(), isTransform, transform);
										} else if (arc.getRailwayType() == ArcData.RAILWAY_JR_SHINKANSEN) {
											if (isTransform) {
												g
														.setStroke(new BasicStroke(jrShinkansenRailwayWidth,
																BasicStroke.CAP_BUTT, BasicStroke.JOIN_MITER, 10,
																new float[] { this.mapPreferences.getJRPreferences()
																		.getWidth() * 15 }, 0));
											} else {
												g.setStroke(new BasicStroke(
														(float) (jrShinkansenRailwayWidth * this.zoom),
														BasicStroke.CAP_BUTT, BasicStroke.JOIN_MITER,
														(float) (10 * this.zoom),
														new float[] { (float) (this.mapPreferences.getJRPreferences()
																.getWidth() * 15 * this.zoom) }, 0));
											}
											g.setColor(this.mapPreferences.getJRShinkansenPreferences().getFillColor());
											this.draw(g, arc.getPath(), isTransform, transform);
										} else {
											this.setVariableStroke(g, otherRailwayWidth, isTransform);
											g.setColor(this.mapPreferences.getRailwayPreferences().getBorderColor());
											this.draw(g, arc.getPath(), isTransform, transform);
										}
									}
								}
							}
						}
					}
					// 丁目の情報がないときは、鉄道、駅、高速道路、国道、市区町村界を描画する
					for (final MapData mapData : this.maps.values()) {
						if (mapData.getBounds().intersects(x, y, w, h)) {
							if (!mapData.hasRoadArc()) {
								if (mapData.hasLargeRoadArc()) {
									if (mapData.hasTyome()) {
										this.setFixedStroke(g, 3, isTransform);
										for (final ArcData arc : mapData.getLargeRoadArc().values()) {
											if (arc.getRoadType() == ArcData.ROAD_HIGHWAY) {
												g
														.setColor(this.mapPreferences.getHighwayPreferences()
																.getBorderColor());
											} else if (arc.getRoadType() == ArcData.ROAD_KOKUDO) {
												g.setColor(this.mapPreferences.getKokudoPreferences().getBorderColor());
											} else if (arc.getRoadType() == ArcData.ROAD_CHIHODO) {
												g
														.setColor(this.mapPreferences.getChihodoPreferences()
																.getBorderColor());
											}
											this.draw(g, arc.getPath(), isTransform, transform);
										}
										this.setFixedStroke(g, 2, isTransform);
										for (final ArcData arc : mapData.getLargeRoadArc().values()) {
											if (arc.getRoadType() == ArcData.ROAD_HIGHWAY) {
												g.setColor(this.mapPreferences.getHighwayPreferences().getFillColor());
											} else if (arc.getRoadType() == ArcData.ROAD_KOKUDO) {
												g.setColor(this.mapPreferences.getKokudoPreferences().getFillColor());
											} else if (arc.getRoadType() == ArcData.ROAD_CHIHODO) {
												g.setColor(this.mapPreferences.getChihodoPreferences().getFillColor());
											}
											this.draw(g, arc.getPath(), isTransform, transform);
										}
									} else {
									}
									this.setFixedStroke(g, 1, isTransform);
									for (final ArcData arc : mapData.getLargeRoadArc().values()) {
										if (arc.getRoadType() == ArcData.ROAD_HIGHWAY) {
											g.setColor(this.mapPreferences.getHighwayPreferences().getBorderColor());
										} else if (arc.getRoadType() == ArcData.ROAD_KOKUDO) {
											g.setColor(this.mapPreferences.getKokudoPreferences().getBorderColor());
										} else if (arc.getRoadType() == ArcData.ROAD_CHIHODO) {
											g.setColor(this.mapPreferences.getChihodoPreferences().getBorderColor());
										}
										this.draw(g, arc.getPath(), isTransform, transform);
									}
								}
							}
							if (!mapData.hasTyome()) {
								if (mapData.hasOthers()) {
									g.setColor(this.mapPreferences.getRailwayPreferences().getBorderColor());
									this.setFixedStroke(g, 1, isTransform);
									for (final ArcData arc : mapData.getOthers().values()) {
										if (arc.getClassification() == ArcData.TYPE_RAILWAY) {
											this.draw(g, arc.getPath(), isTransform, transform);
										}
									}
								}
							}
							// for debug
							//						g2.setColor(this.mapPreferences.getMapBoundsColor());
							//						g2.setStroke(new BasicStroke(1f));
							//						g2.draw(mapData.getBounds());
							//						g2.setFont(new Font("default", Font.PLAIN, 300));
							//						g2.drawString(mapData.getMapName(), (float) mapData.getBounds().getBounds().getX(), (float) mapData.getBounds().getBounds().getMaxY());
						}
					}
					if (isTransform) {
						g.transform(transform.createInverse());
					}
					g.setStroke(new BasicStroke(1f));
					for (final MapData mapData : this.maps.values()) {
						// 建物のラベルを描画する
						if (this.isVisible(mapData.getBounds())) {
							if (mapData.hasTatemono()) {
								g.setColor(this.mapPreferences.getTatemonoPreferences().getAttributeColor());
								final Font tatemonoFont = this.mapPreferences.getTatemonoPreferences().getFont();
								g.setFont(tatemonoFont);
								final FontMetrics metrics = this.getFontMetrics(tatemonoFont);
								final int descent = metrics.getDescent();
								final double pointSize = 4;
								for (final PolygonData polygon : mapData.getTatemono().values()) {
									if (polygon.getAttribute() != null) {
										if (this.getVisibleRectangle().contains(polygon.getAttributeX(),
												polygon.getAttributeY())) {
											g.fill(new Ellipse2D.Double((polygon.getX() * this.zoom) - this.offsetX
													- (pointSize / 2), (polygon.getY() * this.zoom) - this.offsetY
													- (pointSize / 2), pointSize, pointSize));
											g
													.drawString(polygon.getAttribute(), (float) ((polygon
															.getAttributeX() * this.zoom) - this.offsetX),
															(float) ((polygon.getAttributeY() * this.zoom)
																	- this.offsetY - descent));
										}
									}
								}
							}
							// 場地のラベルを描画する
							if (mapData.hasZyouti()) {
								g.setColor(this.mapPreferences.getZyoutiPreferences().getAttributeColor());
								final Font zyoutiFont = this.mapPreferences.getZyoutiPreferences().getFont();
								g.setFont(zyoutiFont);
								final FontMetrics metrics = this.getFontMetrics(zyoutiFont);
								final int descent = metrics.getDescent();
								for (final PolygonData polygon : mapData.getZyouti().values()) {
									if (polygon.getAttribute() != null) {
										if (this.getVisibleRectangle().contains(polygon.getAttributeX(),
												polygon.getAttributeY())) {
											g
													.drawString(polygon.getAttribute(), (float) ((polygon
															.getAttributeX() * this.zoom) - this.offsetX),
															(float) ((polygon.getAttributeY() * this.zoom)
																	- this.offsetY - descent));
										}
									}
								}
							}
							// 内水面のラベルを描画する
							if (mapData.hasMizu()) {
								g.setColor(this.mapPreferences.getMizuPreferences().getAttributeColor());
								final Font mizuFont = this.mapPreferences.getMizuPreferences().getFont();
								g.setFont(mizuFont);
								final FontMetrics metrics = this.getFontMetrics(mizuFont);
								final int descent = metrics.getDescent();
								for (final PolygonData polygon : mapData.getMizu().values()) {
									if (polygon.getAttribute() != null) {
										if (this.getVisibleRectangle().contains(polygon.getAttributeX(),
												polygon.getAttributeY())) {
											g
													.drawString(polygon.getAttribute(), (float) ((polygon
															.getAttributeX() * this.zoom) - this.offsetX),
															(float) ((polygon.getAttributeY() * this.zoom)
																	- this.offsetY - descent));
										}
									}
								}
							}
							// 丁目のラベルを描画する 
							if (mapData.hasTyome()) {
								g.setColor(this.mapPreferences.getTyomePreferences().getAttributeColor());
								for (final PolygonData polygon : mapData.getTyome().values()) {
									final Font tyomeFont = polygon.getTyomeFont();
									if (tyomeFont != null) {
										g.setFont(tyomeFont);
										final FontMetrics metrics = this.getFontMetrics(tyomeFont);
										final int descent = metrics.getDescent();
										if (polygon.getClassificationCode() == PolygonData.CLASSIFICATION_TYOME) {
											if (polygon.getAttribute() != null) {
												if (this.getVisibleRectangle().contains(polygon.getAttributeX(),
														polygon.getAttributeY())) {
													g.drawString(polygon.getAttribute(), (float) ((polygon
															.getAttributeX() * this.zoom) - this.offsetX),
															(float) ((polygon.getAttributeY() * this.zoom)
																	- this.offsetY - descent));
												}
											}
										}
									}
								}
							}
							// 町丁目の読みを描画する
							if (this.zoom >= Const.Zoom.LOAD_ALL) {
								if (mapData.hasTyome()) {
									g.setColor(this.mapPreferences.getTyomePreferences().getAttributeColor());
									final Font yomiFont = this.mapPreferences.getTyomePreferences().getFont();
									g.setFont(yomiFont);
									final int descent = g.getFontMetrics().getDescent();
									for (final PolygonData polygon : mapData.getTyome().values()) {
										if (polygon.getClassificationCode() == PolygonData.CLASSIFICATION_TYOME) {
											if (polygon.getAttribute() != null) {
												if (this.getVisibleRectangle().contains(polygon.getAttributeX(),
														polygon.getAttributeY())) {
													if (polygon.hasYomi()) {
														g.drawString(polygon.getYomi(), (float) (polygon.getYomiX()
																* this.zoom - this.offsetX), (float) (polygon
																.getYomiY()
																* this.zoom - this.offsetY)
																- descent);
													}
												}
											}
										}
									}
								}
							}
							// 駅のラベルを描画する
							if (mapData.hasEki()) {
								final double pointSize = this.mapPreferences.getEkiPreferences().getWidth();
								final Font ekiFont = this.mapPreferences.getEkiPreferences().getFont();
								g.setFont(ekiFont);
								final FontMetrics metrics = this.getFontMetrics(ekiFont);
								final int descent = metrics.getDescent();
								for (final PointData point : mapData.getEki().values()) {
									if (point.getAttribute() != null) {
										final Ellipse2D ellipse = new Ellipse2D.Double((point.getX() * this.zoom)
												- this.offsetX - (pointSize / 2), (point.getY() * this.zoom)
												- this.offsetY - (pointSize / 2), pointSize, pointSize);
										g.setColor(this.mapPreferences.getEkiPreferences().getFillColor());
										g.fill(ellipse);
										g.setColor(this.mapPreferences.getEkiPreferences().getBorderColor());
										g.draw(ellipse);
										if (point.getAttributeX() != 0 && point.getAttributeY() != 0) {
											g
													.drawString(
															point.getAttribute(),
															(float) ((point.getAttributeX() * this.zoom) - this.offsetX),
															(float) ((point.getAttributeY() * this.zoom) - this.offsetY - descent));
										}
									}
								}
							}
							// 道路のラベルを描画する
							final Font roadFont = this.mapPreferences.getNormalRoadPreferences().getFont();
							if (mapData.hasRoadArc()) {
								g.setFont(roadFont);
								final FontMetrics metrics = this.getFontMetrics(roadFont);
								final int descent = metrics.getDescent();
								g.setColor(this.mapPreferences.getNormalRoadPreferences().getAttributeColor());
								for (final ArcData arc : mapData.getRoadArc().values()) {
									if (arc.getAttribute() != null) {
										g.drawString(arc.getAttribute(),
												(float) ((arc.getAttributeX() * this.zoom) - this.offsetX),
												(float) ((arc.getAttributeY() * this.zoom) - this.offsetY - descent));
									}
								}
							}
							// 高速道路、国道のラベルを描画する
							if (!mapData.hasRoadArc() && mapData.hasLargeRoadArc()) {
								g.setFont(roadFont);
								final FontMetrics metrics = this.getFontMetrics(roadFont);
								final int descent = metrics.getDescent();
								g.setColor(this.mapPreferences.getNormalRoadPreferences().getAttributeColor());
								for (final ArcData arc : mapData.getLargeRoadArc().values()) {
									if (arc.getAttribute() != null) {
										g.drawString(arc.getAttribute(),
												(float) ((arc.getAttributeX() * this.zoom) - this.offsetX),
												(float) ((arc.getAttributeY() * this.zoom) - this.offsetY - descent));
									}
								}
							}
							// 鉄道のラベルを描画する
							if (mapData.hasOthers()) {
								final Font railFont = this.mapPreferences.getRailwayPreferences().getFont();
								g.setFont(railFont);
								final FontMetrics metrics = this.getFontMetrics(railFont);
								final int descent = metrics.getDescent();
								g.setColor(this.mapPreferences.getRailwayPreferences().getAttributeColor());
								for (final ArcData arc : mapData.getOthers().values()) {
									if (arc.getAttribute() != null) {
										if (arc.getAttributeX() != 0 && arc.getAttributeY() != 0) {
											g
													.drawString(
															arc.getAttribute(),
															(float) ((arc.getAttributeX() * this.zoom) - this.offsetX),
															(float) ((arc.getAttributeY() * this.zoom) - this.offsetY - descent));
										}
									}
								}
							}
						}
					}
					// 銀行、コンビニ、ファストフード店を描画する
					if (this.getZoom() >= Const.Zoom.LOAD_ALL) {
						final Font font = this.mapPreferences.getTatemonoPreferences().getFont();
						final int descent = this.getFontMetrics(font).getDescent();
						final double pointSize = 4;
						g.setFont(font);
						g.setColor(this.mapPreferences.getTatemonoPreferences().getAttributeColor());
						for (final Prefecture prefecture : this.prefectures) {
							if (prefecture.hasCities()) {
								for (final City city : prefecture.getCities()) {
									if (this.isVisible(city.hasFineShape() ? city.getFineShape() : city.getShape())) {
										if (city.hasShops()) {
											for (final PointData point : city.getShops()) {
												if (point.getAttribute() != null) {
													if (this.getVisibleRectangle().contains(point.getAttributeX(),
															point.getAttributeY())) {
														g.fill(new Ellipse2D.Double((point.getX() * this.zoom)
																- this.offsetX - (pointSize / 2),
																(point.getY() * this.zoom) - this.offsetY
																		- (pointSize / 2), pointSize, pointSize));
														g.drawString(point.getAttribute(), (float) ((point
																.getAttributeX() * this.zoom) - this.offsetX),
																(float) ((point.getAttributeY() * this.zoom)
																		- this.offsetY - descent));
													}
												}
											}
										}
									}
								}
							}
						}
					}
					// 街区レベル位置参照情報を描画してみる
					if (this.getZoom() >= Const.Zoom.LOAD_ALL) {
						final Font font = this.mapPreferences.getIsjPreferences().getFont();
						final int descent = this.getFontMetrics(font).getDescent();
						g.setFont(font);
						g.setColor(this.mapPreferences.getIsjPreferences().getAttributeColor());
						for (final Prefecture prefecture : this.prefectures) {
							if (prefecture.hasCities()) {
								for (final City city : prefecture.getCities()) {
									if (this.isVisible(city.hasFineShape() ? city.getFineShape() : city.getShape())) {
										for (final Map.Entry<Point2D, String> entry : city.getIsjLabels().entrySet()) {
											g
													.drawString(entry.getValue(), (float) (entry.getKey().getX()
															* this.zoom - this.offsetX), (float) (entry.getKey().getY()
															* this.zoom - this.offsetY - descent));
										}
									}
								}
							}
						}
					}
				}
			}
		} catch (final Exception e) {
			e.printStackTrace();
		}
		this.removeMessage();
	}

	/**
	 * 表示倍率に応じて線の幅が変わるように、太めの線を設定します。
	 * @param g 描画対象
	 * @param strokeWidth 線の幅
	 * @param isTransform 描画対象全体を座標変換するかどうか
	 */
	private void setVariableFatStroke(final Graphics2D g, final float strokeWidth, final boolean isTransform) {
		if (isTransform) {
			g.setStroke(new BasicStroke(strokeWidth + (float) (2 / this.zoom)));
		} else {
			g.setStroke(new BasicStroke((float) (strokeWidth * this.zoom) + 2));
		}
	}

	/**
	 * 表示倍率に応じて線の幅が変わるように設定します。
	 * @param g 描画対象
	 * @param strokeWidth 線の幅
	 * @param isTransform 描画対象全体を座標変換するかどうか
	 */
	private void setVariableStroke(final Graphics2D g, final float strokeWidth, final boolean isTransform) {
		if (isTransform) {
			g.setStroke(new BasicStroke(strokeWidth));
		} else {
			g.setStroke(new BasicStroke((float) (strokeWidth * this.zoom)));
		}
	}

	/**
	 * 図形の輪郭を描画します。
	 * @param g 描画対象
	 * @param shape 描画するオブジェクト
	 * @param isTransform 描画対象全体を座標変換するかどうか
	 * @param transform 座標変換
	 */
	private void draw(final Graphics2D g, final Shape shape, final boolean isTransform, final AffineTransform transform) {
		if (isTransform) {
			g.draw(shape);
		} else {
			g.draw(transform.createTransformedShape(shape));
		}
	}

	/**
	 * 図形を塗りつぶします。
	 * @param g 描画対象
	 * @param shape 描画するオブジェクト
	 * @param isTransform 描画対象全体を座標変換するかどうか
	 * @param transform 座標変換
	 */
	private void fill(final Graphics2D g, final Shape shape, final boolean isTransform, final AffineTransform transform) {
		if (isTransform) {
			g.fill(shape);
		} else {
			g.fill(transform.createTransformedShape(shape));
		}
	}

	/**
	 * 表示倍率にかかわらず、一定の太さに設定します。
	 * @param g 描画対象
	 * @param strokeWidth 線の幅
	 * @param isTransform 描画対象全体を座標変換するかどうか
	 */
	private void setFixedStroke(final Graphics2D g, final float strokeWidth, final boolean isTransform) {
		if (isTransform) {
			g.setStroke(new BasicStroke((float) (strokeWidth / this.zoom)));
		} else {
			g.setStroke(new BasicStroke(strokeWidth));
		}
	}

	/**
	 * @return 背景スレッドに再計算を要求するためのアクションリスナ
	 */
	public ActionListener getActionListener() {
		return this.listener;
	}

	/**
	 * 指定された地図全てが収まる長方形を求めます。
	 * @param mapNames 地図の一覧
	 * @return 長方形
	 * @since 3.03
	 */
	public Rectangle2D getBounds(final Collection<String> mapNames) {
		Rectangle2D ret = null;
		for (final String mapName : mapNames) {
			if (this.maps.containsKey(mapName)) {
				final Rectangle bounds = this.maps.get(mapName).getBounds().getBounds();
				if (ret == null) {
					ret = bounds;
				} else {
					ret = ret.createUnion(bounds);
				}
			}
		}
		return ret;
	}

	/**
	 * @return 文字の大きさ
	 */
	public double getFontZoom() {
		return this.fontZoom;
	}

	/**
	 * @return 地図の設定
	 */
	public MapPreferences getMapPreferences() {
		return this.mapPreferences;
	}

	/**
	 * オブジェクトが存在する最も大きい x 座標を取得します。
	 * @return オブジェクトが存在する最も大きい x 座標
	 */
	double getMaxX() {
		return this.maxX;
	}

	/**
	 * オブジェクトが存在する最も大きい y 座標を取得します。
	 * @return オブジェクトが存在する最も大きい y 座標
	 */
	double getMaxY() {
		return this.maxY;
	}

	/**
	 * オブジェクトが存在する最も小さい x 座標を取得します。
	 * @return オブジェクトが存在する最も小さい x 座標
	 */
	double getMinX() {
		return this.minX;
	}

	/** オブジェクトが存在する最も小さい y 座標を取得します。
	 * @return オブジェクトが存在する最も小さい x 座標
	 */
	double getMinY() {
		return this.minY;
	}

	/**
	 * @return 表示倍率の最小値
	 */
	public double getMinZoom() {
		return this.minZoom;
	}

	/**
	 * オブジェクトが存在する範囲を取得します。
	 * @return オブジェクトが存在する範囲（仮想座標）
	 */
	Rectangle2D getObjectArea() {
		return new Rectangle2D.Double(this.minX, this.minY, this.maxX - this.minX, this.maxY - this.minY);
	}

	/**
	 * @return x方向のオフセット
	 */
	public double getOffsetX() {
		return this.offsetX;
	}

	/**
	 * @return y方向のオフセット
	 */
	public double getOffsetY() {
		return this.offsetY;
	}

	/**
	 * @return 都道府県の一覧
	 */
	public Collection<Prefecture> getPrefectures() {
		return this.prefectures;
	}

	/**
	 * @return 地図を検索するためのデータ構造
	 */
	public Search getSearch() {
		return this.search;
	}

	/**
	 * 表示されている範囲を取得します。
	 * @return 表示されている範囲（仮想座標）
	 */
	Rectangle2D getVisibleRectangle() {
		final int width = (this.size == null) ? this.getWidth() : this.size.width;
		final int height = (this.size == null) ? this.getHeight() : this.size.height;
		return new Rectangle2D.Double(this.offsetX / this.zoom, this.offsetY / this.zoom, width / this.zoom, height
				/ this.zoom);
	}

	/**
	 * 倍率を取得します。
	 * @return 倍率
	 */
	public double getZoom() {
		return this.zoom;
	}

	/**
	 * 文字を大きくします。
	 */
	public void increaseFontSize() {
		this.fontZoom /= 0.9;
		this.mapPreferences.setFontZoom(this.fontZoom);
		if (this.listener != null) {
			this.listener.actionPerformed(new ActionEvent(this, this.hashCode(), "increment font size"));
		}
		this.isChanged = true;
		this.repaint();
	}

	/**
	 * 彩度を増やします。
	 */
	public void increaseSaturation() {
		this.saturationDifference += 0.05;
		this.isChanged = true;
		this.repaint();
	}

	/**
	 * 指定したオブジェクトが表示エリア内にあるかどうかを取得します。
	 * @param shape オブジェクト
	 * @return 指定したオブジェクトが表示エリア内にあるかどうか
	 */
	boolean isVisible(final Shape shape) {
		return shape.intersects(this.getVisibleRectangle());
	}

	/**
	 * 指定した点が画面の中央になるように、地図をスクロールさせます。
	 * @param x x座標（仮想座標）
	 * @param y y座標（仮想座標）
	 */
	public void moveTo(final double x, final double y) {
		final int width = (this.size == null) ? this.getWidth() : this.size.width;
		final int height = (this.size == null) ? this.getHeight() : this.size.height;
		this.offsetX = x * this.zoom - width / 2;
		this.offsetY = y * this.zoom - height / 2;
		this.isChanged = true;
	}

	/**
	 * 指定された長方形が表示されるように表示倍率を変更し、表示位置を移動します。
	 * @param rectangle 長方形
	 * @since 3.03
	 */
	public void moveTo(final Rectangle2D rectangle) {
		final int width = (this.size == null) ? this.getWidth() : this.size.width;
		final int height = (this.size == null) ? this.getHeight() : this.size.height;
		final double zoomX = width / rectangle.getWidth();
		final double zoomY = height / rectangle.getHeight();
		if (zoomY < zoomX) {
			this.zoom = zoomY;
		} else {
			this.zoom = zoomX;
		}
		this.moveTo(rectangle.getCenterX(), rectangle.getCenterY());
	}

	/**
	 * 明石市に移動します。
	 */
	public void moveToAkashi() {
		this.zoom = 0.36;
		this.offsetX = 23000;
		this.offsetY = -1382500;
	}

	/**
	 * 地図の中央が画面の中央になるように、地図をスクロールさせます。
	 */
	public void moveToCenter() {
		final int width = (this.size == null) ? this.getWidth() : this.size.width;
		final int height = (this.size == null) ? this.getHeight() : this.size.height;
		this.offsetX = ((this.minX + this.maxX) / 2 * this.zoom) - width / 2;
		this.offsetY = ((this.minY + this.maxY) / 2 * this.zoom) - height / 2;
		this.isChanged = true;
	}

	@Override
	public void paintComponent(final Graphics g) {
		super.paintComponent(g);
		try {
			if (this.isChanged) {
				final Image tempImage = this.createImage(this.getWidth(), this.getHeight());
				final Graphics2D g2 = (Graphics2D) tempImage.getGraphics();
				if (this.isAntialias) {
					g2.setRenderingHint(RenderingHints.KEY_ANTIALIASING, RenderingHints.VALUE_ANTIALIAS_ON);
					if (Const.Fonts.HAS_MS_FONTS) {
						g2.setRenderingHint(RenderingHints.KEY_TEXT_ANTIALIASING,
								RenderingHints.VALUE_TEXT_ANTIALIAS_OFF);
					}
				}
				this.drawMap(g2, true);
				g.drawImage(tempImage, 0, 0, this);
				this.image = tempImage;
				this.isChanged = false;
			} else {
				g.drawImage(this.image, 0, 0, this);
			}
		} catch (final IOException e) {
			e.printStackTrace();
		}
	}

	/**
	 * 現在表示されている地図をダイアログを表示して印刷します。
	 * @throws PrintException 印刷例外
	 */
	public void print() throws PrintException {
		PrintUtil.print(this);
	}

	public int print(final Graphics graphics, final PageFormat pageFormat, final int pageIndex) throws PrinterException {
		try {
			if (pageIndex == 0) {
				final Graphics2D g = (Graphics2D) graphics;
				g.translate(pageFormat.getImageableX(), pageFormat.getImageableY());
				final double newZoom = Math.min(pageFormat.getImageableWidth() / this.getWidth(), pageFormat
						.getImageableHeight()
						/ this.getHeight());
				g.scale(newZoom, newZoom);
				g.setClip(0, 0, this.getWidth(), this.getHeight());
				this.drawMap(g, true);
				return Printable.PAGE_EXISTS;
			} else {
				return Printable.NO_SUCH_PAGE;
			}
		} catch (final IOException e) {
			e.printStackTrace();
			return Printable.NO_SUCH_PAGE;
		}
	}

	/**
	 * 現在表示されている地図をPSファイルに出力します。
	 * @param file ファイル
	 * @throws IOException ファイル入出力例外
	 * @throws PrinterException 印刷例外
	 */
	public void printPS(final File file) throws PrinterException, IOException {
		PSOut.print(file, this);
	}

	/**
	 * 現在表示されている地図をラスタファイルに出力します。
	 * @param file ファイル
	 * @param format ファイル形式（png、jpg、bmp）
	 * @throws IOException 入出力例外 
	 */
	public void printRaster(final File file, final String format) throws IOException {
		final BufferedImage tempImage = new BufferedImage(this.getWidth(), this.getHeight(), BufferedImage.TYPE_INT_RGB);
		final Graphics2D g2 = (Graphics2D) tempImage.getGraphics();
		g2.setRenderingHint(RenderingHints.KEY_ANTIALIASING, RenderingHints.VALUE_ANTIALIAS_ON);
		if (Const.Fonts.HAS_MS_FONTS) {
			g2.setRenderingHint(RenderingHints.KEY_TEXT_ANTIALIASING, RenderingHints.VALUE_TEXT_ANTIALIAS_OFF);
		}
		this.drawMap(g2, true);
		ImageIO.write(tempImage, format, file);
	}

	/**
	 * 文字の大きさを標準に戻します。
	 */
	public void resetFontSize() {
		this.fontZoom = 1;
		this.mapPreferences.setFontZoom(this.fontZoom);
		if (this.listener != null) {
			this.listener.actionPerformed(new ActionEvent(this, this.hashCode(), "reset font size"));
		}
		this.isChanged = true;
		this.repaint();
	}

	/**
	 * 彩度をリセットします。
	 */
	public void resetSaturation() {
		this.saturationDifference = 0;
		this.isChanged = true;
		this.repaint();
	}

	/**
	 * スクロールします。
	 * @param dx x軸方向の移動距離（実座標）
	 * @param dy y軸方向の移動距離（実座標）
	 */
	public void scroll(final double dx, final double dy) {
		this.offsetX += dx;
		this.offsetY += dy;
		if (this.listener != null) {
			this.listener.actionPerformed(new ActionEvent(this, this.hashCode(), "move"));
		}
		this.isChanged = true;
	}

	/**
	 * @param listener 背景スレッドに再計算を要求するためのアクションリスナ
	 */
	public void setActionListener(final ActionListener listener) {
		this.listener = listener;
	}

	/**
	 * 地図が変化したことを伝えます。
	 */
	public void setChanged() {
		this.isChanged = true;
	}

	/**
	 * @param minZoom 表示倍率の最小値
	 */
	public void setMinZoom(final double minZoom) {
		this.minZoom = minZoom;
	}

	/**
	 * @param search 地図を検索するためのデータ構造
	 */
	public void setSearch(final Search search) {
		this.search = search;
	}

	/**
	 * SWT版ではパネルの大きさが取得できないので、強制的に設定します。
	 * @param size パネルの大きさ
	 */
	public void setSWTSize(final Dimension size) {
		this.size = size;
	}

	/**
	 * 実座標を取得します。
	 * @param location 仮想座標
	 * @return 実座標
	 */
	Point2D toRealLocation(final Point2D location) {
		return new Point2D.Double((location.getX() * this.zoom) - this.offsetX, (location.getY() * this.zoom)
				- this.offsetY);
	}

	/**
	 * 仮想座標を取得します。
	 * @param location 実座標
	 * @return 仮想座標
	 */
	Point2D toVirtualLocation(final Point2D location) {
		return new Point2D.Double((this.offsetX + location.getX()) / this.zoom, (this.offsetY + location.getY())
				/ this.zoom);
	}

	/**
	 * 倍率を変更します。
	 * @param newZoom 倍率
	 * @param x 中心のx座標（実座標）
	 * @param y 中心のy座標（実座標）
	 */
	private void zoom(final double newZoom, final int x, final int y) {
		final double newX = ((this.offsetX + x) / this.zoom * newZoom) - x;
		final double newY = ((this.offsetY + y) / this.zoom * newZoom) - y;
		this.offsetX = newX;
		this.offsetY = newY;
		this.zoom = newZoom;
		this.isChanged = true;
	}

	/**
	 * 自動倍率設定します。
	 */
	public void zoomAutomaticaly() {
		final int width = (this.size == null) ? this.getWidth() : this.size.width;
		final int height = (this.size == null) ? this.getHeight() : this.size.height;
		final double zoomX = width / (this.maxX - this.minX);
		final double zoomY = height / (this.maxY - this.minY);
		if (zoomY < zoomX) {
			this.zoom = zoomY;
		} else {
			this.zoom = zoomX;
		}
		this.minZoom = this.zoom;
		this.isChanged = true;
	}

	/**
	 * 詳細表示します。
	 */
	public void zoomDetail() {
		this.zoom(Zoom.LOAD_ALL, this.getWidth() / 2, this.getHeight() / 2);
		if (this.listener != null) {
			this.listener.actionPerformed(new ActionEvent(this, this.hashCode(), "zoom detail"));
		}
	}

	/**
	 * 拡大します。
	 */
	public void zoomIn() {
		this.doWheelRotation(1, this.getWidth() / 2, this.getHeight() / 2);
		if (this.listener != null) {
			this.listener.actionPerformed(new ActionEvent(this, this.hashCode(), "zoom in"));
		}
	}

	/**
	 * 中域表示します。
	 */
	public void zoomMiddle() {
		this.zoom(Zoom.LOAD_GYOUSEI, this.getWidth() / 2, this.getHeight() / 2);
		if (this.listener != null) {
			this.listener.actionPerformed(new ActionEvent(this, this.hashCode(), "zoom middle"));
		}
	}

	/**
	 * 縮小します。
	 */
	public void zoomOut() {
		this.doWheelRotation(-1, this.getWidth() / 2, this.getHeight() / 2);
		if (this.listener != null) {
			this.listener.actionPerformed(new ActionEvent(this, this.hashCode(), "zoom out"));
		}
	}

	/**
	 * 全域表示します。
	 */
	public void zoomWhole() {
		this.zoomAutomaticaly();
		this.moveToCenter();
		if (this.listener != null) {
			this.listener.actionPerformed(new ActionEvent(this, this.hashCode(), "zoom whole"));
		}
	}

	/**
	 * 広域表示します。
	 */
	public void zoomWide() {
		this.zoom(Zoom.LOAD_2500, this.getWidth() / 2, this.getHeight() / 2);
		if (this.listener != null) {
			this.listener.actionPerformed(new ActionEvent(this, this.hashCode(), "zoom wide"));
		}
	}

	/**
	 * 国土数値情報の荒い市区町村界を読み込む縮尺にします。
	 */
	public void zoomCities() {
		this.zoom(Zoom.LOAD_CITIES, this.getWidth() / 2, this.getHeight() / 2);
		if (this.listener != null) {
			this.listener.actionPerformed(new ActionEvent(this, this.hashCode(), "zoom cities"));
		}
	}

	/**
	 * 国土数値情報の細かい市区町村界を読み込む縮尺にします。
	 */
	public void zoomFineCities() {
		this.zoom(Zoom.LOAD_FINE_CITIES, this.getWidth() / 2, this.getHeight() / 2);
		if (this.listener != null) {
			this.listener.actionPerformed(new ActionEvent(this, this.hashCode(), "zoom fine cities"));
		}
	}

	/**
	 * SVGファイルを出力します。
	 * @param file ファイル
	 * @throws FileNotFoundException ファイル未検出例外
	 * @throws SVGGraphics2DIOException SVG関連入出力例外
	 * @throws UnsupportedEncodingException サポート外エンコーディング例外
	 */
	public void printSVG(final File file) throws UnsupportedEncodingException, SVGGraphics2DIOException,
			FileNotFoundException {
		SVGOut.print(file, new Paintable() {
			public void paint(final Graphics g) {
				try {
					g.clipRect(0, 0, MapPanel.this.getWidth(), MapPanel.this.getHeight());
					MapPanel.this.drawMap((Graphics2D) g, false);
				} catch (final IOException e) {
					e.printStackTrace();
				}
			}

			public Dimension getSize() {
				return MapPanel.this.getSize();
			}
		});
	}
}
