package map;
import java.util.ArrayList;
import java.util.Collection;
import java.util.HashMap;
import java.util.Map;

/**
 * 地図を丁目単位で塗り分けるクラスです。
 * 作成日: 2004/01/09
 * @author Kumano Tatsuo
 */
class PaintTyome {
    /**
     * 地図が変化したかどうか
     */
    private boolean isChanged;

    /** 地図を丁目単位で塗り分けます。
     * このメソッドを呼び出した直後に isChanged() を呼び出すと、
     * このメソッドによって地図の状態が変化したかどうかが取得できます。
     * @param maps 地図
     * @throws Exception 例外
     */
    void paintTyome(final Map<String, MapData> maps) throws Exception {
        this.isChanged = false;
        final Map<String, Collection<PolygonData>> attributePolygonMap = new HashMap<String, Collection<PolygonData>>(); // String -> Collection<Polygon> の Map
        // 属性をキー、ポリゴンの集合を値とする Map を初期化する
        for (final MapData mapData : maps.values()) {
            if (mapData.hasTyome()) {
                for (final PolygonData polygon : mapData.getTyome().values()) {
                    final String attribute = polygon.getAttribute();
                    if (attribute != null) {
                        if (!attributePolygonMap.containsKey(attribute)) {
                            attributePolygonMap.put(attribute, new ArrayList<PolygonData>());
                        }
                        attributePolygonMap.get(attribute).add(polygon);
                    }
                }
            }
        }
        //System.out.println("attribute <-> polygon = " + attributePolygonMap);
        // Polygon をキー、Collection<Polygon> を値とする Map を作る
        final Map<PolygonData, Collection<PolygonData>> adjacentGraph = new HashMap<PolygonData, Collection<PolygonData>>();
        for (final MapData mapData : maps.values()) {
            if (mapData.hasTyome()) {
                final Map<String, Collection<PolygonData>> adjacentPolygons = mapData.getAdjacentGraph();
                for (final String polygonName : adjacentPolygons.keySet()) {
                    adjacentGraph.put(mapData.getTyome().get(polygonName), adjacentPolygons
                            .get(polygonName));
                }
            }
        }
        //System.out.println("隣接グラフ：" + adjacentGraph);
        // 塗り分ける
        for (final MapData mapData : maps.values()) {
            if (mapData.hasTyome()) {
                for (final PolygonData polygon : mapData.getTyome().values()) {
                    if (polygon.getPolygonName() != null) {
                        this.fixColorRecursively(polygon, adjacentGraph, attributePolygonMap);
                    }
                }
            }
        }
    }

    /**
     * 直前の塗り分けで、地図の状態が変化したかどうかを取得します。
     * @return 地図の状態が変化したかどうか
     */
    boolean isChanged() {
        return this.isChanged;
    }

    /**
     * 指定したポリゴンの色を再帰的に決定します。
     * red, green, blue, yellow, magenda, cyan の順に色を決めます。
     * @param polygon ポリゴン
     * @param adjacentGraph 隣接グラフ
     * @param attributePolygonMap 属性とポリゴンの関連づけ
     */
    private void fixColorRecursively(final PolygonData polygon,
            final Map<PolygonData, Collection<PolygonData>> adjacentGraph, final Map<String, Collection<PolygonData>> attributePolygonMap) {
        if (polygon.getTyomeColorIndex() != 0) {
            return;
        }
        final boolean[] isUsed = new boolean[8];
        this.addUsedColors(polygon, adjacentGraph, isUsed);
        final String attribute = polygon.getAttribute();
        if (attribute != null) {
            if (attributePolygonMap.containsKey(attribute)) {
                for (final PolygonData polygon2 : attributePolygonMap.get(attribute)){
                    if (polygon != polygon2) {
                        this.addUsedColors(polygon2, adjacentGraph, isUsed);
                    }
                }
            }
        }
        boolean isPainted = false;
        for (int i = 1; i < 8; ++i) {
            if (!isUsed[i]) {
                polygon.setTyomeColorIndex(i);
                isPainted = true;
                break;
            }
            if (!isPainted) {
				polygon.setTyomeColorIndex(8);
			}
        }
        this.isChanged = true;
        if (attribute != null) {
            if (attributePolygonMap.containsKey(attribute)) {
                for (final PolygonData polygon2 : attributePolygonMap.get(attribute)) {
                    if (polygon != polygon2) {
                        polygon2.setTyomeColorIndex(polygon.getTyomeColorIndex());
                    }
                }
            }
        }
        if (adjacentGraph.containsKey(polygon)) {
            for (final PolygonData polygon2 : adjacentGraph.get(polygon)) {
                this.fixColorRecursively(polygon2, adjacentGraph, attributePolygonMap);
            }
        }
    }

    /**
     * あるポリゴンに隣接しているポリゴンの色を調べます。
     * @param polygon ポリゴン
     * @param adjacentGraph 隣接グラフ
     * @param isUsed その色が使われているかどうか
     */
    private void addUsedColors(final PolygonData polygon,
            final Map<PolygonData, Collection<PolygonData>> adjacentGraph, final boolean[] isUsed) {
        if (adjacentGraph.containsKey(polygon)) {
            for (final PolygonData polygon2 : adjacentGraph.get(polygon)) {
                if (polygon2.getTyomeColorIndex() < isUsed.length) {
                    isUsed[polygon2.getTyomeColorIndex()] = true;
                }
            }
        }
    }
}
