// Licensed to the .NET Foundation under one or more agreements.
// The .NET Foundation licenses this file to you under the MIT license.

using System.Text;
using Test.Cryptography;

namespace System.Security.Cryptography.Rsa.Tests
{
    // Note to contributors:
    //   Keys contained in this file should be randomly generated for the purpose of inclusion here,
    //   or obtained from some fixed set of test data. (Please) DO NOT use any key that has ever been
    //   used for any real purpose.
    //
    // Note to readers:
    //   The keys contained in this file should all be treated as compromised. That means that you
    //   absolutely SHOULD NOT use these keys on anything that you actually want to be protected.
    internal class TestData
    {
        public static readonly byte[] HelloBytes = new ASCIIEncoding().GetBytes("Hello");

        public static readonly RSAParameters RSA384Parameters = new RSAParameters
        {
            Modulus = new byte[]
            {
                0xDA, 0xCC, 0x22, 0xD8, 0x6E, 0x67, 0x15, 0x75,
                0x03, 0x2E, 0x31, 0xF2, 0x06, 0xDC, 0xFC, 0x19,
                0x2C, 0x65, 0xE2, 0xD5, 0x10, 0x89, 0xE5, 0x11,
                0x2D, 0x09, 0x6F, 0x28, 0x82, 0xAF, 0xDB, 0x5B,
                0x78, 0xCD, 0xB6, 0x57, 0x2F, 0xD2, 0xF6, 0x1D,
                0xB3, 0x90, 0x47, 0x22, 0x32, 0xE3, 0xD9, 0xF5,
            },
            Exponent = new byte[]
            {
                0x01, 0x00, 0x01,
            },
            D = new byte[]
            {
                0x7A, 0x59, 0xBD, 0x02, 0x9A, 0x7A, 0x3A, 0x9D,
                0x7C, 0x71, 0xD0, 0xAC, 0x2E, 0xFA, 0x54, 0x5F,
                0x1F, 0x5C, 0xBA, 0x43, 0xBB, 0x43, 0xE1, 0x3B,
                0x78, 0x77, 0xAF, 0x82, 0xEF, 0xEB, 0x40, 0xC3,
                0x8D, 0x1E, 0xCD, 0x73, 0x7F, 0x5B, 0xF9, 0xC8,
                0x96, 0x92, 0xB2, 0x9C, 0x87, 0x5E, 0xD6, 0xE1,
            },
            P = new byte[]
            {
                0xFA, 0xDB, 0xD7, 0xF8, 0xA1, 0x8B, 0x3A, 0x75,
                0xA4, 0xF6, 0xDF, 0xAE, 0xE3, 0x42, 0x6F, 0xD0,
                0xFF, 0x8B, 0xAC, 0x74, 0xB6, 0x72, 0x2D, 0xEF,
            },
            DP = new byte[]
            {
                0x24, 0xFF, 0xBB, 0xD0, 0xDD, 0xF2, 0xAD, 0x02,
                0xA0, 0xFC, 0x10, 0x6D, 0xB8, 0xF3, 0x19, 0x8E,
                0xD7, 0xC2, 0x00, 0x03, 0x8E, 0xCD, 0x34, 0x5D,
            },
            Q = new byte[]
            {
                0xDF, 0x48, 0x14, 0x4A, 0x6D, 0x88, 0xA7, 0x80,
                0x14, 0x4F, 0xCE, 0xA6, 0x6B, 0xDC, 0xDA, 0x50,
                0xD6, 0x07, 0x1C, 0x54, 0xE5, 0xD0, 0xDA, 0x5B,
            },
            DQ = new byte[]
            {
                0x85, 0xDF, 0x73, 0xBB, 0x04, 0x5D, 0x91, 0x00,
                0x6C, 0x2D, 0x45, 0x9B, 0xE6, 0xC4, 0x2E, 0x69,
                0x95, 0x4A, 0x02, 0x24, 0xAC, 0xFE, 0x42, 0x4D,
            },
            InverseQ = new byte[]
            {
                0x1A, 0x3A, 0x76, 0x9C, 0x21, 0x26, 0x2B, 0x84,
                0xCA, 0x9C, 0xA9, 0x62, 0x0F, 0x98, 0xD2, 0xF4,
                0x3E, 0xAC, 0xCC, 0xD4, 0x87, 0x9A, 0x6F, 0xFD,
            },
        };

        public static readonly RSAParameters RSA1024Params = new RSAParameters
        {
            Modulus = new byte[]
            {
                0x9F, 0x05, 0x1F, 0xCE, 0x71, 0xCA, 0x2E, 0x17,
                0x0F, 0x43, 0xC6, 0x04, 0x4A, 0x6F, 0xB7, 0x84,
                0xD8, 0xAD, 0x62, 0x5B, 0xDB, 0x87, 0x4B, 0x05,
                0xAD, 0x03, 0x76, 0x19, 0x63, 0xEE, 0x2A, 0x9D,
                0xCC, 0x7D, 0xAF, 0x3A, 0x51, 0x23, 0xAE, 0x7F,
                0x19, 0xA6, 0x63, 0xBF, 0x65, 0x6B, 0x5E, 0x37,
                0x1C, 0x6A, 0x0A, 0xA7, 0xF0, 0x82, 0xB1, 0xBD,
                0xC3, 0x21, 0x50, 0x21, 0x23, 0xEF, 0x40, 0x30,
                0x93, 0x11, 0x11, 0x5C, 0xF7, 0x5B, 0x42, 0x72,
                0x18, 0xDE, 0xFE, 0x1B, 0xF6, 0x0E, 0x02, 0xBF,
                0x28, 0x56, 0xFB, 0x7C, 0xB5, 0xF1, 0x88, 0xCD,
                0x66, 0x8F, 0xC1, 0xCA, 0xBA, 0xF9, 0x83, 0x7E,
                0x09, 0x5B, 0xD2, 0x91, 0xB9, 0x16, 0x5C, 0x2B,
                0x0C, 0x8E, 0x91, 0xBB, 0xC3, 0x84, 0x0D, 0x73,
                0xEE, 0xA2, 0x30, 0x82, 0x2F, 0x3C, 0x90, 0x50,
                0x88, 0xA7, 0x92, 0x96, 0x99, 0x0C, 0x68, 0xBD,
            },
            Exponent = new byte[]
            {
                0x01, 0x00, 0x01,
            },
            D = new byte[]
            {
                0x4A, 0xE6, 0xF9, 0x7F, 0xDE, 0xE6, 0x5A, 0x46,
                0x5F, 0x58, 0xCF, 0x8D, 0x56, 0xD8, 0x7F, 0x6B,
                0x72, 0x3A, 0x5D, 0x21, 0xA2, 0x6A, 0x7C, 0x42,
                0x7C, 0xA7, 0xAC, 0x39, 0xB2, 0x71, 0xCD, 0x1E,
                0x0D, 0xE3, 0xC7, 0xA5, 0x62, 0xF1, 0xB9, 0x30,
                0x42, 0x0F, 0x37, 0x5D, 0xC0, 0x72, 0x4D, 0xEB,
                0x0C, 0x95, 0xC0, 0x56, 0x31, 0x7A, 0x06, 0x29,
                0xB9, 0x9F, 0x57, 0xE4, 0x7C, 0x4E, 0x25, 0xFF,
                0xDD, 0x11, 0x0E, 0xD9, 0xE8, 0xC0, 0x4F, 0x98,
                0xC6, 0x13, 0xD7, 0x24, 0x7E, 0x3C, 0x3E, 0xB4,
                0xC7, 0x85, 0x8F, 0xD5, 0x94, 0x0B, 0x12, 0x54,
                0xF8, 0xA0, 0x0B, 0x61, 0x6F, 0xDD, 0xE8, 0x9B,
                0x91, 0x9D, 0x2A, 0x6F, 0x15, 0x99, 0x8C, 0x91,
                0xB9, 0x3D, 0xEF, 0x65, 0xC7, 0x78, 0xA6, 0x57,
                0x52, 0xC4, 0x17, 0x5B, 0xFC, 0xA8, 0x65, 0x4E,
                0x1E, 0x95, 0x82, 0x6B, 0x15, 0x49, 0x6C, 0x6B,
            },
            P = new byte[]
            {
                0xD2, 0x4B, 0x09, 0x03, 0x92, 0x29, 0x95, 0xE7,
                0x65, 0x94, 0x19, 0x4F, 0x20, 0xD2, 0xB7, 0xF6,
                0xB9, 0xB0, 0x78, 0x1F, 0xB7, 0xC2, 0xDD, 0xB4,
                0x68, 0xB5, 0xFD, 0x7D, 0xD1, 0x92, 0xBA, 0xA8,
                0x50, 0x1E, 0xDC, 0x44, 0x79, 0xBF, 0x21, 0x2F,
                0xC2, 0xB5, 0xA0, 0x08, 0x0E, 0xCE, 0x80, 0x0C,
                0x37, 0xF6, 0x60, 0xEA, 0xBB, 0x57, 0x7E, 0xA5,
                0x74, 0x82, 0x8A, 0x9F, 0xA7, 0x5B, 0x42, 0x03,
            },
            DP = new byte[]
            {
                0x77, 0xCD, 0x77, 0x9D, 0x29, 0x2F, 0xB7, 0xCE,
                0xD3, 0xF7, 0xC3, 0x53, 0x69, 0x07, 0xA2, 0xF6,
                0x54, 0x63, 0x4C, 0x8C, 0x05, 0x4C, 0x66, 0xB1,
                0xD8, 0xD5, 0x95, 0x4C, 0x90, 0x90, 0x5E, 0xF6,
                0x74, 0x6E, 0xA0, 0x5E, 0x02, 0x5D, 0xF8, 0xB2,
                0x14, 0xE3, 0x14, 0x00, 0x83, 0x2E, 0xF1, 0x94,
                0x04, 0x6D, 0xC0, 0x58, 0xF9, 0xD1, 0xA6, 0xBC,
                0xEB, 0xDB, 0x52, 0xCE, 0x11, 0xB1, 0xD3, 0xB1,
            },
            Q = new byte[]
            {
                0xC1, 0x95, 0x31, 0x1A, 0xF4, 0xE5, 0xA4, 0x1F,
                0xAC, 0x86, 0x48, 0x35, 0x9B, 0x7E, 0x72, 0xF9,
                0x82, 0x17, 0x2A, 0x86, 0x0A, 0x3D, 0x31, 0xCC,
                0xD1, 0x1C, 0x23, 0xBB, 0x64, 0x90, 0xDF, 0x1F,
                0x56, 0xA7, 0x48, 0x24, 0x9E, 0xAF, 0x18, 0xE3,
                0x0C, 0xB6, 0xE6, 0x7B, 0xE4, 0x1F, 0xF7, 0x19,
                0x6D, 0x4E, 0xDB, 0xA9, 0x50, 0xCD, 0xD6, 0x28,
                0x62, 0xEF, 0x65, 0x72, 0x48, 0xA9, 0x0E, 0x3F,
            },
            DQ = new byte[]
            {
                0xAE, 0x52, 0x33, 0x0E, 0x1B, 0x4A, 0x50, 0x29,
                0x55, 0xAA, 0xF6, 0x8B, 0x8F, 0xA2, 0xA6, 0xD6,
                0x98, 0x97, 0x53, 0xEB, 0xB0, 0x7C, 0xBA, 0xC3,
                0xBD, 0xEA, 0xA1, 0x22, 0xB6, 0xC4, 0xDE, 0xA7,
                0xD1, 0xD8, 0x81, 0xD6, 0xB8, 0x2E, 0xE5, 0x32,
                0x50, 0xD8, 0xC3, 0x64, 0xFD, 0x60, 0xEB, 0x9B,
                0x32, 0x1B, 0xB9, 0x23, 0x17, 0x68, 0xC4, 0x59,
                0x49, 0xFE, 0x5A, 0x54, 0x37, 0xAA, 0x44, 0xF1,
            },
            InverseQ = new byte[]
            {
                0x2A, 0x39, 0xAF, 0xE2, 0x76, 0x06, 0x48, 0x8F,
                0x49, 0x93, 0xDF, 0x01, 0xCD, 0x7B, 0xB2, 0xC6,
                0x68, 0x15, 0xAF, 0x0E, 0xC9, 0xF1, 0xE0, 0x15,
                0x3D, 0xA9, 0x68, 0x27, 0xD6, 0x64, 0x1E, 0x08,
                0xDC, 0x01, 0x3B, 0x38, 0x32, 0x62, 0x4F, 0xCD,
                0x0C, 0x09, 0xF7, 0x9A, 0x6B, 0xCA, 0x0F, 0x76,
                0xC2, 0xFB, 0xDD, 0x7C, 0x5D, 0xF1, 0x85, 0x16,
                0x67, 0x9F, 0x98, 0xA7, 0xF3, 0x40, 0x7D, 0x1C,
            },
        };

        public static readonly RSAParameters RSA1032Parameters = new RSAParameters
        {
            Modulus = new byte[]
            {
                0xBC, 0xAC, 0xB1, 0xA5, 0x34, 0x9D, 0x7B, 0x35,
                0xA5, 0x80, 0xAC, 0x3B, 0x39, 0x98, 0xEB, 0x15,
                0xEB, 0xF9, 0x00, 0xEC, 0xB3, 0x29, 0xBF, 0x1F,
                0x75, 0x71, 0x7A, 0x00, 0xB2, 0x19, 0x9C, 0x8A,
                0x18, 0xD7, 0x91, 0xB5, 0x92, 0xB7, 0xEC, 0x52,
                0xBD, 0x5A, 0xF2, 0xDB, 0x0D, 0x3B, 0x63, 0x5F,
                0x05, 0x95, 0x75, 0x3D, 0xFF, 0x7B, 0xA7, 0xC9,
                0x87, 0x2D, 0xBF, 0x7E, 0x32, 0x26, 0xDE, 0xF4,
                0x4A, 0x07, 0xCA, 0x56, 0x8D, 0x10, 0x17, 0x99,
                0x2C, 0x2B, 0x41, 0xBF, 0xE5, 0xEC, 0x35, 0x70,
                0x82, 0x4C, 0xF1, 0xF4, 0xB1, 0x59, 0x19, 0xFE,
                0xD5, 0x13, 0xFD, 0xA5, 0x62, 0x04, 0xAF, 0x20,
                0x34, 0xA2, 0xD0, 0x8F, 0xF0, 0x4C, 0x2C, 0xCA,
                0x49, 0xD1, 0x68, 0xFA, 0x03, 0xFA, 0x2F, 0xA3,
                0x2F, 0xCC, 0xD3, 0x48, 0x4C, 0x15, 0xF0, 0xA2,
                0xE5, 0x46, 0x7C, 0x76, 0xFC, 0x76, 0x0B, 0x55,
                0x09,
            },
            Exponent = new byte[]
            {
                0x01, 0x00, 0x01,
            },
            D = new byte[]
            {
                0x9E, 0x59, 0x25, 0xE2, 0xEC, 0x6C, 0xBB, 0x4A,
                0x83, 0xF3, 0xA1, 0x19, 0x37, 0xB6, 0xE2, 0x9E,
                0x8C, 0x64, 0x78, 0x65, 0x2F, 0xDC, 0xFA, 0x9D,
                0xD1, 0x78, 0x82, 0x97, 0x70, 0xE2, 0x53, 0xE2,
                0x07, 0x05, 0x6D, 0x32, 0x01, 0xC8, 0x41, 0x1C,
                0x13, 0xF5, 0xEF, 0xDA, 0xEE, 0x99, 0x08, 0x46,
                0x68, 0xAE, 0x4E, 0x2E, 0xD1, 0x6C, 0x1B, 0x9E,
                0xE4, 0xC7, 0xFD, 0x6E, 0x51, 0x73, 0x14, 0x2D,
                0xC5, 0x9F, 0xC6, 0x45, 0xA4, 0xC2, 0xE6, 0x65,
                0x5D, 0xAC, 0xE6, 0x71, 0xB3, 0x00, 0xDE, 0x0D,
                0x7B, 0xB5, 0x2D, 0x68, 0xA8, 0x60, 0x26, 0xC1,
                0x8F, 0x02, 0x47, 0x26, 0xE7, 0x71, 0xBD, 0x04,
                0x81, 0x99, 0x44, 0x2F, 0x03, 0x42, 0x70, 0x96,
                0x58, 0xEF, 0xF6, 0x4A, 0xE4, 0x2E, 0xA4, 0x17,
                0x32, 0x47, 0xEC, 0xD2, 0x4A, 0x86, 0x29, 0x9E,
                0x8B, 0x9D, 0x11, 0x52, 0x00, 0x02, 0xC8, 0xF5,
                0xF1,
            },
            P = new byte[]
            {
                0x0E, 0x15, 0x30, 0x0A, 0x9D, 0x34, 0xBA, 0x37,
                0xB6, 0xBD, 0xA8, 0x31, 0xBC, 0x67, 0x27, 0xB2,
                0xF7, 0xF6, 0xD0, 0xEF, 0xB7, 0xB3, 0x3A, 0x99,
                0xC9, 0xAF, 0x28, 0xCF, 0xD6, 0x25, 0xE2, 0x45,
                0xA5, 0x4F, 0x25, 0x1B, 0x78, 0x4C, 0x47, 0x91,
                0xAD, 0xA5, 0x85, 0xAD, 0xB7, 0x11, 0xD9, 0x30,
                0x0A, 0x3D, 0x52, 0xB4, 0x50, 0xCC, 0x30, 0x7F,
                0x55, 0xD3, 0x1E, 0x12, 0x17, 0xB9, 0xFF, 0xD7,
                0x45,
            },
            DP = new byte[]
            {
                0x0D, 0x9D, 0xB4, 0xBE, 0x7E, 0x73, 0x0D, 0x9D,
                0x72, 0xA5, 0x7B, 0x2A, 0xE3, 0x73, 0x85, 0x71,
                0xC7, 0xC8, 0x2F, 0x09, 0xA7, 0xBE, 0xB5, 0xE9,
                0x1D, 0x94, 0xAA, 0xCC, 0x10, 0xCC, 0xBE, 0x33,
                0x02, 0x7B, 0x3C, 0x70, 0x8B, 0xE6, 0x8C, 0xC8,
                0x30, 0x71, 0xBA, 0x87, 0x54, 0x5B, 0x00, 0x78,
                0x2F, 0x5E, 0x4D, 0x49, 0xA4, 0x59, 0x58, 0x86,
                0xB5, 0x6F, 0x93, 0x42, 0x81, 0x08, 0x48, 0x72,
                0x55,
            },
            Q = new byte[]
            {
                0x0D, 0x65, 0xC6, 0x0D, 0xE8, 0xB6, 0xF5, 0x4A,
                0x77, 0x56, 0xFD, 0x1C, 0xCB, 0xA7, 0x6C, 0xE4,
                0x1E, 0xF4, 0x46, 0xD0, 0x24, 0x03, 0x1E, 0xE9,
                0xC5, 0xA4, 0x09, 0x31, 0xB0, 0x73, 0x36, 0xCF,
                0xED, 0x35, 0xA8, 0xEE, 0x58, 0x0E, 0x19, 0xDB,
                0x85, 0x92, 0xCB, 0x0F, 0x26, 0x6E, 0xC6, 0x90,
                0x28, 0xEB, 0x9E, 0x98, 0xE3, 0xE8, 0x4F, 0xF1,
                0xA4, 0x59, 0xA8, 0xA2, 0x68, 0x60, 0xA6, 0x10,
                0xF5,
            },
            DQ = new byte[]
            {
                0x0C, 0xF6, 0xFB, 0xDD, 0xE1, 0xE1, 0x8B, 0x25,
                0x70, 0xAF, 0x21, 0x69, 0x88, 0x3A, 0x90, 0xC9,
                0x80, 0x9A, 0xEB, 0x1B, 0xE8, 0x7D, 0x8C, 0xA0,
                0xB4, 0xBD, 0xB4, 0x97, 0xFD, 0x24, 0xC1, 0x5A,
                0x1D, 0x36, 0xDC, 0x2F, 0x29, 0xCF, 0x1B, 0x7E,
                0xAF, 0x98, 0x0A, 0x20, 0xB3, 0x14, 0x67, 0xDA,
                0x81, 0x7E, 0xE1, 0x8F, 0x1A, 0x9D, 0x69, 0x1F,
                0x71, 0xE7, 0xC1, 0xA4, 0xC8, 0x55, 0x1E, 0xDF,
                0x31,
            },
            InverseQ = new byte[]
            {
                0x01, 0x0C, 0xE9, 0x93, 0x6E, 0x96, 0xFB, 0xAD,
                0xF8, 0x72, 0x40, 0xCC, 0x41, 0x9D, 0x01, 0x08,
                0x1B, 0xB6, 0x7C, 0x98, 0x1D, 0x44, 0x31, 0x4E,
                0x58, 0x58, 0x3A, 0xC7, 0xFE, 0x93, 0x79, 0xEA,
                0x02, 0x72, 0xE6, 0xC4, 0xC7, 0xC1, 0x46, 0x38,
                0xE1, 0xD5, 0xEC, 0xE7, 0x84, 0x0D, 0xDB, 0x15,
                0xA1, 0x2D, 0x70, 0x54, 0xA4, 0x18, 0xF8, 0x76,
                0x4F, 0xA5, 0x4C, 0xE1, 0x34, 0xEB, 0xD2, 0x63,
                0x5E,
            },
        };

        public static readonly RSAParameters RSA2048Params = new RSAParameters
        {
            Modulus = new byte[]
            {
                0xB1, 0x7C, 0xEE, 0x77, 0xB4, 0x59, 0xA4, 0x65,
                0x92, 0x8D, 0x7F, 0x55, 0x77, 0x80, 0x39, 0xBA,
                0x22, 0xBA, 0x29, 0xA5, 0xFF, 0xE5, 0x53, 0xFB,
                0x40, 0x98, 0xA8, 0x35, 0xE5, 0x2D, 0x0A, 0xDC,
                0x85, 0x17, 0x6E, 0xE4, 0xD6, 0x93, 0x82, 0x20,
                0x71, 0x8D, 0xE9, 0xDD, 0xC9, 0xD5, 0xBD, 0x30,
                0x47, 0xEE, 0x59, 0xB9, 0xE6, 0xA8, 0x79, 0x9E,
                0x47, 0x96, 0x8E, 0x63, 0xCD, 0xA6, 0x28, 0x9D,
                0x7B, 0x6D, 0x83, 0xAA, 0x68, 0xE2, 0x46, 0xD6,
                0x1C, 0x8C, 0x2B, 0xA1, 0xC4, 0x04, 0x12, 0xAE,
                0x61, 0x07, 0xAF, 0x6F, 0xE9, 0x2B, 0x48, 0x5C,
                0xCA, 0xC2, 0x0E, 0x52, 0x71, 0x21, 0x03, 0xE0,
                0x05, 0x1C, 0x07, 0xC0, 0x56, 0xFD, 0x8A, 0x61,
                0x7A, 0x95, 0x39, 0x4B, 0xAA, 0x5A, 0x9D, 0x03,
                0x6D, 0x7A, 0x51, 0x3E, 0x7A, 0xE4, 0xAB, 0xED,
                0xB4, 0x0A, 0x88, 0x80, 0x3C, 0x07, 0xED, 0x19,
                0x21, 0xA2, 0xDC, 0xD7, 0x9C, 0x3F, 0x3B, 0x19,
                0x59, 0x33, 0x39, 0x8A, 0x25, 0xDB, 0xCE, 0x8D,
                0x8E, 0x10, 0xDA, 0xB1, 0x38, 0x32, 0xCA, 0x59,
                0xA1, 0x69, 0x3C, 0x23, 0x76, 0x50, 0x37, 0xB3,
                0xBF, 0x73, 0x58, 0x77, 0x38, 0xC5, 0x16, 0x03,
                0x60, 0x36, 0x0D, 0x31, 0x8C, 0xBE, 0xA5, 0x12,
                0x2F, 0xE5, 0x16, 0xAD, 0x1C, 0x80, 0x01, 0x50,
                0xEB, 0x3C, 0x86, 0x79, 0x22, 0xD3, 0x7F, 0xD4,
                0x90, 0x85, 0xB8, 0xEB, 0xB0, 0x7D, 0xD8, 0xC8,
                0x8B, 0xBB, 0x88, 0xBE, 0xFE, 0xB8, 0xBA, 0xAD,
                0x61, 0xC7, 0xF9, 0x68, 0x20, 0xB2, 0xA9, 0x1D,
                0xB4, 0xDC, 0xB0, 0x5B, 0xC7, 0xB3, 0xF2, 0x83,
                0x35, 0x43, 0xB0, 0xAE, 0x19, 0x2B, 0xA6, 0xFA,
                0x88, 0x48, 0xB9, 0xFB, 0xB3, 0xCD, 0xF8, 0xA9,
                0x9C, 0x20, 0x6F, 0x63, 0x23, 0xE5, 0xC2, 0x85,
                0xCD, 0x75, 0x7A, 0x55, 0x04, 0xA4, 0x08, 0x99,
            },
            Exponent = new byte[]
            {
                0x01, 0x00, 0x01,
            },
            D = new byte[]
            {
                0x2C, 0x96, 0x86, 0x11, 0xEC, 0x6C, 0xD8, 0xAF,
                0xEB, 0xB1, 0x40, 0x5B, 0xE8, 0x39, 0x7E, 0x47,
                0x14, 0x92, 0x50, 0x04, 0x33, 0xD5, 0x18, 0xD3,
                0xF5, 0xD6, 0x63, 0xEB, 0xA6, 0x37, 0x3A, 0x93,
                0x4B, 0x9C, 0x27, 0x6F, 0xB5, 0xB8, 0x38, 0xE8,
                0x8D, 0x9E, 0x69, 0x32, 0x1E, 0x92, 0x63, 0x84,
                0xCD, 0x8D, 0x43, 0x5D, 0x40, 0x64, 0xF2, 0xA8,
                0xA0, 0xB3, 0x61, 0xF2, 0x10, 0xA7, 0xBD, 0x6C,
                0x52, 0xA5, 0xA0, 0x7E, 0x1E, 0xFB, 0x39, 0x70,
                0x70, 0x9B, 0x86, 0x1A, 0x8D, 0x73, 0xB8, 0x7D,
                0xB6, 0x42, 0x88, 0x00, 0x45, 0x43, 0x6A, 0x5A,
                0x65, 0x55, 0x7A, 0xE3, 0x9B, 0x28, 0x00, 0x21,
                0x37, 0x27, 0x63, 0x8B, 0x1E, 0x4F, 0x73, 0x84,
                0x29, 0x97, 0x73, 0x5D, 0x5E, 0xDE, 0x84, 0xB3,
                0x67, 0xBD, 0x62, 0xCB, 0x9F, 0x73, 0xF2, 0xFD,
                0x34, 0x4D, 0xB1, 0x1D, 0x05, 0xF7, 0xB7, 0xC8,
                0x3C, 0x69, 0xF6, 0x2C, 0x5B, 0x78, 0xC6, 0xB0,
                0x55, 0xBF, 0x8B, 0x58, 0xF1, 0xC2, 0x9F, 0x19,
                0x4A, 0x17, 0x53, 0x60, 0xDA, 0x52, 0x6F, 0x2E,
                0xDE, 0x0F, 0x64, 0x9D, 0x4F, 0xB8, 0xAF, 0x50,
                0x2B, 0x8F, 0x99, 0x1E, 0x15, 0x78, 0x32, 0xD7,
                0x37, 0x06, 0xC3, 0x08, 0x5D, 0x1C, 0x04, 0xB1,
                0x6D, 0x46, 0x84, 0xC9, 0xE8, 0xF6, 0x0E, 0x8E,
                0x18, 0xEC, 0x4B, 0x84, 0xC5, 0xF3, 0x65, 0x6B,
                0x9B, 0x35, 0x0A, 0xDA, 0x7B, 0xC7, 0xAB, 0x6D,
                0x58, 0xFA, 0x17, 0x76, 0xC2, 0x67, 0xF1, 0xBA,
                0x9B, 0xDD, 0x1F, 0xB4, 0x8A, 0x56, 0x06, 0x67,
                0x36, 0x39, 0xE9, 0x1E, 0x82, 0x8A, 0xAA, 0x28,
                0x7A, 0xA7, 0xDA, 0x18, 0x01, 0xD5, 0x1C, 0x19,
                0x0C, 0x50, 0xAB, 0x09, 0x35, 0xAF, 0xCE, 0x26,
                0x00, 0xD4, 0xD0, 0x61, 0xCC, 0x2A, 0x0E, 0x3F,
                0xCA, 0xC3, 0xDB, 0x42, 0x16, 0x9B, 0xFF, 0x41,
            },
            P = new byte[]
            {
                0xBA, 0x61, 0xFD, 0x65, 0x6D, 0x86, 0xBA, 0xDD,
                0x77, 0xB7, 0x11, 0x9E, 0xC8, 0xC9, 0x7D, 0xB8,
                0x6A, 0x23, 0x03, 0x3D, 0xE8, 0xC9, 0xB7, 0xD6,
                0xDE, 0xF8, 0x15, 0x7C, 0xF7, 0x0B, 0xB1, 0x1D,
                0xC8, 0x83, 0xEC, 0x76, 0x57, 0x9F, 0x7F, 0x47,
                0x5E, 0xD5, 0x57, 0x34, 0xD9, 0x16, 0xBD, 0xDC,
                0xDD, 0xA0, 0xB2, 0xE0, 0x18, 0xD1, 0x70, 0x65,
                0x4E, 0x2B, 0x75, 0x15, 0x77, 0xC4, 0x06, 0x3E,
                0x80, 0x3E, 0xC2, 0xB6, 0xA7, 0x40, 0x4A, 0x53,
                0x61, 0x30, 0x35, 0x46, 0x9F, 0xD4, 0xFC, 0x7D,
                0x80, 0xC9, 0x0E, 0xE1, 0x94, 0x9B, 0xC7, 0xFA,
                0xBD, 0x46, 0x9B, 0x4C, 0x12, 0x90, 0x7D, 0x9B,
                0xF5, 0xE6, 0x82, 0x46, 0xB9, 0xA6, 0xA9, 0xF6,
                0x35, 0xBC, 0x07, 0x9E, 0xC4, 0x78, 0x74, 0x6E,
                0x44, 0xAC, 0xBF, 0x5C, 0x59, 0x33, 0xDA, 0x59,
                0xDE, 0x9B, 0xED, 0x3D, 0x39, 0x1B, 0x36, 0x23,
            },
            DP = new byte[]
            {
                0x75, 0xD7, 0x56, 0xBB, 0x36, 0x50, 0xA4, 0xFD,
                0x39, 0x9F, 0xC9, 0xC8, 0x36, 0xF3, 0x0E, 0x45,
                0xF6, 0xF5, 0x44, 0x2B, 0x74, 0x6F, 0x75, 0x88,
                0xA9, 0x58, 0xF9, 0x5D, 0x15, 0x65, 0x93, 0x0A,
                0x5D, 0xA8, 0xEB, 0x6C, 0xB7, 0x61, 0xE4, 0xBB,
                0x5F, 0x3E, 0x4B, 0xF0, 0xE2, 0x00, 0xFA, 0xF2,
                0x16, 0x3E, 0x70, 0x5A, 0x37, 0xD6, 0xD3, 0xD5,
                0x79, 0x63, 0x08, 0x98, 0x16, 0x2D, 0x1E, 0x35,
                0x8E, 0x28, 0x20, 0x3C, 0x13, 0xEB, 0x16, 0x13,
                0x39, 0xB3, 0x9D, 0x3B, 0x95, 0xFA, 0xB7, 0xD9,
                0x31, 0xFF, 0xED, 0x24, 0xBB, 0x2C, 0xF3, 0x77,
                0x99, 0x0C, 0x77, 0x4B, 0xD5, 0xC0, 0xFD, 0x6A,
                0x0A, 0x43, 0x3F, 0xC3, 0x2F, 0xC6, 0x2C, 0x57,
                0xBB, 0x09, 0xB3, 0x57, 0xB2, 0xA8, 0xE6, 0x14,
                0x81, 0xDF, 0x26, 0xEE, 0x60, 0x87, 0xE4, 0x5A,
                0x45, 0xE1, 0x18, 0x52, 0x49, 0x34, 0xE7, 0x39,
            },
            Q = new byte[]
            {
                0xF3, 0xC8, 0x6B, 0xA9, 0xA2, 0xD2, 0xD6, 0xAB,
                0x64, 0x59, 0x0F, 0x68, 0x0C, 0xC8, 0x69, 0xC6,
                0x51, 0xB2, 0x2C, 0x28, 0xB5, 0xDB, 0xDD, 0x00,
                0x70, 0x5E, 0xB3, 0x04, 0x38, 0xC4, 0xAA, 0x45,
                0xEF, 0x79, 0xAC, 0x79, 0x8F, 0xF1, 0xF6, 0x07,
                0x3B, 0xDF, 0xAF, 0x08, 0xE3, 0xDA, 0x21, 0x37,
                0x08, 0x41, 0x1F, 0xA0, 0x84, 0x4D, 0xEB, 0xA9,
                0x91, 0xB5, 0xB1, 0xD6, 0xD5, 0x3A, 0x16, 0xD5,
                0x7E, 0x7A, 0x9D, 0x65, 0xF1, 0x92, 0x87, 0x69,
                0xE7, 0x01, 0xFD, 0x60, 0x95, 0x25, 0x1B, 0xF0,
                0x79, 0xE0, 0xE1, 0xCA, 0xCE, 0x99, 0x7A, 0xED,
                0xC7, 0x8D, 0xA2, 0x3D, 0x8D, 0x30, 0xB1, 0x8E,
                0xDA, 0x99, 0xCF, 0x9E, 0x6B, 0x02, 0x28, 0xD4,
                0xEC, 0x25, 0xD5, 0x89, 0x74, 0xCA, 0xA2, 0x7E,
                0x47, 0xDB, 0xE8, 0xE5, 0x42, 0x6C, 0xD4, 0xDC,
                0xAD, 0xDC, 0x31, 0xB2, 0x42, 0xFB, 0x2C, 0x13,
            },
            DQ = new byte[]
            {
                0x2F, 0x5F, 0x0F, 0xC4, 0xBB, 0xF6, 0x1A, 0x6E,
                0xDD, 0xA6, 0x0C, 0xBF, 0x5C, 0x54, 0x89, 0x71,
                0x57, 0x28, 0xB7, 0x3A, 0x05, 0xF4, 0xBE, 0x62,
                0x3A, 0x73, 0xBC, 0x77, 0xA2, 0x8C, 0x5C, 0xC6,
                0x10, 0x3D, 0xE5, 0x8D, 0x0D, 0xB2, 0xA7, 0xEB,
                0x49, 0xF0, 0x32, 0x74, 0x18, 0xCA, 0xA7, 0x4F,
                0xA9, 0x53, 0xF6, 0x50, 0x5B, 0xC5, 0x44, 0x79,
                0x03, 0xEE, 0x79, 0xAB, 0x54, 0x6D, 0xE0, 0x48,
                0x06, 0x36, 0xCF, 0x65, 0x22, 0xE7, 0x25, 0x57,
                0x27, 0xE3, 0x94, 0x17, 0xF3, 0x83, 0x6D, 0x85,
                0x72, 0x39, 0x87, 0xC6, 0xC0, 0x14, 0xC4, 0xF5,
                0x75, 0xA4, 0x89, 0x15, 0x4A, 0xDD, 0x5E, 0x73,
                0x72, 0xF9, 0x16, 0x86, 0x23, 0x27, 0x1D, 0x46,
                0x1A, 0xC9, 0x53, 0x50, 0x4D, 0x98, 0x9E, 0xB0,
                0xC9, 0x47, 0xEB, 0x5E, 0xB9, 0x64, 0xAA, 0x8C,
                0x63, 0x60, 0x79, 0x6B, 0xB9, 0x66, 0x53, 0x6F,
            },
            InverseQ = new byte[]
            {
                0x0B, 0x16, 0x6B, 0x2E, 0x87, 0xCF, 0x1C, 0x2C,
                0x9C, 0x8A, 0x1C, 0x92, 0xD4, 0xD1, 0x1E, 0xA4,
                0x87, 0x25, 0x1C, 0xA1, 0x29, 0x7A, 0xE4, 0xD6,
                0x65, 0x69, 0x6E, 0x62, 0xBA, 0xEA, 0x41, 0x69,
                0x89, 0xBD, 0x78, 0x5A, 0xA1, 0x2C, 0x51, 0x66,
                0xA3, 0x3D, 0x1E, 0x51, 0xA6, 0x65, 0x61, 0xA6,
                0x6C, 0x2F, 0xAA, 0xDB, 0x5D, 0x6A, 0x94, 0x4E,
                0x65, 0x4A, 0x30, 0x42, 0xE1, 0x7A, 0xA8, 0x5B,
                0x2D, 0x41, 0x05, 0x82, 0x5F, 0x6D, 0xDD, 0x9D,
                0x9A, 0x6A, 0x6E, 0x6E, 0x7C, 0x1B, 0xB3, 0x0B,
                0x93, 0x58, 0x10, 0x1C, 0xEF, 0x91, 0xB6, 0x06,
                0xBB, 0xE5, 0xAD, 0x15, 0x74, 0x69, 0xA4, 0x64,
                0xFF, 0x6B, 0x88, 0x1E, 0xA1, 0x3C, 0xE3, 0xE1,
                0x24, 0x53, 0x4D, 0xC8, 0xD1, 0x0B, 0xF7, 0x7D,
                0x1B, 0x68, 0xEA, 0xB5, 0x34, 0xE4, 0xFB, 0x1C,
                0x6C, 0xCA, 0x7F, 0x2C, 0x71, 0x19, 0x9D, 0xBC,
            },
        };

        public static readonly RSAParameters RSA16384Params = new RSAParameters
        {
            Modulus = new byte[]
            {
                0x9B, 0x2C, 0x70, 0x5F, 0xA9, 0x10, 0x37, 0x1F,
                0x8B, 0x48, 0xC6, 0xA8, 0xD5, 0x2B, 0x42, 0xD6,
                0x9E, 0x6B, 0x28, 0x21, 0x30, 0x70, 0x18, 0xF3,
                0x23, 0x5D, 0xFA, 0x02, 0x7D, 0xC1, 0xFC, 0x18,
                0xED, 0x86, 0x07, 0xB3, 0x00, 0xEB, 0xAE, 0x27,
                0xE7, 0xC0, 0x7C, 0x55, 0x64, 0x8F, 0xB5, 0x47,
                0xE1, 0x98, 0x08, 0xF8, 0x60, 0x85, 0xC7, 0x14,
                0x0B, 0x2B, 0x31, 0x62, 0x98, 0x5A, 0xF5, 0x16,
                0xFA, 0xA7, 0x9C, 0xA8, 0x65, 0xBA, 0x0A, 0xEC,
                0xB8, 0x9F, 0x30, 0x98, 0x9B, 0x55, 0x51, 0x1C,
                0x4E, 0xFA, 0x51, 0x0A, 0x79, 0x92, 0x02, 0x3E,
                0x6D, 0x9C, 0xCF, 0x8D, 0xE7, 0x7C, 0xA0, 0x32,
                0x82, 0xC1, 0x9C, 0x7D, 0x2E, 0xF3, 0x8E, 0x88,
                0x74, 0xB7, 0x0C, 0x88, 0x88, 0x00, 0xBA, 0xFB,
                0xEF, 0x0A, 0xFF, 0x4A, 0xDC, 0xEC, 0xC6, 0xE5,
                0xB8, 0x80, 0x6A, 0xC0, 0x60, 0xF7, 0x8D, 0x8B,
                0x23, 0x07, 0x42, 0xC0, 0x67, 0x5F, 0x20, 0x6F,
                0x72, 0x98, 0xCE, 0xE5, 0x39, 0x97, 0x0B, 0x76,
                0xB8, 0x66, 0x56, 0x22, 0x09, 0xBD, 0x44, 0x67,
                0x35, 0x2F, 0x8A, 0xF8, 0x45, 0xD1, 0x36, 0x34,
                0x27, 0xD4, 0x78, 0x4D, 0x6E, 0x9E, 0x5D, 0x95,
                0x47, 0xBC, 0x81, 0x60, 0x57, 0x1A, 0xB8, 0xC9,
                0x5F, 0x18, 0xBE, 0x5B, 0x90, 0x13, 0x15, 0x84,
                0x49, 0x16, 0x6E, 0xD6, 0xDD, 0x9D, 0x47, 0xEC,
                0x22, 0x8A, 0xD6, 0x7A, 0x97, 0x44, 0x6B, 0x81,
                0xA4, 0xA4, 0xF5, 0xE0, 0xEA, 0x4A, 0x54, 0x3B,
                0x88, 0x81, 0xB8, 0x5D, 0xA4, 0x8A, 0x05, 0x29,
                0xDB, 0x5A, 0xE4, 0x18, 0x8A, 0xF3, 0x07, 0x28,
                0x32, 0xD6, 0x91, 0x50, 0xFE, 0x14, 0x5E, 0x1A,
                0xE0, 0x9B, 0xFD, 0x25, 0x77, 0xAC, 0x5E, 0xD5,
                0xF2, 0xA7, 0xE9, 0x9F, 0xDF, 0x2D, 0x86, 0xA7,
                0xA2, 0x88, 0xF7, 0xF3, 0x89, 0x9E, 0x8A, 0xA6,
                0xB9, 0x57, 0x36, 0xD1, 0x63, 0xFE, 0xE5, 0x40,
                0x12, 0xB5, 0x87, 0x9F, 0x73, 0xF6, 0x76, 0x26,
                0x32, 0xA7, 0x71, 0xD0, 0xC7, 0xC0, 0x46, 0x9A,
                0xF2, 0x51, 0x31, 0xB1, 0xDD, 0xD1, 0x60, 0xEB,
                0x89, 0x02, 0x62, 0x58, 0x44, 0x98, 0xA7, 0x65,
                0x53, 0x07, 0x73, 0x75, 0x75, 0xC0, 0xD3, 0x8F,
                0x91, 0x34, 0x72, 0x47, 0x5F, 0xF5, 0x6B, 0x7E,
                0x25, 0x1E, 0xE0, 0x58, 0xC5, 0xB9, 0x6B, 0x17,
                0xBB, 0xCB, 0x5C, 0x4C, 0xB3, 0x01, 0xC9, 0x59,
                0xA2, 0x3B, 0xEA, 0x39, 0xC1, 0x4C, 0x7A, 0x21,
                0xAA, 0xDB, 0x7E, 0xD9, 0x3E, 0x97, 0x16, 0x5B,
                0x93, 0xA2, 0x76, 0x26, 0x39, 0xD4, 0x3C, 0x0D,
                0xB8, 0x07, 0x05, 0x48, 0x2A, 0x1B, 0xA1, 0xFF,
                0x1C, 0x9B, 0xA0, 0x58, 0x12, 0x26, 0xDF, 0x5F,
                0x78, 0x40, 0x35, 0x61, 0xE7, 0x90, 0x92, 0xB0,
                0x92, 0xC8, 0x9B, 0xC2, 0xF1, 0xC1, 0xC0, 0x60,
                0x79, 0x47, 0x3F, 0x1F, 0xC6, 0x36, 0x15, 0x5A,
                0xF7, 0x5A, 0xD1, 0x2C, 0x7A, 0x97, 0x34, 0xAD,
                0xBD, 0x63, 0x34, 0x32, 0xE4, 0x7F, 0xCB, 0x18,
                0x0B, 0xEA, 0x3A, 0x8E, 0xB0, 0x8C, 0x7A, 0x43,
                0xFD, 0x60, 0xF1, 0x76, 0xB4, 0xF0, 0x31, 0x7F,
                0xF7, 0xA8, 0x77, 0x17, 0xFF, 0xE7, 0xDB, 0x01,
                0x88, 0x2D, 0xB0, 0x5C, 0x53, 0x03, 0x69, 0xF9,
                0xD6, 0xA7, 0x59, 0x71, 0x5E, 0x21, 0x9C, 0x55,
                0x5B, 0xBD, 0x74, 0x4D, 0x8B, 0x7A, 0x09, 0x42,
                0x82, 0x73, 0x14, 0xA6, 0xCF, 0x50, 0x94, 0x90,
                0xF4, 0x69, 0x76, 0x89, 0x1D, 0x88, 0xB6, 0x54,
                0x34, 0x37, 0xCC, 0x57, 0x4C, 0xBE, 0x7F, 0x00,
                0x8A, 0x69, 0x6A, 0x2A, 0xD6, 0x46, 0x74, 0x67,
                0xFD, 0xDD, 0x86, 0x90, 0x83, 0xE6, 0x2A, 0xD3,
                0xB8, 0x5C, 0xE9, 0xFF, 0x15, 0x8B, 0x7A, 0x35,
                0x2D, 0x22, 0xBF, 0xC2, 0x9F, 0x41, 0x6C, 0xD6,
                0x6C, 0xDC, 0x98, 0x41, 0xBF, 0x83, 0x83, 0x20,
                0x03, 0x1F, 0x32, 0xC0, 0x84, 0xA2, 0x4F, 0x14,
                0xFA, 0xF1, 0x61, 0x1D, 0x20, 0xA6, 0xA0, 0xDC,
                0x73, 0x4E, 0x5F, 0x08, 0xF6, 0x9C, 0xFC, 0x28,
                0x92, 0xD1, 0xC8, 0xB4, 0x17, 0xC1, 0xFA, 0x49,
                0xCA, 0x57, 0xD6, 0x63, 0x9B, 0xC0, 0x6E, 0x90,
                0x20, 0x57, 0xCC, 0x1D, 0xCA, 0x37, 0xCA, 0xAE,
                0x8D, 0x90, 0x0D, 0x73, 0x48, 0x60, 0x2A, 0xD5,
                0xA5, 0x09, 0x3D, 0x28, 0x50, 0x8C, 0xC2, 0xD7,
                0xA4, 0x68, 0x0E, 0xFA, 0x2C, 0xA4, 0x50, 0x6A,
                0x2F, 0x79, 0x4D, 0x9B, 0x98, 0x09, 0xBF, 0xB2,
                0x43, 0x4D, 0x32, 0xBE, 0x4D, 0x7B, 0x9F, 0x32,
                0x0F, 0x28, 0x91, 0xEC, 0xBF, 0xF8, 0xC7, 0x2E,
                0xFC, 0x71, 0xA7, 0xD6, 0x33, 0xD4, 0xA7, 0x1B,
                0xEE, 0xB6, 0x56, 0x91, 0x3F, 0x7A, 0xFD, 0xC8,
                0x83, 0x7C, 0x4D, 0x06, 0xF1, 0xF7, 0x35, 0x92,
                0x75, 0x91, 0x43, 0xC4, 0xA6, 0xE6, 0xB5, 0x2C,
                0x73, 0xFA, 0xC7, 0x6E, 0x09, 0x1C, 0xCF, 0x30,
                0x1B, 0xD0, 0x76, 0xB7, 0x45, 0x63, 0xAD, 0xDE,
                0xF1, 0xBF, 0x27, 0x5D, 0x13, 0xD5, 0x82, 0x6B,
                0xD6, 0x02, 0x0E, 0x24, 0x14, 0xA5, 0xA4, 0xD9,
                0x8F, 0x6C, 0x48, 0x49, 0x70, 0x1C, 0xC5, 0x93,
                0xCF, 0x17, 0x3A, 0xAF, 0x43, 0x0B, 0x7C, 0x5C,
                0xD2, 0x3D, 0xD7, 0x93, 0x6F, 0x10, 0xEF, 0x70,
                0x25, 0x70, 0x4A, 0xB4, 0x18, 0xD7, 0x3E, 0x16,
                0xB8, 0xE4, 0x38, 0x9F, 0x9A, 0xCF, 0xA3, 0xDD,
                0x70, 0xAA, 0xA4, 0x0D, 0x02, 0x7C, 0xEB, 0x66,
                0xAB, 0x07, 0x8B, 0xBE, 0xE2, 0xFD, 0x70, 0xE2,
                0xAA, 0xF9, 0xE3, 0xAF, 0x31, 0xF2, 0x57, 0xC0,
                0x97, 0xE7, 0xEF, 0xD8, 0x9E, 0xF6, 0x72, 0x1B,
                0x2B, 0xDF, 0x53, 0x96, 0xA2, 0x74, 0x2C, 0x54,
                0x23, 0xA6, 0x48, 0x1B, 0x6F, 0xDF, 0x8A, 0xFD,
                0x2E, 0xEE, 0x95, 0x53, 0xA9, 0xA4, 0xEA, 0x26,
                0xD4, 0x93, 0x4B, 0xFA, 0xCE, 0xB5, 0x6F, 0xF0,
                0x64, 0xA5, 0x1E, 0xE4, 0x2C, 0x15, 0x03, 0xBB,
                0x46, 0x2E, 0x8F, 0x99, 0x7D, 0xDD, 0xCB, 0x02,
                0x89, 0x96, 0x5A, 0x53, 0x08, 0x54, 0x25, 0x72,
                0xA9, 0x5E, 0x10, 0xF9, 0x88, 0xCB, 0x30, 0xB7,
                0x1F, 0x97, 0xD5, 0xD5, 0x15, 0x73, 0x23, 0x0F,
                0x06, 0xD3, 0x5E, 0xCF, 0x25, 0xBD, 0x5E, 0xAD,
                0x73, 0xF3, 0x92, 0x62, 0xD0, 0x14, 0xFB, 0x26,
                0x4D, 0x21, 0x22, 0x24, 0xCB, 0xE8, 0xAD, 0x5C,
                0xFB, 0x99, 0x58, 0x10, 0x69, 0x3D, 0x31, 0x65,
                0xC2, 0x3A, 0xAB, 0x64, 0x7A, 0x8A, 0x79, 0x2B,
                0xF7, 0x15, 0x35, 0x88, 0x2E, 0xC4, 0xAF, 0x0D,
                0xE7, 0x67, 0xD2, 0x76, 0x3B, 0x4E, 0x83, 0xA1,
                0x98, 0xF7, 0x47, 0x0C, 0xBE, 0xDF, 0x27, 0xC2,
                0x7D, 0xB6, 0xD9, 0x93, 0xEC, 0x92, 0xFE, 0x63,
                0xC8, 0x8A, 0x9A, 0xC7, 0xC8, 0x08, 0x0D, 0x5B,
                0x97, 0xA3, 0x59, 0x2F, 0x6A, 0x24, 0x1B, 0x64,
                0xBA, 0xE6, 0x1C, 0x33, 0xE1, 0xAE, 0x49, 0x79,
                0x84, 0xD4, 0x31, 0xBC, 0x03, 0xFE, 0x17, 0x57,
                0xEC, 0x0C, 0x8C, 0x4C, 0xAA, 0x75, 0x1A, 0x7E,
                0x00, 0xE7, 0x77, 0x38, 0x40, 0xF7, 0xF2, 0xDE,
                0x0C, 0x3B, 0x81, 0x9D, 0xA7, 0x2B, 0x7B, 0xE3,
                0x63, 0x77, 0xA2, 0x51, 0x92, 0x74, 0xD2, 0x6A,
                0x22, 0xE4, 0x63, 0x19, 0x41, 0x9B, 0xD7, 0x59,
                0x79, 0x58, 0x8D, 0xA2, 0x18, 0xB2, 0x8E, 0xD8,
                0x68, 0x16, 0x4D, 0xA9, 0xE7, 0x8C, 0x15, 0x0E,
                0xE7, 0xD9, 0xB7, 0x01, 0xF6, 0xFF, 0x45, 0x5F,
                0xFF, 0xDF, 0x23, 0x2C, 0xAC, 0xF1, 0x4B, 0xF7,
                0x18, 0x34, 0x99, 0x43, 0x69, 0x08, 0x1B, 0x8E,
                0x69, 0x66, 0x07, 0x76, 0x5F, 0xBC, 0x2C, 0x97,
                0x2E, 0xFC, 0x0A, 0x32, 0x60, 0xF4, 0x53, 0x25,
                0x67, 0x7A, 0xDF, 0x4D, 0x0D, 0xE3, 0x1D, 0xEC,
                0x34, 0x1A, 0xB0, 0x7B, 0x2D, 0xF1, 0x22, 0x33,
                0xE2, 0xB4, 0x7B, 0x11, 0x98, 0xD9, 0xBE, 0x83,
                0x21, 0x7A, 0x25, 0x32, 0x1E, 0x6C, 0x55, 0xA7,
                0xD0, 0xAA, 0xE7, 0xBF, 0x60, 0xE6, 0x23, 0x9A,
                0x96, 0x60, 0x80, 0xC8, 0xE2, 0x78, 0x98, 0x13,
                0xF5, 0xBC, 0xAE, 0x84, 0x6D, 0xE5, 0x11, 0x29,
                0xF5, 0x4E, 0x87, 0xE9, 0x55, 0xCF, 0x05, 0x11,
                0x6B, 0xE1, 0xD6, 0x81, 0xD8, 0x9C, 0xFD, 0x85,
                0xCE, 0x57, 0x7E, 0xDF, 0xE2, 0xF7, 0xF6, 0x44,
                0xD8, 0xDD, 0x4F, 0xFE, 0x4A, 0x11, 0x17, 0xDD,
                0x08, 0x12, 0x54, 0x34, 0xDC, 0xAC, 0xE0, 0xA3,
                0xBE, 0xB4, 0x8F, 0x9B, 0xFF, 0x5A, 0xD7, 0xC2,
                0x5E, 0xC6, 0x34, 0x25, 0x6A, 0x1C, 0xDC, 0x90,
                0x90, 0x79, 0x4C, 0x0F, 0xD3, 0xCF, 0x71, 0x08,
                0xB1, 0x45, 0xA2, 0xBC, 0x7D, 0x32, 0xF9, 0xC0,
                0x6A, 0xBC, 0x6D, 0xC1, 0x98, 0xDD, 0xA0, 0x93,
                0xAC, 0x7D, 0x74, 0x95, 0x8B, 0x75, 0xE3, 0xF4,
                0x64, 0xA1, 0x80, 0xC1, 0xF1, 0xC8, 0xAE, 0x1E,
                0xE4, 0x4C, 0x3A, 0x6C, 0xF3, 0x17, 0x41, 0x9E,
                0x35, 0x58, 0xEA, 0x35, 0x1E, 0x97, 0x2A, 0x8D,
                0x94, 0x9F, 0x6C, 0x62, 0xF7, 0xF7, 0xF6, 0x52,
                0x39, 0xA5, 0x10, 0xF0, 0x5C, 0x74, 0x7D, 0xBF,
                0x8D, 0x29, 0xB8, 0x91, 0xC3, 0x4E, 0xB7, 0x31,
                0xB5, 0x12, 0x1A, 0xA4, 0x39, 0xE2, 0xE8, 0x87,
                0x89, 0xFF, 0xA3, 0x38, 0xBB, 0x71, 0x88, 0x21,
                0x2A, 0x34, 0x91, 0x18, 0x02, 0x5E, 0xC0, 0x2C,
                0xDD, 0x47, 0x87, 0xDF, 0x36, 0x12, 0xCF, 0x96,
                0x1C, 0xD1, 0x98, 0x2A, 0x9B, 0x8E, 0x28, 0xF9,
                0xDF, 0x81, 0xAF, 0x34, 0x00, 0xB5, 0xB2, 0x9B,
                0xE5, 0x8C, 0xA3, 0x82, 0xA9, 0xBE, 0x95, 0xE7,
                0x17, 0xEB, 0xEA, 0x57, 0x2E, 0xA7, 0x71, 0x0B,
                0xBF, 0xFB, 0x8D, 0xC5, 0x74, 0x9B, 0x98, 0xC0,
                0x21, 0x70, 0x54, 0xB1, 0xEC, 0xFF, 0xA0, 0x80,
                0xA2, 0x7D, 0x29, 0x09, 0x3F, 0x63, 0x06, 0xA1,
                0xDA, 0x9F, 0xAA, 0x17, 0x9D, 0x73, 0x65, 0x04,
                0xD9, 0x00, 0x07, 0xB8, 0xFB, 0x8D, 0x08, 0xB5,
                0x04, 0xF4, 0x3D, 0x11, 0xAA, 0x32, 0xD3, 0xDD,
                0xF6, 0xD2, 0xEC, 0x9B, 0x53, 0x61, 0xD0, 0xFC,
                0x6A, 0x35, 0xA1, 0xEA, 0x78, 0xE5, 0xAA, 0xDE,
                0x51, 0x3D, 0x54, 0xE6, 0x36, 0xC3, 0x7D, 0xA3,
                0x72, 0x78, 0x6C, 0x88, 0x0C, 0x94, 0x9D, 0x31,
                0xE5, 0x72, 0xF3, 0x89, 0xE7, 0x38, 0xBF, 0xE9,
                0x67, 0xF7, 0xAB, 0xEC, 0xBC, 0x51, 0xF8, 0x37,
                0x5E, 0x8B, 0x4F, 0x54, 0xD2, 0xA8, 0xC6, 0x1D,
                0x3E, 0x01, 0x8C, 0x36, 0x90, 0x9C, 0x83, 0xA3,
                0x87, 0x28, 0x1C, 0x40, 0x67, 0x19, 0x0F, 0x03,
                0xD9, 0x17, 0x37, 0xC0, 0xBD, 0x1D, 0x06, 0xAD,
                0x61, 0x97, 0xCC, 0x01, 0x6E, 0x3A, 0xB4, 0xD1,
                0xDF, 0x43, 0x57, 0x41, 0xE4, 0x9F, 0x49, 0xAF,
                0x84, 0x9E, 0xC9, 0xB2, 0x74, 0x67, 0xFC, 0x16,
                0xD4, 0x4F, 0x18, 0x5E, 0x87, 0x16, 0x7A, 0x36,
                0x00, 0x93, 0x62, 0x81, 0xC0, 0xE9, 0xF0, 0xCE,
                0xAD, 0x88, 0x59, 0xE5, 0x77, 0x58, 0xC3, 0x16,
                0xAD, 0x02, 0x0D, 0xA0, 0x1C, 0xD3, 0xB7, 0x2C,
                0x88, 0xC7, 0xCC, 0x83, 0xC5, 0x99, 0x16, 0x58,
                0x7C, 0x24, 0x46, 0xF6, 0x67, 0x27, 0xBA, 0xF7,
                0x10, 0x76, 0xD6, 0x64, 0x27, 0x35, 0x15, 0x30,
                0xD6, 0xF2, 0x36, 0x4A, 0xCA, 0x42, 0x50, 0xD8,
                0x7A, 0xC7, 0xAB, 0x87, 0xB6, 0x6A, 0x98, 0x36,
                0x21, 0x38, 0x6B, 0x88, 0x53, 0xE8, 0xDE, 0x30,
                0x77, 0x54, 0x94, 0xB8, 0xFD, 0xAD, 0xD9, 0x11,
                0xDA, 0x25, 0xE9, 0x6F, 0x65, 0x6B, 0xE0, 0x3D,
                0x95, 0xAB, 0x35, 0x0F, 0x9E, 0xC6, 0xB9, 0xA2,
                0x14, 0x8B, 0x6C, 0x8B, 0xC1, 0xA1, 0x39, 0xCC,
                0x46, 0xF4, 0x82, 0xA3, 0xD9, 0xEF, 0x19, 0xDB,
                0xF3, 0x2D, 0x23, 0xD7, 0xB8, 0xA6, 0xFC, 0x3D,
                0xF2, 0xEB, 0x9A, 0x33, 0x8C, 0xDD, 0x8A, 0x76,
                0x59, 0xD3, 0x2E, 0x7A, 0x52, 0xAC, 0xD9, 0x3F,
                0xCB, 0xFB, 0xA5, 0x37, 0xFD, 0xA4, 0xB3, 0xFC,
                0xE9, 0x0F, 0xD6, 0x04, 0xF0, 0x75, 0x5C, 0xB3,
                0x63, 0x43, 0x82, 0x07, 0x6F, 0x8F, 0x7C, 0x86,
                0x92, 0x36, 0x73, 0x7A, 0x9A, 0x66, 0xF9, 0x4A,
                0xF3, 0x26, 0xB2, 0x7E, 0x18, 0xBB, 0x9A, 0x24,
                0xF3, 0xCF, 0x1F, 0x73, 0xBF, 0x98, 0x9E, 0x36,
                0xFC, 0x18, 0xB5, 0x5E, 0x6B, 0x1D, 0x06, 0xDE,
                0x70, 0x7A, 0x81, 0xB3, 0x9D, 0x88, 0x11, 0x8C,
                0x44, 0xA2, 0x12, 0x84, 0x1B, 0x08, 0xCA, 0xE8,
                0x5B, 0xA3, 0xA0, 0x1F, 0x70, 0xD9, 0xFF, 0x46,
                0x75, 0x8F, 0x46, 0xF8, 0x3C, 0xB9, 0x68, 0x11,
                0x5D, 0xC0, 0xCD, 0x06, 0x8C, 0x70, 0x3B, 0x7F,
                0xE3, 0xB9, 0xD3, 0xC3, 0xE1, 0x9F, 0xF8, 0x1C,
                0x35, 0xAD, 0xFE, 0x05, 0x7F, 0x0D, 0x12, 0x8B,
                0x2C, 0x06, 0xAE, 0xE1, 0x63, 0x3A, 0x05, 0x45,
                0xA9, 0x26, 0x06, 0x6E, 0xAA, 0x9B, 0x92, 0x42,
                0xE4, 0xD3, 0x51, 0x65, 0x77, 0xC1, 0x42, 0x9F,
                0xD2, 0x92, 0x2A, 0xE1, 0xE2, 0x57, 0x03, 0xAA,
                0x02, 0xF4, 0x24, 0xB8, 0x66, 0x18, 0x6C, 0xBF,
                0x2E, 0x8A, 0xAC, 0x69, 0x32, 0x68, 0x6D, 0xB1,
                0x3D, 0x8C, 0x7F, 0x4F, 0x5D, 0x96, 0xA9, 0x78,
                0xC5, 0xEC, 0x4A, 0x30, 0x16, 0x48, 0x2D, 0x79,
                0x0F, 0x5A, 0xD6, 0xCF, 0x38, 0xAF, 0x80, 0x4E,
                0x28, 0x50, 0xA9, 0x16, 0x21, 0xA6, 0xB6, 0xE1,
                0x3B, 0x27, 0xDB, 0x61, 0x3E, 0xE3, 0x08, 0x7D,
                0x8D, 0x2E, 0xF1, 0x39, 0xF2, 0x75, 0xFF, 0x25,
                0xE7, 0x08, 0xDF, 0x2C, 0xBC, 0xE4, 0x49, 0x70,
                0x4B, 0xFC, 0x4D, 0xEF, 0x72, 0x32, 0x45, 0xC4,
                0x89, 0xFD, 0x7B, 0x56, 0x13, 0x49, 0x86, 0x8E,
                0x3E, 0xA8, 0xB0, 0xF1, 0xB3, 0xB5, 0x5B, 0x51,
                0x82, 0x32, 0x28, 0xD2, 0x65, 0xCF, 0x72, 0xAB,
                0x35, 0x03, 0x01, 0x00, 0x3A, 0x5E, 0xCB, 0x5D,
                0x1F, 0x9E, 0x1C, 0xDA, 0xD0, 0xD2, 0x52, 0x4C,
                0x2E, 0xD1, 0xA1, 0x1A, 0x5F, 0x92, 0x8F, 0x74,
                0xE5, 0xAD, 0x56, 0xA3, 0x8B, 0xED, 0x49, 0xDA,
                0x7B, 0x4F, 0x94, 0xB9, 0x38, 0xF4, 0x4C, 0xB9,
                0x03, 0x5A, 0x3A, 0xAD, 0xAD, 0xD9, 0xF0, 0x61,
                0x75, 0x83, 0x2F, 0x48, 0xD9, 0xE8, 0xD6, 0x67,
                0x1D, 0x50, 0x17, 0xD5, 0x0C, 0x76, 0x05, 0xC3,
                0x1E, 0xB1, 0xB8, 0x51, 0x87, 0x4B, 0x68, 0xD8,
                0x31, 0x10, 0xE7, 0xF3, 0xC3, 0x03, 0x3D, 0xD0,
                0xA6, 0xFF, 0x53, 0x2B, 0x8C, 0x71, 0x03, 0xC7,
                0xAC, 0xF3, 0xC4, 0xC4, 0x75, 0xB6, 0x38, 0xE3,
                0xEB, 0xAA, 0x31, 0x14, 0x94, 0x37, 0xA3, 0xF5,
                0xA1, 0xF2, 0x12, 0x43, 0x2B, 0xE9, 0x0C, 0x0C,
                0x86, 0x1D, 0xFF, 0xF9, 0x05, 0x46, 0xB7, 0x80,
                0x32, 0x15, 0x57, 0xD5, 0xD1, 0x89, 0xFD, 0xAC,
                0x7A, 0xA7, 0xF7, 0x70, 0xE1, 0xF1, 0x3E, 0x69,
                0x61, 0x2B, 0x17, 0x08, 0x1F, 0x67, 0x24, 0xA7,
                0xF6, 0xFA, 0x8A, 0x76, 0x72, 0xF8, 0x9F, 0x58,
                0xFB, 0x6D, 0xE9, 0x96, 0x3B, 0x13, 0x26, 0xAC,
                0x08, 0xA0, 0x73, 0x0B, 0x7A, 0x9E, 0x81, 0xFF,
                0x85, 0x7E, 0x59, 0x3B, 0x95, 0xFC, 0xEB, 0xD4,
                0x95, 0xD5, 0x6B, 0xCB, 0x3A, 0xF8, 0x42, 0xB7,
                0x84, 0x6E, 0xEC, 0xB3, 0xB3, 0xA9, 0x45, 0x0B,
                0x93, 0xE6, 0x37, 0xFA, 0x04, 0x8D, 0x1D, 0x7D,
                0xF8, 0xFF, 0xE2, 0x70, 0xF9, 0x50, 0xAA, 0xE6,
                0x11, 0xE5, 0xA2, 0x53, 0x64, 0xA7, 0x0F, 0xAD,
                0x86, 0xAB, 0xDB, 0x6B, 0xC9, 0x4C, 0xBF, 0x33,
            },
            Exponent = new byte[]
            {
                0x01, 0x00, 0x01,
            },
            D = new byte[]
            {
                0x36, 0x5D, 0x38, 0xD7, 0x82, 0xE3, 0x2F, 0xF4,
                0xC8, 0xB7, 0x0C, 0x06, 0x9D, 0x7C, 0x65, 0x13,
                0xCC, 0xF7, 0xAC, 0xDD, 0x07, 0xDB, 0xD6, 0x19,
                0xE4, 0xD3, 0xA9, 0x62, 0x13, 0x0E, 0x11, 0xD4,
                0xE7, 0xEE, 0x73, 0xA7, 0x16, 0x94, 0xFB, 0x74,
                0x76, 0xA0, 0xBC, 0xCE, 0xAC, 0x07, 0x1F, 0x13,
                0x5A, 0x69, 0x8A, 0x06, 0xFE, 0xC6, 0x0F, 0xAB,
                0xA9, 0x88, 0x0D, 0xEA, 0xC5, 0x86, 0xE4, 0x64,
                0x13, 0x18, 0x02, 0xEE, 0x0A, 0x89, 0xC6, 0x07,
                0x0E, 0x5D, 0x93, 0x48, 0x7B, 0xEF, 0x45, 0x95,
                0xFD, 0xE3, 0x7B, 0xDE, 0x08, 0x1E, 0x8B, 0xD3,
                0xB4, 0x52, 0x62, 0x73, 0x0C, 0x5D, 0x60, 0xC0,
                0xA1, 0x87, 0xD4, 0x5C, 0x15, 0x45, 0x76, 0x33,
                0xBF, 0x74, 0x5C, 0x1B, 0x3F, 0x4E, 0xC6, 0x33,
                0xBA, 0x99, 0x67, 0x3C, 0xA2, 0xAE, 0xFF, 0xB5,
                0xFE, 0xD0, 0x5A, 0xD6, 0x20, 0x82, 0xCA, 0x5F,
                0xAB, 0x3B, 0x81, 0x7A, 0xE8, 0x1C, 0x6F, 0x40,
                0x2C, 0x6D, 0x75, 0xDA, 0xE0, 0x34, 0xD7, 0x6A,
                0xD6, 0x0A, 0xE0, 0x75, 0x57, 0x65, 0x2C, 0x60,
                0x5E, 0x45, 0x54, 0x52, 0x4C, 0x1F, 0x92, 0x76,
                0x6E, 0x57, 0xE5, 0x77, 0xF7, 0x5F, 0xDB, 0x58,
                0x90, 0x03, 0xEE, 0x6D, 0xC2, 0xC6, 0x95, 0x63,
                0xD3, 0x81, 0xD7, 0x3B, 0x3B, 0xFC, 0x3B, 0xA7,
                0x17, 0x76, 0x19, 0x0A, 0x05, 0x25, 0x73, 0xCB,
                0xBA, 0x84, 0x5A, 0xC5, 0x23, 0x56, 0xF0, 0xB5,
                0x40, 0x6A, 0x1C, 0xCC, 0x9A, 0x38, 0xC5, 0x4F,
                0x00, 0x31, 0xB6, 0x56, 0x74, 0x6D, 0x78, 0x86,
                0xA4, 0x17, 0xB1, 0x5F, 0x19, 0xDB, 0x30, 0x30,
                0x38, 0xB3, 0x6C, 0xC0, 0xEB, 0xEB, 0x55, 0xD6,
                0xE3, 0xC7, 0x32, 0xC6, 0xDC, 0x29, 0x13, 0x26,
                0xB0, 0xB7, 0x7C, 0x45, 0xAB, 0x0A, 0x87, 0xA9,
                0x16, 0x6C, 0x17, 0x36, 0x87, 0x09, 0xF8, 0xF8,
                0x95, 0x69, 0x85, 0x46, 0x91, 0xDA, 0x7D, 0x15,
                0xD8, 0x18, 0x2A, 0x36, 0x46, 0x89, 0xDB, 0xF5,
                0xE0, 0xFC, 0x42, 0x07, 0x4B, 0xDC, 0x44, 0xC0,
                0x47, 0x52, 0x03, 0xEF, 0x95, 0x4F, 0xA4, 0x3D,
                0x90, 0xC8, 0x39, 0x8D, 0x44, 0x27, 0x19, 0x4A,
                0x89, 0xEE, 0xCB, 0xD8, 0x24, 0x93, 0x94, 0x6B,
                0x31, 0x8B, 0xA6, 0xD7, 0xF3, 0xBA, 0x07, 0x59,
                0x70, 0x06, 0xE3, 0xC1, 0xA6, 0x30, 0x22, 0x61,
                0x73, 0xC5, 0x39, 0x06, 0x94, 0x16, 0x75, 0xA5,
                0x06, 0x0E, 0x28, 0xAE, 0xE8, 0x56, 0x73, 0x59,
                0x75, 0x43, 0x26, 0x2B, 0x55, 0x5E, 0xD8, 0x91,
                0xE7, 0xE4, 0xDC, 0xB0, 0xE2, 0xC4, 0xC4, 0xEF,
                0x67, 0x80, 0x6A, 0x2C, 0xD0, 0xB7, 0xD2, 0xA7,
                0x0E, 0xAE, 0x45, 0x5C, 0x65, 0xAB, 0xA1, 0xC9,
                0x90, 0x57, 0x2A, 0x7C, 0xAD, 0x04, 0xBD, 0x27,
                0xC8, 0x9D, 0x28, 0xD9, 0x01, 0x1B, 0x08, 0xDB,
                0xA3, 0xD1, 0xBE, 0xBF, 0x52, 0x91, 0xA3, 0x24,
                0xB6, 0x11, 0x66, 0xC4, 0x47, 0x04, 0xCF, 0x8B,
                0x69, 0x15, 0x26, 0xF6, 0x07, 0x90, 0xF0, 0x25,
                0xF2, 0x12, 0xB3, 0x46, 0x47, 0x7B, 0x47, 0xE0,
                0xA7, 0x16, 0x3E, 0x32, 0x81, 0x0C, 0xBF, 0xC2,
                0xDE, 0x10, 0xF2, 0xFC, 0x81, 0x07, 0x09, 0x76,
                0xFD, 0x6F, 0x48, 0x60, 0xE8, 0x22, 0x83, 0x8F,
                0xE8, 0xD0, 0xC3, 0xA8, 0xD0, 0xBA, 0x44, 0xCB,
                0xB0, 0xF2, 0xEC, 0x9A, 0xB8, 0x59, 0x5E, 0x82,
                0x23, 0x32, 0xCF, 0x52, 0xEB, 0xC2, 0x06, 0xF2,
                0xCB, 0xF6, 0x60, 0xF5, 0x83, 0xD7, 0xE0, 0x4D,
                0x1A, 0xE8, 0x4B, 0x9A, 0x70, 0x44, 0x33, 0x39,
                0x8C, 0xC4, 0x86, 0xD2, 0x10, 0x35, 0x29, 0xD6,
                0x3F, 0x1E, 0x4B, 0xE5, 0x38, 0x98, 0xBC, 0x7B,
                0x65, 0x95, 0x83, 0xD3, 0x7F, 0x08, 0x59, 0x1D,
                0xE5, 0x3C, 0xCD, 0xE3, 0x11, 0x29, 0x7A, 0x48,
                0x9B, 0x66, 0x23, 0x9F, 0xB1, 0x49, 0x91, 0x78,
                0x77, 0x2F, 0x88, 0x77, 0xE3, 0x5F, 0xFC, 0x05,
                0xD5, 0x2F, 0x18, 0xFF, 0x7B, 0xEF, 0x74, 0xDF,
                0x17, 0x34, 0xA7, 0x5D, 0xFC, 0x0C, 0x9E, 0xD9,
                0x56, 0x2A, 0xD2, 0x3E, 0xAA, 0x2E, 0x27, 0x12,
                0x90, 0x25, 0xFA, 0xF1, 0xF5, 0xE0, 0xD2, 0xBB,
                0x82, 0xE1, 0x4F, 0x9D, 0x46, 0x3A, 0x0A, 0xB3,
                0xE1, 0xDB, 0xCC, 0xA2, 0x8D, 0x82, 0x47, 0xAF,
                0x86, 0xFC, 0xF8, 0x24, 0xB6, 0x27, 0xC2, 0xA1,
                0x8B, 0x33, 0x7B, 0xBF, 0xD1, 0x41, 0x15, 0xFD,
                0x45, 0xB9, 0xA1, 0x63, 0x78, 0x3E, 0x3E, 0x1F,
                0x65, 0xAD, 0xD7, 0xAF, 0xCD, 0x74, 0x98, 0x9F,
                0x10, 0xA9, 0x7D, 0x8A, 0x6C, 0xED, 0x6B, 0x57,
                0x3E, 0x51, 0x51, 0x64, 0x5D, 0xCF, 0x4C, 0x66,
                0xA0, 0x58, 0xD8, 0x76, 0x55, 0xBF, 0xB0, 0xC6,
                0xFD, 0xAD, 0xE0, 0x4C, 0xA5, 0x1C, 0x1F, 0x74,
                0x3A, 0xE5, 0x54, 0x51, 0x42, 0xCA, 0x04, 0x42,
                0x59, 0x35, 0xCC, 0x85, 0xF4, 0x02, 0x58, 0x62,
                0x39, 0xF4, 0x6C, 0x8B, 0x23, 0xEF, 0xDA, 0xBC,
                0x07, 0x85, 0x19, 0x74, 0xE6, 0x1C, 0x80, 0xEF,
                0x67, 0xD0, 0x5B, 0x84, 0xEF, 0x88, 0xE2, 0xFA,
                0xE7, 0xE1, 0x1F, 0x68, 0x3F, 0x33, 0x8D, 0x53,
                0x0E, 0x14, 0x4C, 0xD6, 0x97, 0x4E, 0x20, 0x59,
                0x62, 0xBD, 0x10, 0xD8, 0x51, 0xEA, 0xA4, 0x80,
                0x6D, 0x55, 0x6C, 0x8E, 0xB1, 0x3F, 0x43, 0xBC,
                0xF1, 0x34, 0xDF, 0xD3, 0xA7, 0x08, 0xC5, 0x17,
                0xB1, 0x3F, 0xCF, 0x6E, 0xD1, 0x66, 0xD3, 0x90,
                0x2A, 0x2E, 0x60, 0x1A, 0xD5, 0xEA, 0xC6, 0x10,
                0x5B, 0xA7, 0x65, 0xC9, 0xFC, 0xDD, 0x33, 0x3A,
                0x41, 0xDE, 0x1B, 0xED, 0x10, 0x30, 0x36, 0x64,
                0xE7, 0x97, 0x7B, 0xF1, 0x03, 0xE4, 0x60, 0x9D,
                0x6F, 0xC9, 0xF3, 0x6C, 0x4D, 0x63, 0x0B, 0x8E,
                0x4D, 0x51, 0x80, 0x16, 0x8A, 0x12, 0x41, 0x88,
                0xAE, 0x24, 0x16, 0xD3, 0xFD, 0x74, 0x9A, 0x8D,
                0xA5, 0xAD, 0x25, 0xE7, 0xF9, 0xBB, 0xB9, 0x0D,
                0x74, 0xF6, 0x9E, 0x0F, 0xC0, 0x05, 0xA2, 0xF8,
                0xB6, 0x1C, 0x6B, 0x73, 0xB8, 0xB0, 0xC6, 0x57,
                0xE8, 0x17, 0x91, 0x29, 0xE3, 0x6F, 0x90, 0x97,
                0x2F, 0xA5, 0xE1, 0xE0, 0x1F, 0xFE, 0x31, 0x13,
                0xB4, 0x27, 0xFC, 0x88, 0xE5, 0x67, 0x75, 0x92,
                0x9E, 0x53, 0x80, 0xFF, 0x46, 0x78, 0x7B, 0x4E,
                0xFD, 0x69, 0xB4, 0xCD, 0x20, 0x75, 0x59, 0xBB,
                0x80, 0x58, 0x46, 0x7E, 0xA5, 0x68, 0x2F, 0x74,
                0xBF, 0xBA, 0x0F, 0x4F, 0x0A, 0x11, 0x96, 0x87,
                0x3D, 0x2E, 0xFC, 0x8C, 0xC4, 0x19, 0xDB, 0x5C,
                0xAB, 0x2C, 0x78, 0x13, 0x3D, 0x0E, 0xF2, 0x55,
                0xFC, 0xC1, 0xF1, 0x6D, 0xF9, 0x37, 0x34, 0x76,
                0x52, 0xD5, 0xEB, 0x5F, 0x84, 0x5C, 0x31, 0x6C,
                0xE3, 0x17, 0x10, 0x48, 0x1B, 0x00, 0x1B, 0x31,
                0xF2, 0x2B, 0xE3, 0xF4, 0x95, 0x22, 0xA6, 0x48,
                0x41, 0xA4, 0x99, 0xCF, 0xE1, 0x6A, 0xCE, 0xCA,
                0x18, 0xAB, 0xC0, 0xEA, 0xE6, 0x99, 0x30, 0x33,
                0x78, 0xFB, 0x22, 0x72, 0x3F, 0xAB, 0xFC, 0x46,
                0x95, 0xE9, 0x82, 0x1E, 0xBE, 0xC0, 0x3F, 0xA1,
                0xE4, 0xB6, 0x3A, 0x60, 0x71, 0xEF, 0x6D, 0x9D,
                0x8A, 0x3D, 0x21, 0x2C, 0xA3, 0xFB, 0x72, 0xB6,
                0x85, 0x97, 0xEF, 0x52, 0x2B, 0x66, 0xBE, 0x69,
                0x55, 0xF6, 0x9C, 0xD4, 0x64, 0x03, 0xB6, 0xC1,
                0x01, 0xA2, 0x82, 0xBE, 0x1A, 0x65, 0xBD, 0x00,
                0xF6, 0x60, 0x93, 0xD9, 0xC7, 0xDF, 0xC2, 0x3D,
                0xF2, 0x66, 0x8A, 0x2B, 0xBE, 0x41, 0x61, 0x84,
                0x65, 0xC2, 0x13, 0xCA, 0x06, 0xB3, 0xB4, 0xF7,
                0xCD, 0x5C, 0x56, 0x13, 0xF6, 0x51, 0x76, 0x6B,
                0xD7, 0x25, 0xAD, 0x2A, 0xDE, 0x45, 0xEB, 0x14,
                0xDD, 0x82, 0xBC, 0xD5, 0xE5, 0xBB, 0xCB, 0x01,
                0xFB, 0xDE, 0xA4, 0x0E, 0x90, 0xF1, 0xB6, 0x73,
                0xDD, 0x3F, 0xCC, 0x13, 0x1C, 0x5F, 0x31, 0x7B,
                0x26, 0x8F, 0xDF, 0xE9, 0x33, 0x51, 0xED, 0xFA,
                0x41, 0x47, 0x49, 0xCD, 0xD9, 0xA3, 0x7A, 0x81,
                0x39, 0x57, 0xC1, 0x28, 0x05, 0x28, 0xE0, 0x89,
                0xEF, 0xBD, 0xEB, 0x13, 0x4E, 0xB4, 0x20, 0x15,
                0x2D, 0x4F, 0x8E, 0xC4, 0xB6, 0x21, 0xDC, 0x48,
                0xA0, 0xC2, 0x9B, 0xB9, 0x01, 0x8E, 0xAE, 0x72,
                0x3B, 0x81, 0x38, 0x0B, 0x96, 0xA0, 0x60, 0x9A,
                0x12, 0xD3, 0x71, 0x8D, 0x72, 0x6F, 0x85, 0x7E,
                0x0E, 0x18, 0xC0, 0xB6, 0x2D, 0x7D, 0x3D, 0xC0,
                0xC9, 0x3B, 0xCE, 0x35, 0xE6, 0x08, 0x9A, 0xC8,
                0x4C, 0x15, 0xF3, 0x2C, 0xF0, 0x4F, 0x3D, 0x6E,
                0x8F, 0x6C, 0x16, 0x08, 0x0D, 0x38, 0xAB, 0xC1,
                0xA7, 0xF3, 0x5C, 0x65, 0xA1, 0xD3, 0x3C, 0x11,
                0xA5, 0x25, 0x44, 0xAA, 0xBD, 0xAF, 0xBE, 0x4F,
                0x6D, 0x81, 0x5B, 0xD9, 0xBD, 0x07, 0xF2, 0xC5,
                0xBD, 0xCD, 0x84, 0x52, 0x6C, 0x26, 0x7A, 0x08,
                0xD8, 0xB0, 0x88, 0x8E, 0x3F, 0x91, 0x63, 0x2F,
                0x2C, 0x3D, 0xD3, 0x35, 0xDA, 0xFC, 0x97, 0xEC,
                0x66, 0xA4, 0x78, 0x52, 0x7B, 0xE7, 0x7B, 0x41,
                0xE3, 0xEE, 0x3D, 0x1A, 0xF5, 0xE8, 0x4E, 0xF9,
                0x5C, 0xEF, 0x31, 0x69, 0xEF, 0xBF, 0xC0, 0x95,
                0x41, 0x5E, 0x8A, 0x4C, 0x7B, 0x49, 0x7E, 0x24,
                0x94, 0xAF, 0x95, 0x36, 0xBE, 0xC8, 0xB0, 0x8D,
                0x1E, 0xB9, 0xED, 0xF1, 0xAA, 0x87, 0x79, 0x94,
                0x47, 0xCF, 0x4B, 0x2C, 0x28, 0x01, 0x75, 0xAE,
                0x20, 0x2F, 0xC5, 0x23, 0x5D, 0xF2, 0x9B, 0x57,
                0xEC, 0x20, 0xEF, 0x7D, 0xA0, 0x70, 0x73, 0x14,
                0xDE, 0x39, 0xD1, 0x7B, 0x76, 0xB0, 0x6F, 0xBD,
                0x3F, 0x4E, 0xD8, 0xD0, 0xDF, 0x90, 0x6B, 0x3C,
                0x42, 0x62, 0x57, 0xF0, 0x2C, 0x19, 0x16, 0x11,
                0xB7, 0x54, 0xB3, 0x15, 0x26, 0x14, 0xA7, 0x41,
                0x32, 0xCF, 0x6B, 0xAB, 0xFD, 0x8F, 0x88, 0x2E,
                0x71, 0x85, 0x09, 0x62, 0x8E, 0xCB, 0x1D, 0xEA,
                0x70, 0xE6, 0xE5, 0x6C, 0xE0, 0xA9, 0xB2, 0x83,
                0x5F, 0xE9, 0x74, 0xD6, 0x5B, 0x6B, 0xFB, 0x18,
                0xB0, 0x44, 0xB8, 0x9D, 0x54, 0x21, 0x09, 0xFD,
                0xC3, 0x72, 0x8A, 0x6E, 0x95, 0x14, 0x29, 0x8A,
                0x07, 0x9F, 0xBE, 0x0A, 0x87, 0xDE, 0xBA, 0xC8,
                0xD0, 0x31, 0x2A, 0x94, 0x0B, 0xB6, 0xF6, 0xA6,
                0x15, 0xE6, 0x22, 0xA8, 0x54, 0x72, 0x03, 0x49,
                0x28, 0xCD, 0x9C, 0x1E, 0x16, 0x2A, 0x0A, 0x20,
                0x8B, 0xCA, 0xA4, 0xDA, 0xEA, 0xDF, 0x3B, 0x72,
                0x4F, 0xEF, 0xF4, 0x54, 0xA6, 0xF3, 0x5A, 0xD5,
                0x4C, 0x8E, 0xFE, 0x07, 0xA6, 0xAA, 0xE6, 0x6E,
                0x95, 0x53, 0x18, 0x1F, 0x12, 0xBE, 0x56, 0x97,
                0x4D, 0x60, 0xE1, 0x43, 0x23, 0x66, 0x23, 0x35,
                0x2D, 0x7C, 0x94, 0x32, 0xE4, 0xF0, 0xFB, 0x96,
                0x8C, 0x69, 0xEB, 0xFB, 0xE0, 0x38, 0x45, 0x94,
                0x62, 0xC8, 0xBF, 0xB8, 0x15, 0x5B, 0x2F, 0x1F,
                0x4B, 0xA4, 0xAC, 0xD9, 0x09, 0xAD, 0xF9, 0xA3,
                0xB5, 0x68, 0xE6, 0xB3, 0x06, 0x5F, 0x21, 0x6D,
                0x93, 0xF6, 0x36, 0x65, 0xF6, 0x4A, 0xA2, 0x1F,
                0x9B, 0x01, 0x27, 0x37, 0x20, 0x61, 0x33, 0x61,
                0xA3, 0xF0, 0xD7, 0x1C, 0xA1, 0x79, 0x2E, 0x2D,
                0xBB, 0xDC, 0x4D, 0x69, 0x16, 0x20, 0xE0, 0x4D,
                0x1E, 0x6E, 0x82, 0x74, 0x0E, 0x89, 0x5B, 0x33,
                0xF8, 0x16, 0xD6, 0x92, 0xE3, 0xF0, 0x06, 0xAF,
                0x38, 0xA7, 0x82, 0xF8, 0x96, 0x9C, 0x13, 0xF5,
                0xE4, 0x81, 0x2D, 0x90, 0x9F, 0x9F, 0x51, 0x1A,
                0x9B, 0xF6, 0x1A, 0x78, 0xD6, 0xF7, 0xCC, 0x07,
                0x3A, 0xA3, 0x1A, 0x91, 0x61, 0x14, 0x00, 0x59,
                0xF5, 0x9A, 0x72, 0x32, 0x0A, 0xD6, 0x3B, 0xC2,
                0x47, 0x36, 0xE3, 0xE4, 0xAA, 0xD9, 0x8A, 0x65,
                0x7B, 0xC8, 0x7F, 0x96, 0xB9, 0xC0, 0xA7, 0xF2,
                0x55, 0xAD, 0x67, 0x2C, 0x14, 0x0C, 0xDF, 0xC9,
                0x4E, 0x36, 0xA3, 0x22, 0xAD, 0xA7, 0x5A, 0x5D,
                0xD6, 0xD1, 0xB4, 0xD9, 0x69, 0x39, 0x80, 0x25,
                0x97, 0x0D, 0x7F, 0xF7, 0x76, 0x08, 0x29, 0x04,
                0x5A, 0xA4, 0x12, 0x4A, 0x29, 0xD9, 0xB9, 0x74,
                0x27, 0x40, 0x5A, 0xC6, 0x66, 0xFF, 0xB2, 0x41,
                0x9F, 0x0B, 0x84, 0xF2, 0x8B, 0x5F, 0x8E, 0xB9,
                0x04, 0x1E, 0x70, 0xFD, 0xAD, 0xBD, 0x03, 0x0C,
                0x50, 0x96, 0x2C, 0x80, 0x80, 0xCD, 0xA0, 0x87,
                0xB0, 0xBC, 0xF0, 0xB1, 0x77, 0x30, 0xB1, 0xCA,
                0xD4, 0xDA, 0xCA, 0xB6, 0xF6, 0xEB, 0xA5, 0x1E,
                0x77, 0x6F, 0x2A, 0x6A, 0xA9, 0x45, 0xEF, 0x1B,
                0x11, 0x2C, 0x5E, 0xEE, 0x3C, 0x53, 0xF9, 0x37,
                0xDB, 0xA1, 0x42, 0x4F, 0x3B, 0x5D, 0xE0, 0xE4,
                0xCE, 0x90, 0xD3, 0x90, 0x20, 0x0C, 0x19, 0x7F,
                0x05, 0x7A, 0xF7, 0xBE, 0x6E, 0xE3, 0xAA, 0x64,
                0x91, 0x41, 0x79, 0x48, 0xA5, 0x72, 0x42, 0xBF,
                0xE2, 0x3F, 0x99, 0x8E, 0x89, 0x9B, 0x7E, 0xA9,
                0x44, 0xEA, 0x99, 0x32, 0xEF, 0xC7, 0x91, 0x84,
                0xEB, 0x6F, 0x48, 0xCF, 0x56, 0x62, 0x60, 0xB5,
                0xFB, 0x3A, 0x40, 0x25, 0x86, 0x08, 0x4E, 0x30,
                0x8B, 0xEE, 0x82, 0x61, 0x1C, 0xA3, 0x8B, 0x94,
                0xBC, 0x25, 0x1E, 0x14, 0xA9, 0x7D, 0xFD, 0x03,
                0x4F, 0xE6, 0xB3, 0xD5, 0x96, 0xB9, 0x3F, 0xDC,
                0x50, 0x1B, 0x2B, 0x40, 0x00, 0xDD, 0xB8, 0x60,
                0x8E, 0xB5, 0xBB, 0x90, 0xED, 0x93, 0x71, 0x91,
                0x4A, 0x9E, 0x92, 0xB1, 0x15, 0x69, 0xC2, 0x03,
                0xD6, 0xB5, 0xED, 0x95, 0xA5, 0x0B, 0x6E, 0x52,
                0xFB, 0x14, 0xDE, 0x1C, 0x3F, 0x99, 0xA0, 0x58,
                0x4F, 0x26, 0x8A, 0x86, 0xC9, 0xE1, 0x2A, 0x5B,
                0xA5, 0x0D, 0x4A, 0x05, 0x17, 0x01, 0x88, 0xA9,
                0x6D, 0x52, 0xAA, 0x7C, 0x9C, 0x22, 0x06, 0x33,
                0x33, 0x3D, 0x6E, 0xD4, 0x79, 0xF8, 0x16, 0x9B,
                0x0C, 0x5F, 0x37, 0xBE, 0x8B, 0x7C, 0x82, 0x3D,
                0x3B, 0xC9, 0xB9, 0xDD, 0x55, 0x8D, 0xFE, 0x6F,
                0x06, 0x80, 0x16, 0x88, 0x06, 0xDE, 0x55, 0xFA,
                0xC9, 0xB6, 0x91, 0x3E, 0x73, 0x47, 0x6D, 0xE9,
                0xD3, 0x5F, 0xEE, 0x03, 0x5F, 0x14, 0x4A, 0xF0,
                0x85, 0xAA, 0xAE, 0xCB, 0x0E, 0x41, 0x72, 0xF6,
                0x71, 0x20, 0x3B, 0x9C, 0x6E, 0xB4, 0xB6, 0xD8,
                0x62, 0xEB, 0xC6, 0x79, 0xA6, 0xB0, 0x02, 0x84,
                0x95, 0xE3, 0xA0, 0xE8, 0x33, 0xF1, 0x33, 0xAE,
                0xC2, 0xC6, 0x3B, 0xE0, 0x6C, 0x80, 0xAD, 0x66,
                0x47, 0x67, 0x79, 0x48, 0xB6, 0x93, 0x9F, 0xCC,
                0x4C, 0xB2, 0x1B, 0x77, 0x48, 0x4C, 0xAB, 0x7D,
                0xD2, 0xCB, 0xC0, 0xC0, 0xD7, 0x92, 0x4E, 0x16,
                0x03, 0xA8, 0x3E, 0x69, 0xAE, 0xDC, 0x00, 0x86,
                0x47, 0x0B, 0x26, 0x9B, 0x08, 0x06, 0xF8, 0x9C,
                0x5A, 0xE9, 0x57, 0xAB, 0x45, 0xDA, 0xAB, 0x0D,
                0x84, 0xEE, 0x07, 0x86, 0xE0, 0xB4, 0x66, 0x06,
                0xB7, 0xAE, 0x55, 0x22, 0x7A, 0xEC, 0x56, 0xE9,
                0xFF, 0x35, 0xA3, 0x76, 0x82, 0x9E, 0x6C, 0x92,
                0xB5, 0xCF, 0x29, 0x6D, 0x5C, 0x59, 0x50, 0xD7,
                0xD8, 0x37, 0xA3, 0xD1, 0x76, 0xAF, 0x59, 0xCC,
                0x0F, 0xEA, 0x22, 0x89, 0xED, 0x04, 0xDD, 0xD4,
                0x40, 0xD5, 0xAA, 0xC8, 0x77, 0xAE, 0xE1, 0xE8,
                0x76, 0x0B, 0xE4, 0x3B, 0x22, 0x5E, 0xC1, 0xFD,
                0x38, 0x29, 0x1A, 0x76, 0x57, 0x53, 0xE8, 0x97,
                0x4E, 0x60, 0x96, 0x51, 0xF6, 0x5C, 0xDC, 0x81,
                0xF3, 0xC3, 0x7A, 0xA8, 0xBE, 0x37, 0x83, 0xB8,
                0x35, 0xCD, 0xCF, 0x3D, 0xBE, 0x76, 0xC2, 0x75,
            },
            P = new byte[]
            {
                0xCA, 0x4D, 0x00, 0x39, 0x39, 0x5E, 0x40, 0xCA,
                0x7C, 0x2B, 0xDE, 0xE0, 0xDD, 0x98, 0xEB, 0xB5,
                0xF5, 0x8E, 0xAD, 0x22, 0x3F, 0xC4, 0x47, 0xD4,
                0x4B, 0x7D, 0xFA, 0xCA, 0x9D, 0x5A, 0xE2, 0xD4,
                0x10, 0x1E, 0x7E, 0x7D, 0x04, 0xFC, 0xB9, 0x1D,
                0xF4, 0x10, 0xC5, 0x8B, 0xC7, 0x43, 0xE1, 0x4E,
                0x39, 0x4E, 0x23, 0x22, 0x95, 0xB0, 0x3A, 0x3B,
                0xFF, 0x7C, 0xF2, 0x34, 0x2C, 0xDC, 0x76, 0xD7,
                0xD4, 0xBC, 0x67, 0xE3, 0x3F, 0xEF, 0xEA, 0x6E,
                0x64, 0xF1, 0x54, 0x51, 0x08, 0x21, 0x8E, 0xC3,
                0x11, 0x93, 0x1B, 0x20, 0x5C, 0x07, 0x03, 0x5B,
                0x60, 0x7F, 0x90, 0x34, 0x83, 0xD4, 0x0E, 0xF6,
                0x05, 0xDA, 0xD9, 0xD6, 0x7A, 0x0B, 0x12, 0xDF,
                0x8E, 0xF0, 0x43, 0x0F, 0xD0, 0x77, 0x57, 0x50,
                0x0E, 0xE5, 0x84, 0x50, 0x28, 0xFE, 0x4F, 0x63,
                0x68, 0xAE, 0x92, 0xF9, 0xD5, 0xB5, 0x83, 0xBB,
                0x44, 0x44, 0x66, 0xF2, 0x36, 0xE5, 0x5A, 0x75,
                0x13, 0x05, 0x8F, 0xF5, 0x79, 0x23, 0xAA, 0x8A,
                0xCC, 0x31, 0x36, 0xBC, 0x35, 0x25, 0x09, 0x10,
                0x03, 0xEA, 0x9B, 0x3F, 0x0B, 0x65, 0xFD, 0xA7,
                0xC3, 0x69, 0xC6, 0xAA, 0x30, 0x7A, 0xAA, 0x79,
                0x6D, 0x80, 0xCA, 0xF2, 0x09, 0xF6, 0xED, 0xA8,
                0x83, 0xF8, 0x58, 0xA2, 0xD7, 0x46, 0x79, 0x8B,
                0xC3, 0x83, 0x51, 0xC6, 0x04, 0x68, 0x12, 0x4D,
                0xD9, 0x85, 0xE2, 0x80, 0x14, 0xEE, 0x2A, 0x9A,
                0x06, 0xC0, 0x5A, 0x64, 0xBB, 0x88, 0x37, 0x79,
                0xB0, 0x4A, 0x57, 0x17, 0x59, 0x74, 0x7E, 0x05,
                0x15, 0xA6, 0x7E, 0x02, 0x60, 0xCB, 0x9B, 0x13,
                0x04, 0x2F, 0x28, 0x83, 0x16, 0xEA, 0xC2, 0xBB,
                0x14, 0xAF, 0x66, 0x1B, 0x52, 0x05, 0x15, 0x38,
                0x34, 0xF7, 0xC3, 0xEE, 0x32, 0x41, 0x38, 0x47,
                0xFC, 0xE1, 0xBB, 0x25, 0x30, 0x7E, 0xA0, 0x56,
                0xB7, 0x3F, 0xB6, 0xE5, 0x76, 0xC6, 0x77, 0x46,
                0xC8, 0xA9, 0x3C, 0x21, 0x2B, 0x85, 0x42, 0x3D,
                0x9E, 0x76, 0x45, 0xCD, 0x69, 0x8C, 0xB1, 0xE6,
                0x46, 0x8C, 0x7B, 0xEE, 0x52, 0x9F, 0x57, 0x12,
                0xB2, 0xF0, 0xF4, 0x43, 0xAE, 0x7E, 0x8F, 0x22,
                0xE5, 0x59, 0xBB, 0xF7, 0xCE, 0xE3, 0x99, 0xE4,
                0x6F, 0x3D, 0xF4, 0xD3, 0x6E, 0xD4, 0xF0, 0xD0,
                0x9F, 0x95, 0x3F, 0x7A, 0xFF, 0x24, 0x82, 0x63,
                0x6C, 0xA6, 0x72, 0xCC, 0x22, 0x20, 0x79, 0xF7,
                0x38, 0xEA, 0x96, 0xD1, 0xAA, 0xCD, 0xF3, 0xF6,
                0x73, 0xAA, 0x69, 0x01, 0x19, 0x9C, 0x17, 0x58,
                0xD2, 0x96, 0x25, 0x9D, 0xB4, 0x85, 0xCF, 0x39,
                0xBA, 0xF2, 0xFF, 0xCD, 0xCA, 0xC0, 0x98, 0x3D,
                0xA3, 0x69, 0x58, 0x11, 0x4A, 0xA0, 0xD9, 0x75,
                0x5A, 0x50, 0x69, 0x95, 0xE6, 0xC3, 0x36, 0x51,
                0x2E, 0x5B, 0x7A, 0x78, 0xB3, 0x13, 0xF1, 0xA4,
                0xDA, 0x3B, 0x22, 0xCB, 0x70, 0x33, 0x59, 0x34,
                0x63, 0x4E, 0x67, 0x72, 0xA4, 0xF2, 0xEA, 0x14,
                0xB6, 0xC2, 0x21, 0x83, 0xE2, 0x48, 0x46, 0x90,
                0xEB, 0xB0, 0x2C, 0x76, 0xB6, 0xAF, 0xEE, 0xE8,
                0x9F, 0x15, 0x4F, 0x01, 0x42, 0x7C, 0xF6, 0xBD,
                0xC0, 0xC3, 0x78, 0xC6, 0xF9, 0xE0, 0xB9, 0x4B,
                0x26, 0x2F, 0xB0, 0x4D, 0x40, 0x5D, 0xDE, 0x0B,
                0xA1, 0xB9, 0x01, 0x38, 0xB4, 0xE4, 0xB7, 0x3B,
                0x8D, 0xEE, 0xC6, 0x62, 0xA7, 0x97, 0x5A, 0x50,
                0x56, 0x62, 0xF9, 0xAD, 0xAD, 0x10, 0x7F, 0xB9,
                0x3B, 0x7B, 0x20, 0xC1, 0xAF, 0x81, 0x11, 0x50,
                0x70, 0xA7, 0xC2, 0xC4, 0x8D, 0x37, 0x9C, 0xD1,
                0xB4, 0x64, 0x9B, 0xE4, 0xFE, 0x0D, 0xA3, 0xC1,
                0xC4, 0x75, 0x51, 0x78, 0x23, 0x3F, 0x1B, 0xCF,
                0x0E, 0x21, 0x09, 0x71, 0x3C, 0xC9, 0xBF, 0xBC,
                0x0E, 0x42, 0x93, 0x01, 0x5D, 0x96, 0xF5, 0x8D,
                0x9D, 0x81, 0xEE, 0x50, 0x03, 0x02, 0x83, 0xE9,
                0xD4, 0xEC, 0x16, 0x09, 0x0D, 0x36, 0x0A, 0x56,
                0xAE, 0xA3, 0xC7, 0x43, 0x68, 0xA7, 0x8E, 0xEA,
                0x2A, 0xB5, 0x77, 0xED, 0x0A, 0xD4, 0x41, 0xE4,
                0x11, 0xA6, 0xF8, 0x04, 0x9E, 0xB3, 0x2E, 0x98,
                0xC3, 0x18, 0x78, 0xBD, 0xBD, 0x8E, 0x8B, 0xB6,
                0x9E, 0x98, 0xB5, 0x6D, 0x0F, 0xEF, 0xDC, 0x12,
                0x22, 0x1C, 0xBC, 0x4A, 0x7B, 0xFF, 0xD7, 0xC6,
                0x0B, 0x67, 0xB0, 0xA3, 0xA5, 0x04, 0xF7, 0xC1,
                0x85, 0xD6, 0xEB, 0x74, 0xEC, 0x8B, 0x4A, 0x03,
                0x7C, 0x06, 0x27, 0x56, 0xD8, 0x45, 0xD9, 0x2E,
                0x30, 0x64, 0xEE, 0x3C, 0x80, 0x38, 0xBC, 0x20,
                0xC3, 0x03, 0x98, 0x1F, 0xD3, 0xDC, 0x9A, 0x94,
                0x34, 0xF2, 0x87, 0x61, 0x2A, 0x38, 0x7C, 0xDC,
                0x43, 0xB9, 0xA6, 0x26, 0x11, 0x54, 0x8E, 0x6B,
                0x9B, 0x60, 0x5A, 0x6B, 0xB8, 0x3D, 0x68, 0x3F,
                0x5C, 0x6E, 0x16, 0xE1, 0x54, 0xBA, 0x4D, 0xEE,
                0x21, 0xFA, 0x71, 0xCD, 0xEC, 0x8A, 0x96, 0xCF,
                0xD3, 0x50, 0x54, 0x53, 0xC4, 0xBF, 0x9D, 0x10,
                0x4B, 0xD3, 0x33, 0x02, 0x3C, 0xC6, 0x3F, 0xA9,
                0xC3, 0x5C, 0x04, 0x87, 0xC2, 0xA7, 0x99, 0x09,
                0x72, 0x74, 0x03, 0x5B, 0xA3, 0x52, 0x5A, 0x71,
                0xCB, 0x6F, 0x24, 0x8D, 0xCB, 0xCB, 0xE3, 0x17,
                0x90, 0xE5, 0xEC, 0x7B, 0xBC, 0x85, 0xD5, 0x60,
                0x24, 0x88, 0xB3, 0x6D, 0x0B, 0xA3, 0xBD, 0x43,
                0xDA, 0x44, 0xD6, 0xED, 0xCC, 0xC0, 0x1F, 0x6C,
                0x4B, 0x9C, 0x71, 0x63, 0x60, 0x0A, 0xEC, 0x73,
                0xCF, 0xD1, 0x1D, 0x70, 0x9E, 0xAA, 0xBD, 0x7C,
                0xFE, 0x21, 0x46, 0x47, 0xAB, 0x21, 0x46, 0x04,
                0x56, 0x4E, 0x6A, 0xA5, 0xC9, 0xD0, 0x6B, 0xCB,
                0xBA, 0xD4, 0xE4, 0xFC, 0x6C, 0x03, 0xA2, 0xF2,
                0x22, 0x11, 0x25, 0x85, 0x6D, 0x0B, 0x52, 0xB8,
                0x6F, 0x42, 0x15, 0x6E, 0x9C, 0xA2, 0x67, 0xFD,
                0x16, 0x60, 0x4B, 0x6D, 0xF0, 0x2E, 0xAB, 0xBA,
                0x66, 0x42, 0x45, 0x7E, 0x7C, 0x9A, 0xB6, 0xF8,
                0x33, 0x3F, 0xB2, 0x5A, 0x9C, 0x75, 0x7C, 0xD4,
                0xDE, 0x4A, 0xD6, 0xD6, 0x7E, 0xCA, 0x54, 0xE0,
                0x5A, 0xA8, 0x57, 0x0C, 0xF9, 0x53, 0x08, 0xD1,
                0x05, 0x9D, 0x92, 0xB5, 0xE3, 0x24, 0x75, 0xF9,
                0x42, 0x7F, 0x2C, 0xF7, 0x2B, 0x06, 0x13, 0x44,
                0x91, 0x40, 0xE2, 0x18, 0x49, 0x02, 0xA3, 0xA8,
                0xFB, 0xFC, 0x05, 0x89, 0x75, 0xBF, 0x4D, 0x27,
                0x43, 0x44, 0xB7, 0xD7, 0xE5, 0x13, 0x33, 0x1A,
                0x6A, 0xC2, 0x20, 0x67, 0x56, 0x3F, 0x25, 0x2B,
                0x8F, 0xCD, 0x74, 0xFA, 0x0F, 0x77, 0x8C, 0x81,
                0x5E, 0x88, 0xA0, 0x0C, 0x33, 0x7A, 0xBA, 0xE1,
                0x63, 0xB7, 0x94, 0x28, 0x29, 0xEC, 0x0D, 0xEF,
                0x80, 0x3B, 0x6F, 0x9E, 0x80, 0xC0, 0xCF, 0x53,
                0xDB, 0xB9, 0xB1, 0x93, 0x23, 0x1D, 0x99, 0x40,
                0x3A, 0x7D, 0xB0, 0x87, 0x67, 0x93, 0x25, 0x27,
                0xB1, 0x73, 0xA1, 0x44, 0x3F, 0xDC, 0x42, 0x72,
                0x3C, 0x5F, 0x5C, 0x2C, 0x83, 0x50, 0xFA, 0x15,
                0x5B, 0x0B, 0xFB, 0x0A, 0x41, 0xF8, 0x85, 0xC8,
                0x0F, 0xF0, 0xEB, 0xBE, 0xCD, 0x84, 0xEF, 0xF2,
                0x77, 0xED, 0x73, 0xAA, 0xBC, 0x77, 0x2B, 0x9D,
                0x3A, 0x62, 0xC5, 0x08, 0x61, 0x4A, 0xD9, 0xDD,
                0x37, 0x88, 0x33, 0xC3, 0xA4, 0xB0, 0xDC, 0xA7,
                0x8B, 0xA5, 0xF5, 0x5C, 0xCF, 0xFF, 0x0C, 0x03,
                0x11, 0xCF, 0xAB, 0xB4, 0x34, 0xC0, 0x28, 0xAE,
                0x12, 0xAA, 0x3A, 0x42, 0x10, 0x46, 0x11, 0x2D,
                0xF9, 0xA5, 0x45, 0x2C, 0x81, 0x9B, 0xDC, 0x4E,
                0x57, 0xCC, 0x13, 0xB4, 0xC2, 0x1B, 0x99, 0xA1,
                0xE9, 0x71, 0x45, 0x9A, 0x2E, 0x7D, 0xF2, 0x98,
                0xDE, 0xDA, 0x70, 0x53, 0x63, 0xE1, 0x29, 0xBF,
            },
            DP = new byte[]
            {
                0x18, 0x12, 0xF7, 0xBE, 0xD7, 0x93, 0xDE, 0xD3,
                0xF9, 0xD8, 0xE2, 0xAA, 0x11, 0xD4, 0xDB, 0xE0,
                0x08, 0x7B, 0xD5, 0x20, 0xA9, 0x43, 0xFB, 0x64,
                0x49, 0x23, 0x91, 0xCF, 0xC0, 0xD0, 0x0B, 0x04,
                0x3F, 0x72, 0xD1, 0x8C, 0xA1, 0x26, 0x4E, 0x05,
                0x41, 0x81, 0x29, 0x71, 0x0B, 0xE2, 0x89, 0x12,
                0x5D, 0x01, 0x6E, 0x6E, 0xF4, 0x2F, 0x47, 0x8E,
                0xD2, 0x45, 0x95, 0x31, 0x1E, 0x51, 0x92, 0x16,
                0xF7, 0x2B, 0x00, 0x95, 0xEB, 0x8A, 0xEA, 0x73,
                0xFE, 0xB1, 0x35, 0x5E, 0x7B, 0x40, 0x3B, 0x13,
                0xFD, 0xA8, 0x6A, 0xE6, 0xFB, 0xEC, 0x9D, 0xBA,
                0xA7, 0x0E, 0x27, 0x24, 0x08, 0xB8, 0x18, 0x9B,
                0xB0, 0x70, 0xAD, 0xD1, 0xB7, 0x2E, 0x50, 0x2D,
                0xA8, 0x7D, 0xF1, 0x0D, 0x15, 0xBA, 0xCD, 0xFA,
                0x29, 0xFB, 0xA8, 0x36, 0x3D, 0xDA, 0x9D, 0xA9,
                0xEF, 0xD0, 0x2E, 0x8F, 0x6A, 0x9E, 0x32, 0x31,
                0xFB, 0xDA, 0xC4, 0x01, 0x79, 0x04, 0xEC, 0x31,
                0xD8, 0x74, 0xA6, 0x00, 0x09, 0x4D, 0x74, 0x43,
                0x16, 0x2F, 0x99, 0x1A, 0xE6, 0x9C, 0x24, 0xAA,
                0xF2, 0x3C, 0x5E, 0x03, 0x2F, 0xA1, 0x10, 0x81,
                0x81, 0x60, 0xBA, 0x12, 0x90, 0xB8, 0x58, 0x47,
                0x20, 0xFF, 0xDD, 0xA6, 0xD6, 0x06, 0xBB, 0x9B,
                0x7D, 0x30, 0xF5, 0xA3, 0x53, 0x49, 0x00, 0xB7,
                0xE0, 0x29, 0x65, 0x76, 0xD2, 0x19, 0x6C, 0x6C,
                0x35, 0x41, 0x98, 0x85, 0xB3, 0x77, 0xF0, 0x3B,
                0xEA, 0x27, 0xC3, 0xDA, 0x0E, 0xF3, 0x13, 0xDE,
                0xF8, 0x5A, 0xB0, 0x68, 0x87, 0xED, 0xB3, 0xFD,
                0x78, 0xE9, 0x1A, 0x3F, 0xC0, 0x33, 0x1A, 0x9E,
                0x35, 0xB6, 0x42, 0xF4, 0xEE, 0xAA, 0x3B, 0x48,
                0x36, 0x1A, 0xF5, 0x64, 0xB4, 0xEB, 0x03, 0xEE,
                0x6F, 0x67, 0x38, 0xBA, 0xC4, 0xE2, 0x3C, 0x07,
                0x5D, 0x11, 0xA3, 0xCA, 0xB6, 0x2D, 0xAB, 0x79,
                0x06, 0x4F, 0x9F, 0xBD, 0x48, 0xD8, 0x2F, 0x63,
                0x8E, 0x07, 0x8D, 0xAF, 0x48, 0xD5, 0x8F, 0xDF,
                0x73, 0x57, 0x11, 0xD1, 0x73, 0x09, 0x1A, 0x36,
                0x94, 0x18, 0xAD, 0xBA, 0xDB, 0xBC, 0x38, 0x89,
                0x72, 0x1F, 0xF8, 0x81, 0x81, 0x67, 0x70, 0x33,
                0x2F, 0xE5, 0xF0, 0xD7, 0x79, 0x98, 0x5E, 0x3C,
                0xEF, 0xFC, 0x08, 0x81, 0x8C, 0xC3, 0xEC, 0x70,
                0x77, 0x3D, 0x34, 0x93, 0xB7, 0x7F, 0x29, 0xC1,
                0x19, 0x31, 0xE9, 0xA1, 0x5F, 0x42, 0x4C, 0x21,
                0x5E, 0x75, 0x94, 0x43, 0x19, 0x37, 0x6F, 0x1B,
                0xDA, 0x01, 0xE2, 0x83, 0x0E, 0x00, 0x24, 0x4B,
                0x1E, 0xAC, 0x5D, 0x87, 0x99, 0xEE, 0xFE, 0x8D,
                0x19, 0x31, 0x47, 0xBD, 0xBE, 0xAE, 0x12, 0xAF,
                0xEB, 0x1D, 0x63, 0x2C, 0x93, 0x9B, 0xF6, 0xA4,
                0xDF, 0x7D, 0x88, 0x43, 0x1D, 0x76, 0x07, 0xA5,
                0xBB, 0x85, 0x89, 0x5A, 0x89, 0xBD, 0x0A, 0xD9,
                0x9A, 0x5A, 0xC5, 0x36, 0x3E, 0x80, 0xED, 0xD1,
                0xAD, 0x2B, 0xAC, 0x65, 0xD9, 0x39, 0x4B, 0x1F,
                0xF1, 0xEB, 0xC2, 0x3F, 0x46, 0x93, 0x61, 0x4A,
                0x67, 0xB1, 0xCC, 0x68, 0xC8, 0x2E, 0xC1, 0x98,
                0x8F, 0x2D, 0xE2, 0xFB, 0xFC, 0x64, 0x90, 0x9C,
                0x5E, 0x2F, 0x24, 0xD5, 0x50, 0xF1, 0x2C, 0x3B,
                0xC4, 0x2C, 0x92, 0xA7, 0x6E, 0xCC, 0x7C, 0xDB,
                0x17, 0x80, 0xC3, 0xA3, 0x72, 0xEB, 0x70, 0xDE,
                0xB6, 0x72, 0x3E, 0xCB, 0x88, 0xB4, 0x1B, 0x3C,
                0x4A, 0x3B, 0x77, 0x08, 0xF2, 0xFA, 0x6E, 0xA8,
                0xA5, 0x6A, 0x6E, 0xA8, 0x7D, 0xF1, 0x37, 0x15,
                0x42, 0x82, 0xC4, 0x4B, 0xCD, 0x9E, 0x5B, 0x9C,
                0x1D, 0x02, 0x88, 0x06, 0xC5, 0x30, 0xEC, 0x56,
                0xE7, 0xC1, 0x2A, 0x53, 0xC8, 0xA5, 0xFE, 0xF2,
                0x31, 0xF5, 0x3E, 0x81, 0x6A, 0x41, 0x7B, 0xFE,
                0xAE, 0x17, 0xC0, 0x14, 0xBE, 0x85, 0x73, 0x6D,
                0x49, 0xDC, 0x27, 0x77, 0x00, 0x14, 0xB1, 0x8C,
                0x07, 0x19, 0x9D, 0x39, 0xB0, 0x87, 0xC8, 0xCD,
                0x2D, 0xF5, 0x31, 0x86, 0x55, 0x12, 0xF3, 0x8F,
                0xEC, 0x4B, 0x32, 0x1D, 0x54, 0x57, 0x94, 0x0B,
                0xC7, 0x09, 0xFE, 0xA3, 0xD6, 0x1A, 0xEE, 0xA5,
                0xA1, 0x39, 0xED, 0x4C, 0x6F, 0x1D, 0x62, 0x84,
                0xF5, 0xF4, 0xA8, 0x4A, 0x75, 0x46, 0x0F, 0x03,
                0x5D, 0x69, 0xDC, 0x02, 0x65, 0x25, 0x3A, 0x11,
                0x48, 0x54, 0x2B, 0x92, 0x1D, 0xD6, 0x2C, 0x81,
                0xAC, 0x22, 0xBA, 0x5C, 0x6C, 0xB5, 0xDA, 0xB5,
                0xF5, 0x71, 0x6A, 0x07, 0x0C, 0xAF, 0xAB, 0x3B,
                0xB2, 0xE8, 0x9F, 0xED, 0x35, 0x39, 0x0B, 0x32,
                0x3E, 0xE2, 0xD3, 0x9C, 0x9E, 0x02, 0xB7, 0xA6,
                0x81, 0x72, 0x87, 0x27, 0xC9, 0xF5, 0x74, 0xEE,
                0x65, 0x64, 0xD7, 0x5F, 0xDA, 0x5A, 0x1C, 0xA4,
                0xB3, 0x95, 0xD0, 0xCC, 0xD6, 0xDC, 0xFF, 0xE5,
                0xE2, 0x62, 0xFB, 0x78, 0x0F, 0x34, 0x28, 0x87,
                0xF9, 0x25, 0x2B, 0x9B, 0xDC, 0xD5, 0x55, 0x43,
                0x20, 0x1B, 0x84, 0x1D, 0x7F, 0xE1, 0x69, 0x98,
                0x81, 0xDD, 0x7D, 0x49, 0x7B, 0xDF, 0xFF, 0xBD,
                0x7D, 0x11, 0x1B, 0x3C, 0xE8, 0xAE, 0x37, 0x29,
                0x07, 0xA4, 0xC4, 0xAD, 0x88, 0x0F, 0x09, 0xD2,
                0x56, 0xEA, 0x40, 0x08, 0x5B, 0xC3, 0x44, 0xA0,
                0x0E, 0x4F, 0x3E, 0x48, 0x2F, 0x54, 0x21, 0xE3,
                0x52, 0x15, 0xAE, 0x7C, 0x80, 0x91, 0x18, 0xB9,
                0xD5, 0x64, 0xB1, 0xCB, 0x14, 0xBD, 0x9C, 0x3F,
                0xAF, 0xF3, 0xCB, 0x0E, 0x8F, 0x64, 0x5D, 0x65,
                0x1E, 0xCA, 0xFC, 0xDC, 0xE5, 0x14, 0xDE, 0x7D,
                0xDC, 0x64, 0x2B, 0x4F, 0xE6, 0x0E, 0x8C, 0x9D,
                0x81, 0x83, 0xCD, 0x6F, 0x33, 0x48, 0x09, 0x3B,
                0xF2, 0x5C, 0xD9, 0x6F, 0x2C, 0x8F, 0x76, 0x39,
                0xA8, 0x52, 0x30, 0x0B, 0xE3, 0xC1, 0x20, 0x33,
                0xF0, 0x91, 0x85, 0xA9, 0x67, 0x1C, 0x70, 0x91,
                0x8E, 0xB3, 0x20, 0xE6, 0xD1, 0x59, 0x4C, 0x78,
                0x5F, 0x28, 0xED, 0xCA, 0x32, 0x9B, 0xDA, 0xC0,
                0x48, 0xA1, 0x00, 0xE1, 0x85, 0x92, 0xF9, 0xAA,
                0xFF, 0x55, 0x1A, 0xA1, 0xE5, 0xEE, 0xC0, 0x10,
                0xFE, 0xD8, 0xDF, 0x9B, 0x1C, 0xA4, 0x83, 0xFD,
                0x13, 0xD4, 0xFF, 0x9B, 0x83, 0x8F, 0x58, 0x36,
                0xB4, 0x72, 0x1B, 0xF0, 0xC1, 0xFE, 0xF4, 0x16,
                0x09, 0xCF, 0x15, 0xD8, 0xDB, 0xFF, 0x63, 0x68,
                0x7D, 0xAC, 0x2D, 0x20, 0x81, 0x91, 0xA5, 0x65,
                0xD1, 0xBC, 0x80, 0xC0, 0x41, 0x73, 0x7A, 0x76,
                0x5F, 0x54, 0x00, 0xB5, 0x2B, 0x6F, 0x52, 0x46,
                0x0F, 0xD3, 0xDC, 0x62, 0xD1, 0xAA, 0x61, 0x5F,
                0x17, 0xD7, 0xDC, 0x6B, 0xF7, 0x48, 0x58, 0xAA,
                0xEF, 0xC9, 0xED, 0xE8, 0xA5, 0xAC, 0x80, 0xB0,
                0x0A, 0xAB, 0x88, 0x09, 0xED, 0xBA, 0x84, 0x31,
                0xAF, 0x89, 0x36, 0x97, 0x92, 0xEB, 0x37, 0xCC,
                0x8B, 0xE9, 0x5F, 0x33, 0x8D, 0xE0, 0xD5, 0xE0,
                0x16, 0x5E, 0xF3, 0x47, 0x02, 0xEE, 0x7C, 0x3D,
                0xC9, 0xEF, 0x73, 0x31, 0x9C, 0xE2, 0xEB, 0x0F,
                0xD5, 0x88, 0xE4, 0x74, 0x01, 0x0B, 0xC9, 0x27,
                0xD8, 0xB5, 0xCB, 0xE8, 0x25, 0xDE, 0xF7, 0x0A,
                0xFC, 0xB8, 0x96, 0x36, 0x30, 0x3D, 0x62, 0x44,
                0x50, 0xA9, 0x66, 0x57, 0x2B, 0xF4, 0xD3, 0x5E,
                0x5E, 0xF8, 0x67, 0x68, 0x95, 0xD5, 0xB2, 0x3C,
                0x82, 0x02, 0xDA, 0xE3, 0x13, 0xA1, 0x7F, 0x55,
                0x72, 0x2E, 0x2B, 0x79, 0xC3, 0x79, 0x46, 0x9E,
                0x08, 0x7C, 0x97, 0x78, 0x3B, 0x25, 0x8B, 0x6F,
                0xD4, 0x30, 0x95, 0xBD, 0xC9, 0x22, 0xBA, 0x21,
                0xDC, 0x92, 0xDD, 0x99, 0x7A, 0x2B, 0xFC, 0xA9,
                0x66, 0xF5, 0x62, 0xDA, 0x09, 0x44, 0x55, 0xB5,
                0x59, 0x77, 0xD7, 0x3C, 0x25, 0x3B, 0xAB, 0x53,
            },
            Q = new byte[]
            {
                0xC4, 0x5C, 0xFD, 0xC2, 0x75, 0xB0, 0x8E, 0x6B,
                0x0B, 0x8A, 0xCE, 0xCD, 0x4B, 0x98, 0xEA, 0x40,
                0x0C, 0xAB, 0xDB, 0x9A, 0xF9, 0x5F, 0x57, 0xA6,
                0x5B, 0x2C, 0x0D, 0xAB, 0xC6, 0xAC, 0x6E, 0x14,
                0xB9, 0x36, 0x29, 0xA3, 0x5D, 0xDA, 0x8D, 0x4E,
                0x03, 0x84, 0xF2, 0xC8, 0x69, 0x45, 0x57, 0x62,
                0x5C, 0xBD, 0xFC, 0xBE, 0x3A, 0x15, 0x71, 0xD9,
                0x80, 0x7E, 0x9D, 0x42, 0x2C, 0xE7, 0x9F, 0xD4,
                0x8D, 0x6A, 0x63, 0x0B, 0x83, 0x77, 0xAC, 0x71,
                0x9B, 0x58, 0x1C, 0x94, 0x1F, 0x00, 0x39, 0x4F,
                0x27, 0x93, 0xD5, 0x8A, 0x08, 0xF1, 0x94, 0x67,
                0x08, 0xB1, 0xA2, 0xA8, 0xC8, 0x14, 0xE8, 0x33,
                0x51, 0x07, 0xB2, 0x76, 0xA3, 0xEA, 0xCB, 0x81,
                0x1D, 0x77, 0xAE, 0x24, 0xBD, 0xD1, 0xC3, 0xD3,
                0x5E, 0xBE, 0x8C, 0xF3, 0x28, 0xA6, 0x0C, 0xB8,
                0x44, 0xDC, 0x40, 0x83, 0x12, 0x3F, 0xB2, 0xF7,
                0x0F, 0x75, 0x03, 0x42, 0x43, 0x99, 0x39, 0x47,
                0x64, 0x6D, 0x33, 0xC4, 0xFE, 0xA6, 0xB7, 0xCE,
                0xBF, 0xCC, 0xA2, 0x59, 0x9E, 0xFE, 0x5D, 0xEA,
                0x22, 0xD8, 0x68, 0x1E, 0x99, 0xD2, 0x7E, 0x76,
                0xBF, 0xEC, 0xED, 0x1A, 0x54, 0x0A, 0x98, 0x57,
                0x1C, 0x63, 0xBD, 0xE0, 0xEF, 0x96, 0x71, 0x28,
                0x83, 0x03, 0xDE, 0xC8, 0x8B, 0x6A, 0x39, 0xB9,
                0x55, 0x35, 0x6E, 0x1A, 0x5D, 0x7C, 0x25, 0x29,
                0x0D, 0x6D, 0x7B, 0xCD, 0x5C, 0x51, 0xCE, 0xE9,
                0xD1, 0x89, 0xAA, 0x1A, 0xC5, 0x8B, 0x46, 0x96,
                0x97, 0xB9, 0x09, 0xCD, 0x46, 0xFB, 0x2B, 0xCF,
                0x0C, 0x0D, 0x9F, 0xE0, 0x10, 0x83, 0xBB, 0xF0,
                0x86, 0x79, 0xCD, 0xAE, 0xCF, 0x19, 0xDF, 0x22,
                0x62, 0x29, 0x70, 0xC4, 0x14, 0x83, 0x3A, 0xED,
                0xF2, 0x5E, 0x19, 0x45, 0x33, 0xD4, 0x5F, 0x4D,
                0x94, 0x50, 0x10, 0x1F, 0xE0, 0x46, 0x79, 0xB3,
                0x0C, 0x54, 0x0E, 0x31, 0x85, 0xBF, 0xA7, 0xA0,
                0xC4, 0x6A, 0xAF, 0x00, 0xBD, 0x88, 0x83, 0x67,
                0x16, 0xE8, 0x4B, 0x1A, 0xAF, 0x5C, 0x19, 0xDA,
                0x20, 0x43, 0xB1, 0x76, 0x9B, 0x6A, 0x63, 0x36,
                0x30, 0xE7, 0x57, 0x40, 0x0C, 0xB3, 0x47, 0xCD,
                0xC7, 0xB6, 0x97, 0xC8, 0x05, 0x7C, 0x0D, 0x9B,
                0xD0, 0x34, 0xAA, 0x28, 0x29, 0x3F, 0xFB, 0x31,
                0xBC, 0x23, 0xCA, 0x0D, 0x7F, 0x9F, 0x3E, 0xCF,
                0x7E, 0x80, 0x32, 0x92, 0x29, 0xA0, 0xE7, 0xC5,
                0x3B, 0xCD, 0x08, 0x80, 0x82, 0x6C, 0x01, 0x5D,
                0x10, 0xC4, 0xB4, 0x7F, 0x08, 0xF5, 0x15, 0x4E,
                0x03, 0x91, 0xBC, 0x7F, 0xC3, 0x1B, 0xED, 0x33,
                0x3F, 0x44, 0xD6, 0x71, 0x3B, 0x1D, 0x85, 0x54,
                0x8D, 0x9B, 0x9E, 0x38, 0xB7, 0x11, 0x50, 0x68,
                0x64, 0xD7, 0xFB, 0x32, 0xB2, 0x64, 0x0C, 0xC8,
                0x8B, 0x8D, 0x33, 0x9D, 0x9A, 0x44, 0x78, 0xE6,
                0x08, 0x44, 0x0D, 0x62, 0x56, 0x85, 0x18, 0x9A,
                0x48, 0x38, 0x48, 0xA3, 0x7A, 0xBE, 0xC5, 0x1A,
                0xC6, 0x01, 0xE4, 0x62, 0xE7, 0x98, 0x4C, 0xB4,
                0x3F, 0x73, 0xC5, 0xC8, 0x09, 0x73, 0x32, 0xFC,
                0x95, 0x19, 0x41, 0x94, 0xAE, 0x0B, 0xA0, 0x52,
                0x31, 0xA9, 0x3E, 0x71, 0x22, 0xB7, 0x13, 0x1D,
                0x1E, 0xDA, 0xAD, 0x99, 0x46, 0x35, 0x1E, 0xC5,
                0x90, 0x52, 0x82, 0x37, 0x3A, 0xA5, 0x49, 0xEF,
                0xA7, 0x6E, 0x81, 0x00, 0x4B, 0x4A, 0xEB, 0x65,
                0xCB, 0xFD, 0x2E, 0x4E, 0x83, 0x2C, 0xAB, 0x07,
                0xD3, 0x15, 0x5B, 0xFB, 0xC4, 0x55, 0xDE, 0x93,
                0x9A, 0x9E, 0xAC, 0x2F, 0x22, 0x24, 0x73, 0xE6,
                0x58, 0xF0, 0x83, 0xA8, 0x78, 0x07, 0x29, 0x93,
                0x17, 0x62, 0x83, 0xCB, 0x7F, 0x08, 0xD7, 0x3A,
                0xA8, 0x32, 0xD9, 0xDC, 0x75, 0xB4, 0x09, 0xF0,
                0x5D, 0x77, 0x42, 0x4D, 0xA1, 0xFF, 0xEF, 0x5C,
                0xF1, 0x7F, 0xE4, 0x8C, 0x33, 0x1B, 0x94, 0x44,
                0xB7, 0xD9, 0x87, 0x23, 0xDA, 0x6D, 0xD2, 0x0C,
                0x0D, 0x93, 0x2C, 0xB9, 0x0B, 0xEB, 0x73, 0x4A,
                0xDA, 0xFE, 0xF0, 0x86, 0xB4, 0x6B, 0xF8, 0x63,
                0x1E, 0x83, 0x4F, 0x33, 0xE7, 0xF7, 0x25, 0x86,
                0x39, 0x99, 0x73, 0x68, 0xED, 0xBB, 0x2C, 0x2E,
                0x76, 0x45, 0x63, 0x89, 0x2A, 0x49, 0x29, 0x0E,
                0x2A, 0xDD, 0x40, 0xB9, 0xFD, 0x58, 0x9B, 0x17,
                0xA1, 0x63, 0x5F, 0xA3, 0x94, 0x04, 0x9B, 0xD2,
                0x3A, 0xFC, 0x86, 0x0C, 0xBE, 0x13, 0x05, 0xA0,
                0xDE, 0xD1, 0x9D, 0x4D, 0xC5, 0x09, 0x46, 0x24,
                0x94, 0x37, 0x6B, 0x85, 0x1C, 0xAA, 0xE9, 0x98,
                0xBD, 0xDA, 0x59, 0xC9, 0xA9, 0xF2, 0x8B, 0x23,
                0x67, 0x54, 0xCA, 0x6B, 0x2D, 0x64, 0xCB, 0x69,
                0x72, 0xC2, 0x61, 0x60, 0x95, 0x2C, 0x07, 0x32,
                0x23, 0xA6, 0x0A, 0x18, 0x17, 0x37, 0x54, 0x18,
                0xA9, 0x69, 0xB3, 0xDE, 0xF7, 0xBB, 0xAD, 0xA4,
                0x23, 0x2A, 0x63, 0xAC, 0x3A, 0xC1, 0x37, 0x8A,
                0xB5, 0xEE, 0x9F, 0x25, 0xD9, 0x62, 0xF2, 0x53,
                0x28, 0xC8, 0xD0, 0xBC, 0x42, 0xB6, 0x3E, 0x58,
                0xA0, 0x6A, 0xA4, 0x42, 0xBB, 0x71, 0x8A, 0xEF,
                0x73, 0xA2, 0x37, 0xBF, 0xFF, 0x5D, 0x59, 0x3E,
                0x05, 0xC0, 0x82, 0x85, 0xB6, 0x72, 0xC9, 0x60,
                0xD8, 0xE8, 0x69, 0xB0, 0x18, 0xFC, 0x9A, 0xC8,
                0xE0, 0x0A, 0x40, 0xD6, 0x4C, 0x64, 0x0D, 0xA3,
                0xE8, 0x20, 0xFB, 0x9E, 0xEB, 0xCE, 0x49, 0xDB,
                0x74, 0x3A, 0xC9, 0xCB, 0x6D, 0xD4, 0x82, 0x1F,
                0x49, 0xD5, 0xB7, 0x37, 0x3B, 0x2D, 0x81, 0x23,
                0x28, 0x68, 0xB9, 0xA6, 0x18, 0x25, 0xA4, 0x5D,
                0xA1, 0x69, 0xA2, 0x6B, 0x2B, 0x01, 0xFE, 0x93,
                0x62, 0xEA, 0x0D, 0x73, 0x02, 0x1F, 0x64, 0x5A,
                0x82, 0x7A, 0x94, 0x84, 0xC1, 0xCA, 0xC5, 0x5C,
                0x17, 0x0D, 0x74, 0xB9, 0x5B, 0xAB, 0x6F, 0xC4,
                0xAE, 0x94, 0x4E, 0x41, 0xDE, 0x62, 0x10, 0xB8,
                0x2E, 0xF1, 0x2D, 0xAC, 0x37, 0x56, 0x33, 0x08,
                0x7C, 0x58, 0xC3, 0xD5, 0xFC, 0x6B, 0xDD, 0x9E,
                0x6C, 0x80, 0xB1, 0xB8, 0xE8, 0x54, 0x04, 0x07,
                0x54, 0x9C, 0x9A, 0x28, 0x89, 0xF9, 0xE0, 0x77,
                0xF1, 0xED, 0x92, 0xE3, 0x26, 0xEA, 0x0F, 0xF2,
                0xDD, 0x03, 0xCB, 0x7B, 0x42, 0x3F, 0xAD, 0x6D,
                0x19, 0x18, 0x3B, 0x5E, 0x8E, 0xA5, 0x33, 0xB1,
                0x69, 0x88, 0x3F, 0x5E, 0x5A, 0x3F, 0xE5, 0xA6,
                0xF5, 0x9F, 0x2C, 0xE7, 0x88, 0xC3, 0xFB, 0x61,
                0x53, 0xAA, 0x0A, 0xD5, 0xD1, 0xB6, 0x88, 0x3C,
                0x70, 0x72, 0x01, 0x20, 0x4D, 0x37, 0x06, 0x11,
                0x1F, 0x0A, 0x74, 0xC6, 0x36, 0xAF, 0xC9, 0x76,
                0x38, 0x34, 0xB6, 0x32, 0x88, 0x4B, 0xC5, 0x1A,
                0xD9, 0x69, 0x51, 0xBC, 0xDB, 0xA2, 0x05, 0x79,
                0x17, 0x46, 0x67, 0xB1, 0xD3, 0xCE, 0xAE, 0x17,
                0x4E, 0x4F, 0x36, 0xBF, 0x3F, 0x86, 0x75, 0xAC,
                0x1C, 0xAE, 0x74, 0x60, 0xFF, 0x4D, 0xF0, 0xCE,
                0x41, 0xA4, 0x95, 0x4E, 0x90, 0x7D, 0xAC, 0x97,
                0xF7, 0x28, 0x78, 0x9E, 0x92, 0x27, 0x76, 0x98,
                0x6F, 0x30, 0xF0, 0x75, 0x2D, 0x14, 0xB2, 0xD7,
                0x27, 0xCF, 0x48, 0xF4, 0xC0, 0xEC, 0xF9, 0xFD,
                0x57, 0xC2, 0x6A, 0xC0, 0x6A, 0xA4, 0x33, 0x7C,
                0x42, 0x4E, 0xC8, 0xF4, 0x5A, 0x66, 0x08, 0xA0,
                0x47, 0x55, 0x2A, 0x66, 0x25, 0x3A, 0x69, 0xA3,
                0x17, 0xB4, 0x84, 0xCC, 0xEF, 0x89, 0x78, 0x68,
                0xA0, 0xFF, 0xE7, 0x60, 0x52, 0x61, 0xA4, 0xAA,
                0x8D, 0x71, 0xDF, 0x17, 0x47, 0x8B, 0xB8, 0xCB,
                0xDF, 0xEE, 0x5D, 0xA2, 0x1F, 0xA3, 0x98, 0x3E,
                0xC3, 0xA1, 0x2F, 0x33, 0xEC, 0xDD, 0xAE, 0x8A,
                0x8D, 0x0C, 0x34, 0x3A, 0x44, 0x2D, 0x7F, 0x8D,
            },
            DQ = new byte[]
            {
                0x0D, 0x79, 0x58, 0x0C, 0x4C, 0xE9, 0x15, 0x8C,
                0xB0, 0xD9, 0x10, 0x81, 0xB3, 0xCB, 0x45, 0x5F,
                0xA9, 0xBE, 0xED, 0x2D, 0xC0, 0x28, 0xD3, 0xA9,
                0xDD, 0x9D, 0xB3, 0x3E, 0x73, 0x3E, 0x87, 0xBB,
                0x32, 0x4E, 0x4E, 0x23, 0x20, 0xA0, 0x8B, 0x8B,
                0xAB, 0xE0, 0x26, 0x8C, 0xAB, 0xF4, 0x8F, 0x1F,
                0x77, 0xBF, 0xAD, 0xA5, 0x1B, 0xF5, 0x36, 0xBF,
                0xB6, 0xFA, 0x79, 0x2D, 0xFE, 0x48, 0xD2, 0x85,
                0xD2, 0x42, 0x57, 0x93, 0x85, 0xAC, 0xE3, 0x8F,
                0x54, 0x1A, 0x82, 0xB3, 0x83, 0x41, 0x0F, 0xAD,
                0xA7, 0xC8, 0x94, 0x21, 0x89, 0xA5, 0x92, 0x0A,
                0x53, 0xE5, 0x64, 0x84, 0xF2, 0x5D, 0xC4, 0xE5,
                0x28, 0x8D, 0x3F, 0xA8, 0xB6, 0x6C, 0xB9, 0x14,
                0x1E, 0x02, 0x85, 0x57, 0x8E, 0x12, 0xE3, 0xBE,
                0x10, 0x45, 0x41, 0x04, 0xBA, 0x68, 0x52, 0x7D,
                0x1E, 0x74, 0x82, 0x94, 0xBB, 0xDE, 0xD5, 0x17,
                0xF0, 0xDE, 0x95, 0x9F, 0xA9, 0x65, 0xCD, 0x31,
                0x61, 0xE9, 0xC0, 0x60, 0xA7, 0x1C, 0xA7, 0x86,
                0x2F, 0x51, 0x0A, 0x5E, 0xDD, 0xF3, 0x14, 0x5C,
                0xA9, 0x91, 0x71, 0xEB, 0x8F, 0xA0, 0x8A, 0xFE,
                0xF9, 0x02, 0x77, 0xEE, 0x93, 0x8F, 0xBA, 0x8E,
                0x57, 0xAB, 0x5C, 0x6F, 0x1F, 0xE1, 0x91, 0xD8,
                0x36, 0xCD, 0x40, 0x2F, 0x40, 0xA9, 0xC4, 0x56,
                0x3C, 0x4B, 0x93, 0x47, 0x89, 0xDC, 0xA7, 0xEC,
                0x1E, 0x38, 0xC2, 0x03, 0x00, 0x6F, 0xB8, 0xA4,
                0x00, 0xB5, 0xD2, 0x8F, 0x4D, 0xB8, 0xD5, 0xDA,
                0x25, 0x85, 0x13, 0xF0, 0x1B, 0x0B, 0xC7, 0x20,
                0xC8, 0xF1, 0xF2, 0x63, 0x7C, 0x9E, 0x9D, 0x79,
                0xCE, 0xB0, 0x72, 0xF5, 0xA8, 0xCE, 0x5C, 0xAA,
                0x4E, 0x54, 0x0B, 0xA1, 0xD8, 0xCA, 0x7C, 0x73,
                0xB1, 0x6A, 0xD5, 0x8F, 0x13, 0x14, 0x62, 0x89,
                0xBF, 0x40, 0x93, 0x2A, 0xAC, 0xC8, 0x09, 0x37,
                0xC9, 0x03, 0xC7, 0x89, 0x8C, 0x64, 0xEF, 0x4A,
                0xAF, 0xCF, 0xA6, 0x3C, 0x85, 0xC4, 0xE5, 0x47,
                0x60, 0xA2, 0x05, 0xED, 0x49, 0xD5, 0x27, 0x0C,
                0xF9, 0xA3, 0xCB, 0x7C, 0x99, 0x03, 0x7E, 0xD5,
                0x4C, 0x1B, 0xC3, 0xB7, 0xE8, 0x67, 0x30, 0xE9,
                0x24, 0xE8, 0x19, 0x98, 0x27, 0x10, 0x31, 0x1A,
                0xDC, 0xF9, 0x90, 0x27, 0x7B, 0x55, 0x21, 0x96,
                0x73, 0x13, 0x7D, 0x9C, 0xD9, 0x82, 0x02, 0xDC,
                0x58, 0x10, 0xD1, 0xE7, 0x87, 0xA5, 0xBB, 0xE1,
                0xA3, 0xCD, 0xD8, 0xE4, 0x80, 0x8E, 0x8A, 0xB5,
                0x69, 0x1E, 0x26, 0x48, 0x85, 0x43, 0xD3, 0xF7,
                0x6B, 0x75, 0x47, 0x9A, 0xF8, 0xB7, 0x64, 0xDA,
                0x5C, 0x60, 0x0B, 0xDA, 0xEF, 0x34, 0x82, 0x5E,
                0x9F, 0xEC, 0xEA, 0xB9, 0x77, 0x8E, 0x5F, 0x97,
                0x1A, 0x3C, 0x5B, 0xC7, 0x49, 0xC8, 0x65, 0xBC,
                0x29, 0x1F, 0x42, 0x48, 0x71, 0x3A, 0x7B, 0x95,
                0x45, 0x41, 0xEE, 0x2D, 0x2E, 0x44, 0xA9, 0xC0,
                0x84, 0x6E, 0x20, 0x45, 0xDF, 0x79, 0x51, 0xAB,
                0x19, 0xA5, 0x2D, 0x97, 0xBF, 0xCE, 0x8A, 0x8C,
                0xDF, 0xC1, 0xC0, 0xF3, 0x8D, 0xFA, 0x72, 0xA8,
                0x34, 0x4E, 0xEC, 0x5A, 0x18, 0x6B, 0x41, 0xD0,
                0x02, 0x0A, 0x5B, 0xF7, 0x85, 0x6B, 0x4C, 0x8B,
                0x75, 0xFF, 0x89, 0x63, 0xF8, 0x16, 0x53, 0x0B,
                0x39, 0x70, 0xFF, 0xF0, 0x6C, 0x3C, 0xC5, 0x4B,
                0x05, 0x91, 0x26, 0x96, 0xEF, 0x93, 0xAF, 0x7D,
                0x67, 0xB6, 0xF2, 0xC3, 0x7E, 0x6A, 0xC5, 0x3D,
                0x9F, 0x35, 0x5A, 0xFF, 0x76, 0xA4, 0x71, 0xC6,
                0x6D, 0x18, 0xB0, 0x35, 0xF7, 0xC0, 0xCA, 0x97,
                0x26, 0xC9, 0x32, 0x2F, 0x90, 0x34, 0xE5, 0x9C,
                0x6B, 0x41, 0x5E, 0x4B, 0xCB, 0x66, 0xBE, 0xE6,
                0x0E, 0x7E, 0x96, 0xC6, 0x72, 0xE9, 0x2C, 0xB9,
                0x6A, 0xA0, 0x71, 0x53, 0x44, 0x67, 0x7C, 0x74,
                0x43, 0x3A, 0x04, 0x63, 0xBE, 0x6A, 0x09, 0x0D,
                0x82, 0x14, 0x12, 0x1A, 0xDA, 0xB5, 0x28, 0x71,
                0x9D, 0x48, 0xD9, 0x0B, 0xC5, 0x8E, 0x9D, 0x75,
                0xA8, 0x7B, 0x4F, 0xE3, 0xDE, 0x63, 0x7E, 0x42,
                0xC6, 0xE8, 0x39, 0xBA, 0x15, 0x13, 0xB7, 0x66,
                0x73, 0x73, 0x5D, 0x20, 0xF9, 0x17, 0x1B, 0xDC,
                0x00, 0x4F, 0x98, 0x99, 0xE3, 0xE5, 0xEB, 0x44,
                0x46, 0x9A, 0xB2, 0x03, 0x51, 0x28, 0x10, 0x54,
                0x59, 0xC5, 0xA8, 0xDD, 0x5F, 0x9D, 0xC5, 0x57,
                0x72, 0xA6, 0xBB, 0x48, 0x0A, 0x8E, 0xE1, 0x96,
                0xFD, 0x53, 0x22, 0xD9, 0x20, 0x49, 0x17, 0xA0,
                0x10, 0xEF, 0x90, 0x98, 0x2C, 0xB4, 0x69, 0x9D,
                0x0A, 0x81, 0xFE, 0x21, 0x41, 0x61, 0x1C, 0x3D,
                0x0C, 0xAD, 0x1B, 0xCA, 0xA8, 0xED, 0xBE, 0xAB,
                0x78, 0xAD, 0x6F, 0xE3, 0x21, 0xB9, 0x48, 0xB1,
                0x1F, 0x13, 0x18, 0xA4, 0x38, 0x8B, 0x9A, 0x60,
                0xAA, 0xD7, 0x4E, 0xF9, 0x60, 0xC4, 0x67, 0x10,
                0xD9, 0x3C, 0xE2, 0x64, 0x02, 0x10, 0x1A, 0x10,
                0x91, 0x95, 0x53, 0x24, 0x54, 0xBB, 0x5E, 0x67,
                0x68, 0x68, 0x07, 0x32, 0xB3, 0xDD, 0x2F, 0x80,
                0x03, 0x59, 0xED, 0xF5, 0x8A, 0x49, 0x2B, 0x40,
                0x6D, 0x0D, 0xD7, 0x87, 0x7C, 0x2D, 0x5A, 0x9F,
                0x4F, 0xE3, 0xBA, 0xD1, 0x14, 0x08, 0xE2, 0x5A,
                0x9B, 0xE7, 0xAE, 0xC0, 0xDF, 0xA6, 0x0D, 0xB4,
                0xF4, 0xF2, 0x9F, 0x2A, 0x55, 0x49, 0x94, 0x4A,
                0x07, 0xF6, 0xA3, 0x6D, 0x6F, 0x3E, 0xD3, 0x80,
                0x94, 0x84, 0x28, 0x2E, 0xA6, 0x55, 0xAB, 0xAC,
                0x1B, 0xF3, 0xE7, 0x43, 0x89, 0xF3, 0x00, 0x66,
                0x7A, 0x32, 0x5D, 0x72, 0xD1, 0x3B, 0x5C, 0x00,
                0xF4, 0xBF, 0x7B, 0xDE, 0xA0, 0xAF, 0xDD, 0x59,
                0x1D, 0xB3, 0x73, 0xC2, 0xCD, 0x85, 0x7B, 0xEC,
                0x5E, 0x01, 0xAD, 0x49, 0x42, 0xC6, 0xF8, 0x2D,
                0xEE, 0x83, 0x04, 0xDC, 0x67, 0x36, 0xE7, 0x4F,
                0x76, 0xAF, 0x44, 0x83, 0x6E, 0x90, 0x0A, 0x17,
                0xA9, 0x58, 0xF1, 0x4A, 0x5C, 0xED, 0xB4, 0x48,
                0xC3, 0xC7, 0x8F, 0x80, 0x92, 0x76, 0x90, 0x59,
                0x7B, 0x96, 0x85, 0x1B, 0x6C, 0x0D, 0xBF, 0xCD,
                0x04, 0xF8, 0x4D, 0xF7, 0xB4, 0x28, 0x0A, 0x48,
                0x10, 0xB8, 0xBD, 0x98, 0x8A, 0xA6, 0x0A, 0xFC,
                0x1E, 0x40, 0x51, 0x69, 0x1E, 0xCD, 0xA1, 0xC7,
                0xD7, 0xAD, 0xFA, 0xFC, 0xED, 0x0A, 0xD6, 0xF6,
                0x38, 0x88, 0x99, 0xD8, 0x8E, 0x96, 0xCD, 0xDA,
                0x5C, 0x06, 0x55, 0x1D, 0x7A, 0x1C, 0x00, 0x96,
                0xD8, 0x17, 0xD5, 0xF0, 0x80, 0x18, 0x67, 0x56,
                0xC6, 0x5C, 0x7C, 0x49, 0x4C, 0x23, 0x17, 0x6E,
                0x7B, 0x53, 0xCB, 0x59, 0x82, 0x8D, 0x23, 0x64,
                0xD2, 0x4C, 0x83, 0xA8, 0x0A, 0x04, 0x30, 0xFD,
                0x3B, 0xA7, 0xE7, 0x6C, 0xEF, 0x21, 0x3A, 0x00,
                0xEA, 0x20, 0xF1, 0xA2, 0x99, 0x27, 0x18, 0x79,
                0x0E, 0xED, 0x8E, 0x93, 0x26, 0xF7, 0xC2, 0x01,
                0x52, 0x62, 0x82, 0x27, 0xD9, 0xBC, 0xEE, 0x66,
                0x28, 0xC8, 0xF3, 0x59, 0xD5, 0xBE, 0x59, 0xE6,
                0x7F, 0x96, 0x2E, 0x58, 0xAB, 0x2C, 0xBC, 0x0D,
                0x8F, 0xAB, 0xAB, 0x04, 0x4F, 0x58, 0x99, 0x40,
                0xFC, 0x41, 0x53, 0xFF, 0x3F, 0x71, 0xC3, 0x61,
                0xA7, 0xF0, 0xF6, 0x44, 0x47, 0xCD, 0x06, 0x9F,
                0x63, 0x29, 0x66, 0xCE, 0xA5, 0x99, 0x54, 0xC4,
                0x32, 0xC0, 0xDC, 0x4C, 0xF0, 0x07, 0x07, 0x12,
                0xA2, 0xE0, 0xC7, 0x30, 0x0C, 0xCB, 0xC1, 0x10,
                0x64, 0xE3, 0xCA, 0x6B, 0xA8, 0x9B, 0x7C, 0xA2,
                0x1B, 0x5C, 0x91, 0xF3, 0x55, 0xFC, 0x77, 0x6C,
                0xBD, 0xEB, 0xEC, 0x4E, 0xE0, 0x9E, 0xE8, 0x26,
                0x7E, 0x64, 0x04, 0x0A, 0x2B, 0x19, 0xB7, 0x0D,
                0x58, 0xCA, 0x5A, 0x97, 0x18, 0xF0, 0x8A, 0x4D,
            },
            InverseQ = new byte[]
            {
                0x9F, 0xEA, 0xE6, 0xF8, 0x95, 0x54, 0x7E, 0x51,
                0x36, 0xC4, 0xBF, 0x4B, 0x07, 0xE2, 0x31, 0x52,
                0x67, 0xFC, 0xE5, 0xA4, 0xAB, 0x11, 0x7B, 0x72,
                0x78, 0x90, 0xD6, 0x38, 0xDB, 0x3B, 0xEA, 0x51,
                0x0B, 0xDE, 0x89, 0x8D, 0xF2, 0xB2, 0xCF, 0xE0,
                0x5A, 0xBF, 0xB3, 0x69, 0x15, 0x9F, 0x83, 0x6B,
                0xBA, 0x86, 0xE2, 0xE2, 0x63, 0xC6, 0xDA, 0x82,
                0x0D, 0xBE, 0xD5, 0x3E, 0x5D, 0xAC, 0x55, 0x21,
                0xBF, 0xFD, 0xD2, 0xBA, 0xAD, 0x62, 0x68, 0xF9,
                0x51, 0x07, 0xA4, 0xAD, 0xF2, 0x15, 0x8E, 0x4A,
                0x2C, 0xA8, 0xF7, 0xAF, 0x6E, 0x89, 0xD7, 0x88,
                0x71, 0x56, 0x1A, 0xDB, 0xD0, 0x04, 0x75, 0x65,
                0xCC, 0x83, 0xF5, 0xAA, 0x87, 0x8D, 0x8A, 0x6E,
                0x4F, 0xFF, 0x75, 0x27, 0x08, 0x0D, 0x3B, 0x49,
                0xA2, 0xB9, 0x74, 0x5A, 0xCD, 0x49, 0x8A, 0xF4,
                0xA4, 0x01, 0x42, 0xB9, 0xD1, 0x1C, 0xC1, 0xEE,
                0xAC, 0x02, 0xA4, 0x84, 0xE4, 0xFA, 0x26, 0xB3,
                0x08, 0xDA, 0x71, 0x17, 0x5E, 0x85, 0x70, 0x0B,
                0x78, 0xF3, 0xEE, 0xC1, 0x87, 0x09, 0x5F, 0x59,
                0xFF, 0x2F, 0xA0, 0x8C, 0x09, 0x4A, 0x5B, 0xD1,
                0xCD, 0xBA, 0x8A, 0x5C, 0x62, 0x86, 0x28, 0x4B,
                0xC0, 0x38, 0xA4, 0xBD, 0x3E, 0x79, 0xE8, 0x07,
                0xC1, 0xD1, 0x01, 0x00, 0xB0, 0xD4, 0x6B, 0x34,
                0xE4, 0x73, 0xBC, 0xCB, 0x59, 0x0F, 0x51, 0xB3,
                0x74, 0x86, 0x84, 0x63, 0xED, 0xA6, 0x50, 0x2A,
                0xC8, 0xCF, 0xF8, 0xC0, 0xCC, 0xFE, 0xEC, 0x95,
                0x64, 0xC4, 0x38, 0x7A, 0x07, 0xB6, 0x02, 0x1E,
                0x14, 0xD1, 0x63, 0xEC, 0x06, 0x77, 0xCE, 0xD7,
                0x36, 0x19, 0xC2, 0x00, 0x9A, 0x22, 0x23, 0x48,
                0xF6, 0xD7, 0x3E, 0xC6, 0x82, 0x64, 0xD2, 0x0D,
                0xB5, 0x3D, 0x63, 0xBC, 0x81, 0x33, 0x45, 0x29,
                0x6D, 0x72, 0x14, 0x25, 0xD5, 0xC9, 0x39, 0x04,
                0xF8, 0xB4, 0x28, 0x22, 0x7C, 0x5A, 0xF6, 0xA3,
                0x7D, 0x28, 0xFF, 0xB8, 0xCD, 0x9E, 0xD0, 0x8E,
                0x36, 0x91, 0x52, 0x43, 0x6E, 0x79, 0x0A, 0x4C,
                0x6B, 0x1B, 0x51, 0x49, 0x56, 0x9E, 0xCF, 0x93,
                0x5A, 0x2A, 0xA0, 0xBD, 0x2F, 0x54, 0xA4, 0xC8,
                0xDF, 0x77, 0x36, 0x6C, 0xBD, 0x88, 0x5A, 0x6C,
                0xDE, 0x78, 0xFF, 0x55, 0x6A, 0x82, 0xFA, 0x55,
                0x86, 0x0D, 0x7F, 0x64, 0xED, 0xCC, 0x5A, 0x40,
                0xE6, 0x36, 0x1B, 0x05, 0x2F, 0x2A, 0x29, 0x9B,
                0xF1, 0x47, 0xC0, 0x4D, 0xCA, 0xCA, 0x52, 0x55,
                0xFB, 0x33, 0xD9, 0xF9, 0x9C, 0x13, 0x9B, 0xFA,
                0x07, 0xEB, 0x7F, 0x45, 0x6E, 0xFC, 0xA4, 0xD5,
                0x62, 0x17, 0x41, 0x2F, 0xF9, 0x3D, 0x56, 0x52,
                0x4D, 0x24, 0x9F, 0x0E, 0x01, 0x1F, 0x2A, 0x87,
                0x0E, 0x4C, 0x51, 0x17, 0xFB, 0xD5, 0x85, 0x0C,
                0xA2, 0x6F, 0xAC, 0x89, 0xC0, 0x42, 0x5F, 0x98,
                0x7B, 0x27, 0xDD, 0xFD, 0xD3, 0x8A, 0x2F, 0xE5,
                0x4F, 0xE5, 0x81, 0x62, 0x59, 0x14, 0x22, 0x89,
                0x40, 0x97, 0x26, 0xB6, 0x1F, 0xA9, 0x51, 0xC9,
                0xD3, 0xF4, 0x18, 0xCB, 0xE0, 0xEA, 0xAA, 0x08,
                0xA1, 0x4B, 0x26, 0x24, 0x44, 0x9B, 0x85, 0x8F,
                0xED, 0x33, 0xB9, 0x4B, 0xD8, 0x9A, 0x32, 0x19,
                0xAB, 0x06, 0x02, 0x79, 0xC2, 0x58, 0xED, 0xF7,
                0x1E, 0x0E, 0x45, 0x05, 0x4B, 0x09, 0x11, 0x1F,
                0x22, 0x4A, 0xE3, 0xBD, 0x43, 0xAF, 0x4C, 0xD5,
                0xC6, 0x84, 0xB3, 0x5B, 0x43, 0x37, 0xF7, 0x9B,
                0xBD, 0x27, 0x49, 0x00, 0xFB, 0x5B, 0x10, 0x22,
                0x04, 0xEB, 0xA4, 0xB9, 0x55, 0x29, 0x74, 0x8A,
                0xE2, 0x73, 0x58, 0x12, 0x1B, 0xA1, 0x0C, 0x3F,
                0x23, 0x6B, 0x5C, 0x86, 0x5B, 0x5E, 0xD7, 0xBD,
                0xD3, 0x0F, 0x97, 0x6A, 0x39, 0x44, 0x4F, 0xBD,
                0xFD, 0xF6, 0xD2, 0x65, 0x73, 0x35, 0x04, 0x55,
                0xE8, 0x61, 0x5D, 0xA0, 0x39, 0xA7, 0x22, 0xE9,
                0xD2, 0x50, 0x72, 0x26, 0x60, 0x7A, 0x96, 0x69,
                0xE8, 0x03, 0x7C, 0x31, 0x03, 0xAB, 0x12, 0x60,
                0x2C, 0x27, 0x2B, 0xC4, 0xFD, 0x99, 0xCC, 0xB5,
                0xAE, 0x87, 0x67, 0x79, 0xE0, 0xC0, 0xFC, 0x42,
                0x6E, 0xF4, 0xF7, 0x72, 0x50, 0x74, 0xEE, 0xE5,
                0x31, 0x0A, 0xC0, 0xF2, 0xFA, 0x88, 0x29, 0xC3,
                0x73, 0x8B, 0xDD, 0xC2, 0x33, 0xDB, 0xD9, 0x76,
                0xD5, 0x10, 0xD9, 0x69, 0xC5, 0x91, 0xF6, 0xB3,
                0x08, 0x65, 0x3C, 0x40, 0x0F, 0x8D, 0x9A, 0x34,
                0xDE, 0xAA, 0xE3, 0xD3, 0x27, 0x26, 0xDE, 0x0D,
                0xF5, 0xC3, 0x13, 0x7F, 0xCC, 0x37, 0x3F, 0x90,
                0x76, 0xAC, 0xB3, 0xB8, 0x44, 0xA6, 0x83, 0x9D,
                0xA1, 0x56, 0xC0, 0x73, 0x0D, 0xD9, 0x23, 0xC1,
                0xA5, 0x16, 0xD1, 0x3F, 0x58, 0x85, 0x14, 0x87,
                0x9D, 0xB5, 0x6F, 0x72, 0xCE, 0x21, 0x97, 0xAB,
                0xF0, 0x62, 0x50, 0x3D, 0x52, 0xE5, 0xF3, 0x65,
                0x3B, 0xAA, 0xF8, 0x1A, 0x2E, 0x1B, 0x67, 0xF7,
                0x48, 0x52, 0xD1, 0xF1, 0x41, 0x8D, 0x99, 0xF0,
                0x0E, 0x24, 0x22, 0xC9, 0x31, 0x4C, 0xDA, 0x75,
                0x8A, 0x99, 0x44, 0xE7, 0x3A, 0x6D, 0x37, 0xD8,
                0xFE, 0x18, 0x90, 0x8E, 0xC6, 0x0D, 0x2A, 0xC5,
                0x7B, 0x7F, 0xCA, 0xBD, 0x9C, 0xFB, 0x26, 0xF3,
                0xBD, 0x8E, 0x67, 0x9A, 0x4F, 0xCE, 0x36, 0x3A,
                0x61, 0xB3, 0x83, 0x27, 0x4C, 0xE3, 0x5B, 0xF0,
                0xED, 0x58, 0x3A, 0x3A, 0x61, 0x50, 0x35, 0xD6,
                0xEC, 0xAA, 0x5C, 0x5E, 0xCF, 0x47, 0xBB, 0xAF,
                0x2B, 0xAA, 0x44, 0x81, 0x11, 0xAF, 0x8C, 0x84,
                0x69, 0x83, 0xCF, 0xC3, 0x5B, 0x0D, 0x33, 0xF5,
                0x43, 0x86, 0xD5, 0x80, 0x02, 0x19, 0xB4, 0x4A,
                0x16, 0x76, 0x55, 0x8C, 0x10, 0x37, 0x44, 0x32,
                0x28, 0x63, 0x53, 0x9E, 0x57, 0x80, 0xD4, 0x01,
                0x40, 0x00, 0x9F, 0x84, 0xA9, 0xF9, 0x9C, 0x97,
                0xBE, 0x48, 0x88, 0x31, 0x8E, 0x8A, 0x86, 0x65,
                0xCF, 0xB5, 0xDE, 0xE6, 0x43, 0x72, 0x8C, 0x0D,
                0xC9, 0x20, 0x57, 0xAF, 0xC4, 0x75, 0xCA, 0xE8,
                0xEF, 0xC0, 0xDD, 0x44, 0xD8, 0xE0, 0xD7, 0xDA,
                0xEA, 0x7B, 0x4F, 0xD9, 0x85, 0x0D, 0x7B, 0x2B,
                0x84, 0x50, 0x64, 0x4F, 0xBF, 0xA6, 0x5B, 0x59,
                0x8E, 0x1C, 0x95, 0x31, 0x34, 0x9D, 0xCB, 0xEF,
                0x40, 0x7E, 0xEE, 0x27, 0xA9, 0xBD, 0x4D, 0x4D,
                0x11, 0xB0, 0x4B, 0xDA, 0xD5, 0x46, 0x72, 0xE7,
                0x92, 0x30, 0x67, 0x6E, 0x53, 0x15, 0xF1, 0x4D,
                0xE6, 0xCD, 0x84, 0x20, 0x90, 0xC6, 0xB8, 0xC2,
                0x18, 0x93, 0x4A, 0x73, 0x98, 0xC2, 0x01, 0x58,
                0x0A, 0xEF, 0x44, 0xB3, 0x1D, 0x0F, 0xC7, 0x5B,
                0x53, 0xC0, 0xA4, 0x0F, 0x15, 0x99, 0xAF, 0x80,
                0x57, 0xD6, 0x8E, 0xA2, 0x9C, 0x37, 0xD8, 0xCD,
                0x22, 0xB7, 0x59, 0x00, 0xBE, 0xEC, 0xC7, 0x2D,
                0xB6, 0x53, 0x1E, 0x9F, 0x0E, 0xA8, 0x52, 0x01,
                0x58, 0x48, 0x73, 0x50, 0x2C, 0x4A, 0xFD, 0xC5,
                0x45, 0x2B, 0xCF, 0xB2, 0x8B, 0x1A, 0x98, 0x53,
                0xDF, 0xC4, 0x93, 0x2F, 0x77, 0xEA, 0x3C, 0xF0,
                0x68, 0xEE, 0xBE, 0x40, 0x04, 0xDD, 0x0B, 0xF9,
                0x09, 0x8D, 0x33, 0x93, 0x28, 0x04, 0x38, 0xC8,
                0x84, 0x7F, 0xAB, 0x51, 0xD6, 0xD7, 0xAB, 0x27,
                0xE0, 0x3A, 0x3C, 0x02, 0xC4, 0x62, 0xF9, 0x1B,
                0x5D, 0xC7, 0xB1, 0xBD, 0x51, 0x96, 0xE0, 0xBB,
                0xCA, 0xC2, 0x4A, 0x1B, 0x85, 0x1F, 0x37, 0xC5,
                0xF3, 0x7B, 0x87, 0xC0, 0x7B, 0x25, 0xF2, 0x48,
                0x5A, 0xCC, 0xE4, 0x1E, 0xB6, 0xE6, 0xA7, 0x04,
                0x6F, 0xC6, 0xC4, 0x17, 0x5A, 0x78, 0x82, 0x45,
                0x99, 0x3C, 0xA6, 0x2F, 0x52, 0xCE, 0xAE, 0xEE,
                0xBD, 0x98, 0x0F, 0x60, 0xC2, 0x1B, 0x85, 0x15,
            },
        };

        public static readonly RSAParameters DiminishedDPParameters = new RSAParameters
        {
            Modulus = new byte[]
            {
                0xB7, 0x3F, 0x59, 0xF5, 0xEE, 0x8B, 0xD5, 0x5E,
                0x24, 0xB7, 0xFF, 0x02, 0x9A, 0xD1, 0x6A, 0x85,
                0x43, 0xC9, 0x6D, 0x25, 0x3E, 0x54, 0x31, 0x9B,
                0x93, 0x53, 0x2C, 0x41, 0xC3, 0xC1, 0x47, 0x7B,
                0x89, 0xDB, 0x2D, 0x2F, 0x33, 0xD7, 0xB9, 0xA8,
                0x74, 0x58, 0x48, 0xE0, 0x6A, 0x8E, 0x68, 0x3B,
                0x50, 0x44, 0x51, 0x1E, 0x1B, 0xCA, 0x36, 0x25,
                0x27, 0xE1, 0x7C, 0xF4, 0x2B, 0x38, 0x06, 0x15,
            },
            Exponent = new byte[]
            {
                0x01, 0x00, 0x01,
            },
            D = new byte[]
            {
                0xAF, 0xE6, 0xF2, 0x36, 0x2F, 0x9C, 0xAF, 0x5E,
                0xC5, 0xA4, 0x91, 0xF8, 0x30, 0x21, 0x22, 0x3D,
                0x76, 0x8A, 0x9E, 0x69, 0x07, 0xE1, 0xCE, 0x14,
                0xE7, 0x61, 0x09, 0xB4, 0xBF, 0x72, 0x83, 0x68,
                0x22, 0x8C, 0x1A, 0xE9, 0x62, 0x46, 0xAD, 0xDD,
                0xCD, 0xA7, 0x59, 0xBD, 0x04, 0x18, 0x68, 0xB9,
                0x69, 0x68, 0x17, 0xC5, 0x01, 0xE8, 0x8B, 0xBC,
                0xFD, 0xEF, 0xF0, 0x02, 0x0F, 0x85, 0xF7, 0x29,
            },
            P = new byte[]
            {
                0xF3, 0x72, 0x7B, 0x1E, 0x86, 0x76, 0x20, 0x55,
                0x94, 0xD3, 0xC7, 0x3A, 0x02, 0x27, 0x4D, 0x4F,
                0x1A, 0x98, 0x04, 0x4A, 0x29, 0x51, 0xC8, 0x5C,
                0xCD, 0x12, 0xC6, 0xFC, 0x57, 0xAD, 0x19, 0x7B,
            },
            DP = new byte[]
            {
                // Note the leading 0x00 byte.
                0x00, 0x08, 0x8E, 0xFD, 0xC5, 0x14, 0xF5, 0x12,
                0x2D, 0xF0, 0x0D, 0x81, 0xF3, 0x88, 0x1F, 0xD9,
                0x97, 0xEE, 0x57, 0x69, 0xCF, 0x31, 0xA4, 0xAE,
                0x66, 0x94, 0xCF, 0x14, 0x2F, 0xCA, 0xE5, 0x4B,
            },
            Q = new byte[]
            {
                0xC0, 0xB2, 0x3A, 0xE8, 0xEA, 0x52, 0xA9, 0xFB,
                0x43, 0x4E, 0xFD, 0x4A, 0x51, 0xF3, 0x5E, 0xEB,
                0xE8, 0x72, 0xA2, 0x1D, 0xB7, 0x82, 0x0C, 0xD4,
                0x49, 0x88, 0x96, 0xB9, 0x54, 0xF4, 0x61, 0xAF,
            },
            DQ = new byte[]
            {
                0xAC, 0x24, 0xCC, 0xF1, 0xD4, 0x9B, 0xA2, 0x95,
                0x00, 0x0D, 0x69, 0xC3, 0xE2, 0x30, 0x2B, 0x85,
                0x4E, 0x74, 0x52, 0x15, 0x80, 0x21, 0xA3, 0x3A,
                0x66, 0xB2, 0xAA, 0x0B, 0xC9, 0x34, 0x44, 0xAB,
            },
            InverseQ = new byte[]
            {
                0xC2, 0xC9, 0x95, 0x94, 0xC6, 0x8C, 0x40, 0x76,
                0x37, 0xEB, 0x04, 0x6B, 0x31, 0xF9, 0x4E, 0x81,
                0x1C, 0xCD, 0x0C, 0xCA, 0xAA, 0x9E, 0xED, 0xF6,
                0x3B, 0x86, 0x35, 0xB3, 0x8F, 0x86, 0x81, 0x0B,
            }
        };

        public static readonly RSAParameters UnusualExponentParameters = new RSAParameters()
        {
            Modulus = new byte[]
            {
                0xF6, 0xBA, 0x82, 0x83, 0x26, 0x0C, 0x39, 0x91,
                0x1B, 0x8B, 0x5D, 0xB1, 0x8A, 0x0F, 0xF3, 0x6A,
                0x78, 0xD5, 0x59, 0xA8, 0x0D, 0x64, 0x29, 0x3D,
                0xD0, 0x0C, 0x35, 0x87, 0x56, 0x00, 0x9B, 0x3C,
                0xE8, 0x91, 0xE1, 0xC2, 0x08, 0xAE, 0xDB, 0x9C,
                0x15, 0xAB, 0xB5, 0x24, 0x94, 0x10, 0x08, 0xF7,
                0x53, 0xCE, 0xD7, 0x7C, 0xCF, 0x75, 0xCF, 0x17,
                0x45, 0x3F, 0x4C, 0xD1, 0x02, 0x92, 0x11, 0xCB,
                0x31, 0xDF, 0xB5, 0xED, 0x6B, 0x23, 0x8F, 0x8D,
                0x96, 0x37, 0x8E, 0x1A, 0x81, 0x20, 0x71, 0x49,
                0x17, 0x05, 0xE0, 0x43, 0x1D, 0xA4, 0xD7, 0x7B,
                0xB9, 0x99, 0x0A, 0xA9, 0x0B, 0x2F, 0x80, 0x89,
                0x9B, 0xF1, 0x79, 0xDA, 0xC9, 0x50, 0xF6, 0xD5,
                0x2D, 0xBC, 0xBF, 0xAF, 0xDA, 0x2D, 0xEF, 0x2A,
                0x35, 0x29, 0xC5, 0xBC, 0x88, 0x32, 0xE5, 0xAD,
                0x4E, 0x5C, 0x8F, 0x5C, 0xD0, 0x1E, 0x8E, 0x93,
            },
            Exponent = new byte[]
            {
                0x01, 0xB1,
            },
            D = new byte[]
            {
                0x7D, 0x5B, 0xCE, 0x6E, 0x62, 0x8E, 0x31, 0x59,
                0xB0, 0x94, 0xD9, 0xE0, 0x69, 0x9E, 0xDD, 0xD1,
                0x96, 0xAB, 0x11, 0xC3, 0xF1, 0x85, 0x11, 0xFF,
                0x7A, 0xD9, 0xDC, 0x86, 0xFA, 0x9F, 0xF0, 0x47,
                0x26, 0x59, 0x7D, 0xEF, 0xE3, 0x4D, 0x9B, 0xEB,
                0xFA, 0x74, 0xCD, 0x8C, 0xF7, 0xDD, 0x94, 0x39,
                0x14, 0xB4, 0xC4, 0xFC, 0x9B, 0x11, 0xE1, 0x3C,
                0xE5, 0x1A, 0xD7, 0x36, 0xC2, 0x0B, 0x8B, 0xB2,
                0x82, 0x93, 0x62, 0x80, 0x02, 0x30, 0xAF, 0x15,
                0x9E, 0x5A, 0x39, 0x7C, 0x6F, 0xCA, 0x09, 0xC9,
                0xD8, 0xC5, 0x21, 0x88, 0x8D, 0x52, 0xEE, 0x3A,
                0x50, 0x4D, 0xB3, 0xFA, 0xA0, 0x88, 0x0D, 0x67,
                0xDE, 0x9D, 0x68, 0x32, 0x03, 0xC8, 0x35, 0xCE,
                0x73, 0x38, 0x19, 0xED, 0x38, 0xFE, 0xD2, 0x5C,
                0xD6, 0x12, 0xF0, 0x17, 0x33, 0x99, 0x0D, 0x1F,
                0xFB, 0x3D, 0xA1, 0x35, 0x24, 0x03, 0x16, 0xB1,
            },
            P = new byte[]
            {
                0xFE, 0xF8, 0x94, 0xF4, 0xC5, 0x2D, 0x9A, 0xA9,
                0xA5, 0x40, 0x6E, 0x27, 0xE9, 0x27, 0x46, 0xCF,
                0x29, 0xB4, 0xBD, 0x93, 0xE1, 0x99, 0x29, 0xA5,
                0xDA, 0x8B, 0x76, 0x28, 0xE3, 0xD1, 0x84, 0xFF,
                0x00, 0x19, 0xFD, 0xD3, 0x8C, 0x41, 0xDE, 0xF9,
                0x63, 0xC6, 0x7C, 0x85, 0x5A, 0x70, 0x37, 0x6F,
                0x6D, 0x9C, 0x96, 0x4A, 0xD8, 0x0C, 0x37, 0x1D,
                0x04, 0xB4, 0xAB, 0x34, 0x41, 0xC0, 0x72, 0x8D,
            },
            DP = new byte[]
            {
                0x74, 0x00, 0xC3, 0x79, 0x69, 0xAC, 0x1A, 0x06,
                0x3C, 0x67, 0x37, 0x70, 0x29, 0xA2, 0x20, 0xCE,
                0x95, 0xA2, 0xA3, 0x1C, 0x42, 0x93, 0x22, 0x51,
                0xF6, 0x0D, 0xC9, 0x90, 0x88, 0xC2, 0x0E, 0xFB,
                0xFF, 0x74, 0x78, 0xCD, 0x9F, 0x97, 0x2B, 0x81,
                0x6D, 0x3F, 0x1B, 0xAE, 0xC7, 0x00, 0xCC, 0xF4,
                0x06, 0xB5, 0xCC, 0xF3, 0x58, 0x3E, 0x50, 0xA6,
                0x54, 0x52, 0x32, 0xB2, 0x15, 0xA3, 0x3B, 0xCD,
            },
            Q = new byte[]
            {
                0xF7, 0xB9, 0x69, 0x93, 0xFA, 0x16, 0x23, 0x46,
                0x31, 0x27, 0x4A, 0xBB, 0x5A, 0x34, 0xD3, 0xB8,
                0x4A, 0xD1, 0xCC, 0xE7, 0x21, 0x3D, 0x66, 0xEC,
                0x68, 0x90, 0x92, 0xD9, 0xDB, 0x1F, 0x01, 0xBD,
                0x02, 0xC9, 0x3E, 0x14, 0x82, 0x6A, 0x47, 0x8E,
                0xC4, 0x47, 0xD8, 0x48, 0x79, 0x74, 0x66, 0x7F,
                0x68, 0x08, 0x7E, 0x77, 0x39, 0x24, 0x80, 0xF3,
                0x16, 0x83, 0x89, 0xC8, 0x1C, 0x6F, 0x4D, 0x9F,
            },
            DQ = new byte[]
            {
                0x32, 0xEA, 0xFC, 0xDE, 0x90, 0x39, 0xC2, 0xAB,
                0x1A, 0x10, 0xF1, 0xCC, 0xA4, 0x92, 0xD6, 0xF8,
                0xF2, 0x68, 0x9C, 0x38, 0xF7, 0x75, 0xDB, 0xCE,
                0x72, 0xE7, 0xEA, 0x28, 0x0C, 0x85, 0x7C, 0x83,
                0xAC, 0x07, 0x12, 0xAC, 0x1F, 0x89, 0x22, 0x37,
                0xF3, 0x22, 0x47, 0x0F, 0x7C, 0xE1, 0x88, 0x5B,
                0x38, 0xDB, 0x50, 0xFA, 0x5A, 0x60, 0xC7, 0x24,
                0x5D, 0xE7, 0x02, 0x4E, 0x60, 0xE4, 0x9F, 0x9F,
            },
            InverseQ = new byte[]
            {
                0xB1, 0xE1, 0x44, 0xDD, 0xFB, 0x21, 0xA7, 0xF8,
                0x54, 0x8E, 0x05, 0x1D, 0x11, 0x2B, 0xC3, 0xE3,
                0x56, 0x29, 0x17, 0xEB, 0xD9, 0x9B, 0x3D, 0xAA,
                0xBA, 0x8E, 0x55, 0x68, 0x00, 0xD8, 0x10, 0x04,
                0xD6, 0x53, 0x8D, 0xE4, 0xBE, 0x81, 0xF6, 0x20,
                0x73, 0xF3, 0x7C, 0xAB, 0xB4, 0x61, 0x2D, 0xD8,
                0x81, 0x32, 0x0C, 0x1C, 0xFD, 0xCB, 0xB0, 0xAB,
                0x22, 0x5F, 0x7B, 0x41, 0xD8, 0x32, 0x59, 0xA3,
            },
        };

        internal static readonly RSAParameters RsaBigExponentParams = new RSAParameters
        {
            Modulus = (
                        "AF81C1CBD8203F624A539ED6608175372393A2837D4890E48A19DED369731156" +
                        "20968D6BE0D3DAA38AA777BE02EE0B6B93B724E8DCC12B632B4FA80BBC925BCE" +
                        "624F4CA7CC606306B39403E28C932D24DD546FFE4EF6A37F10770B2215EA8CBB" +
                        "5BF427E8C4D89B79EB338375100C5F83E55DE9B4466DDFBEEE42539AEF33EF18" +
                        "7B7760C3B1A1B2103C2D8144564A0C1039A09C85CF6B5974EB516FC8D6623C94" +
                        "AE3A5A0BB3B4C792957D432391566CF3E2A52AFB0C142B9E0681B8972671AF2B" +
                        "82DD390A39B939CF719568687E4990A63050CA7768DCD6B378842F18FDB1F6D9" +
                        "FF096BAF7BEB98DCF930D66FCFD503F58D41BFF46212E24E3AFC45EA42BD8847").HexToByteArray(),
            Exponent = new byte[] { 0x02, 0x00, 0x00, 0x04, 0x41 },
            D = (
                        "64AF9BA5262483DA92B53F13439FD0EF13012F879ABC03CB7C06F1209904F352" +
                        "C1F223519DC48BFAEEBB511B0D955F6167B50E034FEA2ABC590B4EA9FBF0C51F" +
                        "9FFEA16F7927AE681CBF7358452BCA29D58705E0CAA106013B09A6F5F5911498" +
                        "D2C4FD6915585488E5F3AD89836C93C8775AFAB4D13C2014266BE8EE6B8AA66C" +
                        "9E942D493466C8E3A370F8E6378CE95D637E03673670BE4BCACE5FCDADD238D9" +
                        "F32CA35DE845776AC4BF36118812328C493F91C25A9BD42672D0AFAFDE0AF7E6" +
                        "19078D48B485EF91933DDCFFB54587B8F512D223C81894E91784982F3C5C6587" +
                        "1351F4655AB023C4AD99B6B03A96F9046CE124A471E828F05F8DB3BC7CCCF2D1").HexToByteArray(),
            P = (
                        "E43A3826A97204AE3CD8649A84DB4BBF0725C4B08F8C43840557A0CD04E313AF" +
                        "6D0460DDE69CDC508AD043D72514DA7A66BC918CD9624F485644B9DEEAB2BE0E" +
                        "112956D472CF0FD51F80FD33872D2DCC562A0588B012E8C90CE7D254B94792C6" +
                        "E7A02B3CCAA150E67A64377ACC49479AD5EB555493B2100CB0410956F7D73BF5").HexToByteArray(),
            Q = (
                        "C4DD2D7ADD6CA50740D3973F40C4DEBDBAB51F7F5181ABAE726C32596A3EDD0A" +
                        "EE44DAADDD8A9B7A864C4FFDAE00C4CB1F10177BA01C0466F812D522610F8C45" +
                        "43F1C3EF579FA9E13AE8DA1A4A8DAE307861D2CEAC03560279B61B6514989883" +
                        "FE86C5C7420D312838FC2F70BED59B5229654201882664CEFA38B48A3723E9CB").HexToByteArray(),
            DP = (
                        "09ECF151F5CDD2C9E6E52682364FA5B4ED094F622E4031BF46B851358A584DCC" +
                        "B5328B0BD9B63589183F491593D2A3ACAD14E0AACDA1F181B5C7D93C57ED26E6" +
                        "2C9FC26AF37E4A0644ECE82A7BA8AED88FF1D8E9C56CC66385CDB244EB3D57D1" +
                        "7E6AD420B19C9E2BEE18192B816265B74DA55FA3825F922D9D8E835B76BF3071").HexToByteArray(),
            DQ = (
                        "89B33B695789174B88368C494639D4D3267224572A40B2FE61910384228E3DBD" +
                        "11EED9040CD03977E9E0D7FC8BFC4BF4A93283529FF1D96590B18F4EABEF0303" +
                        "794F293E88DC761B3E23AFECB19F29F8A4D2A9058B714CF3F4D10733F13EA72B" +
                        "BF1FBEC8D71E106D0CE2115F3AD2DE020325C3879A091C413CD6397F83B3CB89").HexToByteArray(),
            InverseQ = (
                        "7C57ED74C9176FBA76C23183202515062C664D4D49FF3E037047A309DA10F159" +
                        "0CE01B7A1CD1A4326DC75883DFF93110AB065AAED140C9B98176A8810809ADEC" +
                        "75E86764A0951597EF467FA8FD509181CD2E491E43BE41084E5BE1B562EE76E9" +
                        "F92C9AB1E5AEAD9D291A6337E4DE85BDE67A0D72B4E55ADCF207F7A5A5225E15").HexToByteArray()
        };

        public static readonly RSAParameters CspTestKey = new RSAParameters
        {
            Modulus = ByteUtils.HexToByteArray("e06aac9ec3e98bae9ebaf921eb7898f34a58a6a4f6370a9d767cd1f0492e7969b4defdb11b1795a63fefb3359b55c392ecb22f5791e1d925ea5cf74bc5094ddc0164ebb021028423151c7641181940112b0d46e9562d3cec8364d58be8d9c84910e196fa458f633cf6431354df98b773c32c0cf6d18147222a96824b64019ae1"),
            Exponent = ByteUtils.HexToByteArray("010001"),
            D = ByteUtils.HexToByteArray("143d3aa534e8feaa7871475faa435d93ef7c104767572e736608bacc4f654c18def18f72a60d59f73ce3eac72663b5382e75a17465d93702c6e0ac82de59c8f627b01b1bc02b0aa98925b4a010d2c5c563544daeabf148997d8d016b63fa3ce05b3788c5ae9ba0d5ea9b990804f40ac7ebbe62f8b9b884c154a0f8628b00ac43"),
            P = ByteUtils.HexToByteArray("e1aab100887245692770c5059cf3b6f2dabb83b015c61a229806e298a79bd360609d4b5894a1c231c9b47fd7b7a4f1a44b3870acf80373b484e5296e9f3ab47b"),
            DP = ByteUtils.HexToByteArray("4966e0fe0063d2e9fa37370eb5579ca96fb6508644fed3df6ebdc694cae7e7a050acb9264dea33a5482b9aedcac12f0c369f5c1f16e8e088d63547fdc07332e3"),
            Q = ByteUtils.HexToByteArray("fe94f7aa687940d862b0f6f44165656bf81acc5790a9d065624dd0f9d239d39e77d5c3038a317593ce7b24f31e76ce2654ca3cccf878a12088ae8d87b5111553"),
            DQ = ByteUtils.HexToByteArray("6c501eeb1e95f013e03160705d5e717f3548d985abe3c3e94ea0c2f7770ce94f33b6fbc886c4323d178d671414f3011467e0bf6b898f71263160ea9041662a47"),
            InverseQ = ByteUtils.HexToByteArray("97d9b81076a4b08a1427168b3deacfb3d65d2a5ce23e098671cd1150882161f3911b60e02f6ebbc9a5009d06ef50f2c51ed2da8f787c5b7d63bc7bc0fe1cf75a"),
       };
    }
}
