/*
cdcd - Command Driven CD player
Copyright (C) 1998-99 Tony Arcieri
Copyright (C) 2001, 2002, 2003 Fabrice Bauzac

This program is free software; you can redistribute it and/or modify
it under the terms of the GNU General Public License as published by
the Free Software Foundation; either version 2 of the License, or
(at your option) any later version.

This program is distributed in the hope that it will be useful,
but WITHOUT ANY WARRANTY; without even the implied warranty of
MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
GNU General Public License for more details.

You should have received a copy of the GNU General Public License
along with this program; if not, write to the Free Software
Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
*/

#include <stdio.h>
#include <string.h>
#include <sys/types.h>
#include <sys/socket.h>
#include <unistd.h>

#include "mycdaudio.h"
#include "global.h"
#include "config.h"
#include "interface.h"

/* This function isn't declared in cdaudio.h yet...  */
int cdindex_http_submit (int cd_desc, struct cddb_host host,
			 struct cddb_server *proxy);

int
cddb_get_siteslist (struct cddb_serverlist *list, struct cddb_host host)
{
  int sock;
  struct cddb_hello hello;
  char http_string[512];

  strncpy (hello.hello_program, PACKAGE, 256);
  strncpy (hello.hello_version, VERSION, 256);

  switch (host.host_protocol)
    {
    case CDDB_MODE_CDDBP:
      printf ("Connecting to cddbp://%s:%d/\n",
	      host.host_server.server_name, host.host_server.server_port);
      if ((sock = cddb_connect_server (host, NULL, hello)) < 0)
	{
	  perror ("Couldn't connect to CDDB server!");
	  return -1;
	}
      puts ("Connection established.\n" "Downloading server list...");
      if (cddb_sites (sock, CDDB_MODE_CDDBP, list) < 0)
	{
	  fprintf (stderr, "CDDB error: %s\n", cddb_message);
	  return -1;
	}
      puts ("Done.");
      break;

    case CDDB_MODE_HTTP:
      printf ("Connecting to http://%s:%d/%s\n",
	      host.host_server.server_name,
	      host.host_server.server_port, host.host_addressing);
      if ((sock = cddb_connect_server (host, NULL, hello,
				       http_string, 512)) < 0)
	{
	  perror ("Couldn't connect to CDDB server!");
	  return -1;
	}
      puts ("Connection established.\n" "Downloading server list...");
      if (cddb_sites (sock, CDDB_MODE_HTTP, list, http_string) < 0)
	{
	  fprintf (stderr, "CDDB error: %s\n", cddb_message);
	  return -1;
	}
      puts ("Done.");
    }

  return 0;
}

int
cdcd_cd_stat (int cd_desc, struct disc_info *disc)
{
  cd_stat (cd_desc, disc);
  if (!disc->disc_present)
    {
      cd_close (cd_desc);
      cd_stat (cd_desc, disc);
      if (!disc->disc_present)
	{
	  puts ("No disc in drive");
	  return -1;
	}
    }

  return 0;
}

/* FIXME: cddb_lookup does its job (fill DATA) only the first time it
   is called.  This function fixes it, until I understand
   cddb_lookup's behaviour. */
void
lookup_now (int cd_desc, struct disc_data *data)
{
  static struct disc_data sd;
  cddb_lookup (cd_desc, &sd);
  memcpy (data, &sd, sizeof (sd));
}

/* CDDB processor */
void
cddb_lookup (int cd_desc, struct disc_data *data)
{
  int index, serverindex, selection, sock = -1;
  struct disc_info disc;
  struct disc_data outdata;
  struct cddb_conf conf;
  struct cddb_serverlist list;
  struct cddb_server *proxy;
  struct cddb_entry entry;
  struct cddb_hello hello;
  struct cddb_query query;
  char inbuffer[256], http_string[512], discid[CDINDEX_ID_SIZE];

  if (cdcd_cd_stat (cd_desc, &disc) < 0)
    return;

  if (!verbosity)
    cddb_read_disc_data (cd_desc, data);
  else
    {
      cddb_stat_disc_data (cd_desc, &entry);

      if (entry.entry_present)
	{
	  if (entry.entry_timestamp == timestamp
	      && entry.entry_id == timestamped_discid)
	    return;

	  cddb_read_disc_data (cd_desc, data);
	  timestamp = entry.entry_timestamp;
	  timestamped_discid = entry.entry_id;
	}
      else
	{
	  proxy = ((struct cddb_server *)
		   xmalloc (sizeof (struct cddb_server)));
	  cddb_read_serverlist (&conf, &list, proxy);
	  if (conf.conf_access == CDDB_ACCESS_LOCAL)
	    {
	      free (proxy);
	      cddb_generate_unknown_entry (cd_desc, data);
	      return;
	    }
	  if (list.list_len < 1)
	    {
	      puts ("Couldn't perform CDDB/CD Index lookup: "
		    "No servers available!");
	      return;
	    }
	  if (!conf.conf_proxy)
	    {
	      free (proxy);
	      proxy = NULL;
	    }
	  else
	    {
	      printf ("Using proxy http://%s:%d/\n",
		      proxy->server_name, proxy->server_port);
	    }

	  strncpy (hello.hello_program, PACKAGE, 256);
	  strncpy (hello.hello_version, VERSION, 256);

	  serverindex = 0;

	  do
	    {
	      switch (list.list_host[serverindex].host_protocol)
		{
		case CDDB_MODE_CDDBP:
		  printf ("Trying CDDB server cddbp://%s:%d/\n",
			  list.list_host[serverindex].host_server.server_name,
			  list.list_host[serverindex].host_server.
			  server_port);
		  sock =
		    cddb_connect_server (list.list_host[serverindex++], proxy,
					 hello);
		  break;

		case CDDB_MODE_HTTP:
		  printf ("Trying CDDB server http://%s:%d/%s\n",
			  list.list_host[serverindex].host_server.server_name,
			  list.list_host[serverindex].host_server.server_port,
			  list.list_host[serverindex].host_addressing);
		  sock = cddb_connect_server (list.list_host[serverindex++],
					      proxy, hello, http_string, 512);
		  break;

		case CDINDEX_MODE_HTTP:
		  printf ("Trying CD Index server http://%s:%d/%s\n",
			  list.list_host[serverindex].host_server.server_name,
			  list.list_host[serverindex].host_server.server_port,
			  list.list_host[serverindex].host_addressing);
		  sock =
		    cdindex_connect_server (list.list_host[serverindex++],
					    proxy, http_string, 512);
		  break;

		default:
		  puts ("Invalid protocol selected!");
		  return;
		}
	      if (sock == -1)
		printf ("Connection error: %s\n", cddb_message);
	    }
	  while (serverindex < list.list_len && sock == -1);

	  if (sock == -1)
	    {
	      puts ("Could not establish connection with any CDDB servers!");
	      if (conf.conf_proxy)
		free (proxy);
	      cddb_generate_unknown_entry (cd_desc, data);
	      return;
	    }
	  serverindex--;
	  puts ("Connection established.");

	  switch (list.list_host[serverindex].host_protocol)
	    {
	    case CDDB_MODE_CDDBP:
	      printf ("Retrieving information on %02lx.\n",
		      cddb_discid (cd_desc));
	      if (cddb_query (cd_desc, sock, CDDB_MODE_CDDBP, &query) < 0)
		{
		  fprintf (stderr, "CDDB query error: %s", cddb_message);
		  if (conf.conf_proxy)
		    free (proxy);
		  cddb_generate_unknown_entry (cd_desc, data);
		  return;
		}
	      break;

	    case CDDB_MODE_HTTP:
	      printf ("Retrieving information on %02lx.\n",
		      cddb_discid (cd_desc));
	      if (cddb_query (cd_desc, sock, CDDB_MODE_HTTP,
			      &query, http_string) < 0)
		{
		  fprintf (stderr, "CDDB query error: %s", cddb_message);
		  if (conf.conf_proxy)
		    free (proxy);
		  cddb_generate_unknown_entry (cd_desc, data);
		  return;
		}
	      shutdown (sock, 2);
	      close (sock);

	      if ((sock =
		   cddb_connect_server (list.list_host[serverindex],
					proxy, hello, http_string, 512)) < 0)
		{
		  perror ("HTTP server reconnection error");
		  if (conf.conf_proxy)
		    free (proxy);
		  cddb_generate_unknown_entry (cd_desc, data);
		  return;
		}
	      break;

	    case CDINDEX_MODE_HTTP:
	      cdindex_discid (cd_desc, discid, CDINDEX_ID_SIZE);
	      printf ("Retrieving information on %s.\n", discid);
	      if (cdindex_read (cd_desc, sock, data, http_string) < 0)
		{
		  printf ("No match for %s.\n", discid);
		  if (conf.conf_proxy)
		    free (proxy);
		  cddb_generate_unknown_entry (cd_desc, data);
		  return;
		}
	      printf ("Match for %s: %s / %s\nDownloading data...\n",
		      discid, data->data_artist, data->data_title);

	      memcpy (&outdata, data, sizeof (struct disc_data));
	      if (cddb_write_disc_data (cd_desc, outdata) < 0)
		fprintf (stderr, "CDDB write error: %s", cddb_message);
	      return;
	    }

	  if (conf.conf_proxy)
	    free (proxy);

	  /* FIXME: Hey!  */
	  if (list.list_host[serverindex].host_protocol == CDINDEX_MODE_HTTP);

	  switch (query.query_match)
	    {
	    case QUERY_EXACT:
	      if (strlen (query.query_list[0].list_artist) > 0)
		{
		  printf ("Match for %02lx: %s / %s\nDownloading data...\n",
			  cddb_discid (cd_desc),
			  query.query_list[0].list_artist,
			  query.query_list[0].list_title);
		}
	      else
		{
		  printf ("Match for %02lx: %s\nDownloading data...\n",
			  cddb_discid (cd_desc),
			  query.query_list[0].list_title);
		}
	      entry.entry_genre = query.query_list[0].list_genre;
	      entry.entry_id = query.query_list[0].list_id;
	      switch (list.list_host[serverindex].host_protocol)
		{
		case CDDB_MODE_CDDBP:
		  if (cddb_read (cd_desc, sock, CDDB_MODE_CDDBP,
				 entry, data) < 0)
		    {
		      perror ("CDDB read error");
		      cddb_generate_unknown_entry (cd_desc, data);
		      return;
		    }
		  cddb_quit (sock);
		  break;

		case CDDB_MODE_HTTP:
		  if (cddb_read (cd_desc, sock, CDDB_MODE_HTTP,
				 entry, data, http_string) < 0)
		    {
		      perror ("CDDB read error");
		      cddb_generate_unknown_entry (cd_desc, data);
		      return;
		    }

		  shutdown (sock, 2);
		  close (sock);
		  break;
		}
	      break;

	    case QUERY_INEXACT:
	      printf ("Inexact match for %02lx.\n", cddb_discid (cd_desc));
	      puts ("Please choose from the following inexact matches:");
	      for (index = 0; index < query.query_matches; index++)
		if (strlen (query.query_list[index].list_artist) < 1)
		  printf ("%d: %s\n",
			  index + 1, query.query_list[index].list_title);
		else
		  printf ("%d: %s / %s\n",
			  index + 1,
			  query.query_list[index].list_artist,
			  query.query_list[index].list_title);
	      printf ("%d: None of the above.\n", index + 1);
	      printf ("> ");
	      fgets (inbuffer, 256, stdin);
	      selection = strtol (inbuffer, NULL, 10);
	      if (selection > 0 && selection <= query.query_matches)
		{
		  entry.entry_genre =
		    query.query_list[selection - 1].list_genre;
		  entry.entry_id = query.query_list[selection - 1].list_id;
		  puts ("Downloading data...");
		  switch (list.list_host[serverindex].host_protocol)
		    {
		    case CDDB_MODE_CDDBP:
		      if (cddb_read (cd_desc, sock, CDDB_MODE_CDDBP,
				     entry, data) < 0)
			{
			  perror ("CDDB read error");
			  cddb_generate_unknown_entry (cd_desc, data);
			  return;
			}
		      cddb_quit (sock);
		      break;

		    case CDDB_MODE_HTTP:
		      if (cddb_read (cd_desc, sock, CDDB_MODE_HTTP,
				     entry, data, http_string) < 0)
			{
			  perror ("CDDB read error");
			  cddb_generate_unknown_entry (cd_desc, data);
			  return;
			}
		      shutdown (sock, 2);
		      close (sock);
		      break;
		    }
		  break;
		}

	    case QUERY_NOMATCH:
	      printf ("No match for %02lx.\n", cddb_discid (cd_desc));
	      cddb_generate_unknown_entry (cd_desc, data);
	    }
	  close (sock);

	  memcpy (&outdata, data, sizeof (struct disc_data));
	  if (cddb_write_disc_data (cd_desc, outdata) < 0)
	    fprintf (stderr, "CDDB write error: %s\n", cddb_message);
	}
    }
  return;
}

void
cddb_submit (int cd_desc, char *email_address)
{
  int index = 0;
  struct cddb_conf conf;
  struct cddb_serverlist list;
  struct disc_data data;
  struct cddb_server *proxy;

  proxy = (struct cddb_server *) xmalloc (sizeof (struct cddb_server));
  cddb_read_disc_data (cd_desc, &data);
  cddb_read_serverlist (&conf, &list, proxy);

  if (conf.conf_access == CDDB_ACCESS_LOCAL)
    {
      free (proxy);
      puts ("Cannot complete CDDB submission in local mode");
      return;
    }

  if (!conf.conf_proxy)
    {
      free (proxy);
      proxy = NULL;
    }

  while (list.list_host[index].host_protocol != CDINDEX_MODE_HTTP
	 && list.list_host[index].host_protocol != CDDB_MODE_HTTP
	 && index < list.list_len)
    index++;

  if (index >= list.list_len)
    {
      puts ("Could not locate any HTTP servers.  "
	    "Try running 'sites refresh'.");
      if (conf.conf_proxy)
	free (proxy);
      return;
    }

  printf ("Submitting to %s server http://%s:%d%s\n",
	  (list.list_host[index].host_protocol == CDINDEX_MODE_HTTP) ?
	  "CD Index" : "CDDB",
	  list.list_host[index].host_server.server_name,
	  list.list_host[index].host_server.server_port,
	  (list.list_host[index].host_protocol == CDINDEX_MODE_HTTP) ?
	  CDINDEX_SUBMIT_CGI : HTTP_SUBMIT_CGI);
  switch (list.list_host[index].host_protocol)
    {
    case CDINDEX_MODE_HTTP:
      cdindex_write_disc_data (cd_desc, data);
      if (cdindex_http_submit (cd_desc, list.list_host[index], proxy) < 0)
	fprintf (stderr, "Submission error: %s\n", cddb_message);
      else
	puts ("Success!");
      break;

    case CDDB_MODE_HTTP:
      if (cddb_http_submit (cd_desc, list.list_host[index],
			    proxy, email_address) < 0)
	fprintf (stderr, "Submission error: %s\n", cddb_message);
      else
	puts ("Success!");
    }
}
