/****************************************************************************
 *    lib/c_taid/parser.y - This file is part of coala						*
 *																			*
 *    Copyright (C) 2009  Torsten Grote										*
 *																			*
 *    This program is free software; you can redistribute it and/or modify	*
 *    it under the terms of the GNU General Public License as published by	*
 *    the Free Software Foundation; either version 3 of the License, or		*
 *    (at your option) any later version.									*
 *																			*
 *    This program is distributed in the hope that it will be useful,		*
 *    but WITHOUT ANY WARRANTY; without even the implied warranty of		*
 *    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the			*
 *    GNU General Public License for more details.							*
 *																			*
 *    You should have received a copy of the GNU General Public License		*
 *    along with this program; if not, see http://www.gnu.org/licenses		*
 ****************************************************************************/
%name CtaidParser

%define LSP_NEEDED
%define ERROR_BODY = 0
%define LEX_BODY = 0
%define CONSTRUCTOR_PARAM Coala::CompilerOptions* options
%define CONSTRUCTOR_CODE {debug=options->getDebug(); neg=options->getFakeClassicalNegation()?"neg_":"-";}

%define MEMBERS void postProcessing(); bool cout_neg(bool); int debug; string neg; virtual ~CtaidParser(){}\
private:\
	   typedef set<string, less<string> > StringSet;\
	   StringSet all_fluents;\
	   StringSet default_fluents;\
	   StringSet inertial_fluents;\
	   StringSet all_actions;\
	   StringSet triggered_actions;\
	   StringSet allowed_actions;\
	   StringSet exogenous_actions;

// TODO 
// - adapt debug value
// - get the end time right
// - action rules for prediction only for non-triggered ones
// - check whether fluents and actions are defined

%header{
#include <stdio.h>
#include <iostream>
#include <string>
#include <list>
#include <set>
#include <algorithm>

#include "../options.h"

using namespace std;
//using namespace C_taid;
%}

%{
typedef list<pair<string,bool> > ListPair;

int latest_obs = 0;
%}

%union {
	int number;
	const string *istring;
	list<pair<string,bool> > *ilist;
}

%token <istring> IDENTIFIER
%token <istring> NEG_IDENTI
%token <number> NUMBER
%token <number> ACT		// action
%token <number> FLU		// fluent
%token <number> DEF		// default
%token <number> CAUS	// causes
%token <number> IF		// if
%token <number> TRIG	// triggers
%token <number> ALLOW	// allows
%token <number> INHIB	// inhibits
%token <number> NCONC	// noconcurrency
%token <number> AT		// at
%token <number> OC_AT	// occurs_at
%token <number> EXP		// explain
%token <number> PLAN	// plan
%token <number> PRED	// predict
%token <number> COMMA	// ,
%token <number> DOT		// .

%type <ilist> fluent_list

%start program

%%

program: /* empty */ | program rule DOT;

rule: desc_rule | obs_rule | query_rule;

desc_rule: fact | static_rule | dyn_rule | trig_rule | allow_rule | inhib_rule | nconc_rule;

obs_rule: flu_obs | act_obs;

query_rule: exp_rule | plan_rule | pred_rule;

fact: act_fact | flu_fact | def_fact;

// (action a_1,...,a_n)
act_fact: ACT fluent_list {	// not really a fluent_list, but I still use it for actions
	for(ListPair::iterator n = $2->begin(); n != $2->end(); ++n) {
		if(neg == "neg_") {
			cout << ":- action_"+n->first+"(T), neg_action_"+n->first+"(T), time(T).\n";
			
			// hides every neg_action
			if(debug < 2) cout << "#hide neg_action_"+n->first+"(T).\n";
		}

		all_actions.insert(n->first);
	}
	delete $2;
}

// (fluent f_1,...,f_n)
flu_fact: FLU fluent_list {
	for(ListPair::iterator n = $2->begin(); n != $2->end(); ++n) {
		all_fluents.insert(n->first);
	}
	delete $2;
}

// (default f_1,...,f_n)
def_fact: DEF fluent_list {
	for(ListPair::iterator n = $2->begin(); n != $2->end(); ++n) {
		cout_neg(n->second);
		cout << "fluent_"+n->first+"(T) :- not ";
		cout_neg(!n->second);
		cout << "fluent_"+n->first+"(T), time(T).\n";
		
		if(neg == "neg_") {
			// relationship between fluent and its negation
			cout << ":- fluent_"+n->first+"(T), neg_fluent_"+n->first+"(T), time(T).\n";
			
			// hides neg_fluent for every default fluent
			if(debug < 2) {
				cout << "#hide neg_fluent_"+n->first+"(T).\n";
			}
		}
		
		if(debug < 1) cout << "#hide fluent_"+n->first+"(T).\n";

		default_fluents.insert(n->first);
	}
	delete $2;
}

// (a causes f_1,...,f_n if g_1,...,g_m)
dyn_rule: IDENTIFIER CAUS fluent_list IF fluent_list {
	for(ListPair::iterator n = $3->begin(); n != $3->end(); ++n) {
		// f_1,...,f_n
		cout_neg(n->second);
		cout << "fluent_"+n->first+"(T+1) :- action_"+*$1+"(T), ";

		for(ListPair::iterator m = $5->begin(); m != $5->end(); ++m) {
			// g_1,...,g_m
			cout_neg(m->second);
			cout << "fluent_"+m->first+"(T), ";
		}
		cout << "time(T; T+1).\n";
	}
	delete $1;
	delete $3;
	delete $5;
}
// (a causes f_1,...,f_n)
| IDENTIFIER CAUS fluent_list {
	for(ListPair::iterator n = $3->begin(); n != $3->end(); ++n) {
		cout_neg(n->second);
		cout << "fluent_"+n->first+"(T+1) :- action_"+*$1+"(T), time(T; T+1).\n";
	}
	delete $1;
	delete $3;
}

// (f_1,...,f_n if g_1,...,g_m)
static_rule: fluent_list IF fluent_list {
	for(ListPair::iterator n = $1->begin(); n != $1->end(); ++n) {
		cout_neg(n->second);
		cout << "fluent_"+n->first+"(T) :- ";

		for(ListPair::iterator m = $3->begin(); m != $3->end(); ++m) {
			cout_neg(m->second);
			cout << "fluent_"+m->first+"(T), ";
		}
		cout << "time(T).\n";
	}
	delete $1;
	delete $3;
}

// (f_1,...,f_n triggers a)
trig_rule: fluent_list TRIG IDENTIFIER {
	cout << "trig_action_"+*$3+"(T) :- ";
	
	for(ListPair::iterator n = $1->begin(); n != $1->end(); ++n) {
		cout_neg(n->second);
		cout << "fluent_"+n->first+"(T), ";
	}
	cout << "time(T).\n";
	
	// trigger only when not inhibited
	cout << "action_"+*$3+"(T) :- trig_action_"+*$3+"(T), not "+neg+"action_"+*$3+"(T), time(T).\n";
	
	// hides every trig_action
	if(debug < 2) cout << "#hide trig_action_"+*$3+"(T).\n";

	triggered_actions.insert(*$3);

	delete $1;
	delete $3;
}

// (f_1,...,f_n allows a)
allow_rule: fluent_list ALLOW IDENTIFIER {
	cout << "allow_action_"+*$3+"(T) :- ";
	
	for(ListPair::iterator n = $1->begin(); n != $1->end(); ++n) {
		cout_neg(n->second);
		cout << "fluent_"+n->first+"(T), ";
	}
	cout << "time(T).\n";
	
	// hides allow_action for every allowed action
	if(debug < 2) cout << "#hide allow_action_"+*$3+"(T).\n";

	allowed_actions.insert(*$3);

	delete $1;
	delete $3;
}

// (f_1,...,f_n inhibits a)
inhib_rule: fluent_list INHIB IDENTIFIER {
	cout << neg+"action_"+*$3+"(T) :- ";
	
	for(ListPair::iterator n = $1->begin(); n != $1->end(); ++n) {
		cout_neg(n->second);
		cout << "fluent_"+n->first+"(T), ";
	}
	cout << "time(T).\n";

	delete $1;
	delete $3;
}

// (noconcurrency a_1,...,a_n)
nconc_rule: NCONC fluent_list {	// not really a fluent_list, but I still use it for actions
	cout << ":- 2 {";
	
	for(ListPair::iterator n = $2->begin(); n != $2->end(); ++n) {
		if(n != $2->begin()) cout << ", ";
		cout << "action_"+n->first+"(T)";
	}
	cout << "}, time(T).\n";

	delete $2;
}

// f at t_i
flu_obs: fluent_list AT NUMBER {
	for(ListPair::iterator n = $1->begin(); n != $1->end(); ++n) {
		if($3 == 0) {
			cout_neg(n->second);
			cout << "fluent_"+n->first+"(0).\n";
		}
		else {
			cout << ":- not ";
			cout_neg(n->second);
			cout << "fluent_"+n->first+"("<<$3<<"), time("<<$3<<").\n";
		}
	}
	
	// remember a later observation time
	latest_obs = ($3 > latest_obs) ? $3 : latest_obs ;

	delete $1;
}

// a occurs_at t_i
act_obs: fluent_list OC_AT NUMBER {
	for(ListPair::iterator n = $1->begin(); n != $1->end(); ++n) {
		// for non-exogenous actions (not triggered or allowed)
		if(triggered_actions.find(n->first) != triggered_actions.end() || allowed_actions.find(n->first) != allowed_actions.end()) {
			cout << ":- ";
			cout_neg(!n->second);
			cout << "action_"+n->first+"("<<$3<<"), time("<<$3<<").\n";
		}
		// for exogenous actions
		else {
			cout_neg(n->second);
			cout << "action_"+n->first+"("<<$3<<").\n";
		}
	}
	
	// remember a later observation time
	latest_obs = ($3 > latest_obs) ? $3 : latest_obs ;

	delete $1;
}

// explain n
exp_rule: EXP {
	// rules for generating all possible initial states
	for(StringSet::iterator n = all_fluents.begin(); n != all_fluents.end(); ++n) {
		cout << "fluent_"+*n+"(0) :- not "+neg+"fluent_"+*n+"(0).\n";
		cout << neg+"fluent_"+*n+"(0) :- not fluent_"+*n+"(0).\n";
	}
	
	// all exogenous and allowed actions in one set: all - triggered
	StringSet exoallow_actions;
	set_difference(all_actions.begin(), all_actions.end(), triggered_actions.begin(), triggered_actions.end(),
				insert_iterator<StringSet>(exoallow_actions, exoallow_actions.begin()) );

	for(StringSet::iterator n = exoallow_actions.begin(); n != exoallow_actions.end(); ++n) {
		cout << "action_"+*n+"(T) :- allow_action_"+*n+"(T), not "+neg+"action_"+*n+"(T), time(T), T<n.\n";
		cout << neg+"action_"+*n+"(T) :- not action_"+*n+"(T), time(T), T<n.\n";
	}
}

// plan f_1,...,f_n
plan_rule: PLAN fluent_list {
	// rules for generating all possible initial states
	for(StringSet::iterator n = all_fluents.begin(); n != all_fluents.end(); ++n) {
		cout << "fluent_"+*n+"(0) :- not "+neg+"fluent_"+*n+"(0).\n";
		cout << neg+"fluent_"+*n+"(0) :- not fluent_"+*n+"(0).\n";
	}

	// all exogenous and allowed actions in one set: all - triggered
	StringSet exoallow_actions;
	set_difference(all_actions.begin(), all_actions.end(), triggered_actions.begin(), triggered_actions.end(),
				insert_iterator<StringSet>(exoallow_actions, exoallow_actions.begin()) );

	for(StringSet::iterator n = exoallow_actions.begin(); n != exoallow_actions.end(); ++n) {
		cout << "action_"+*n+"(T) :- allow_action_"+*n+"(T), not "+neg+"action_"+*n+"(T), not achieved(T), time(T).\n";
		cout << neg+"action_"+*n+"(T) :- not action_"+*n+"(T), time(T).\n";
	}

	cout << ":- not achieved.\n";
	cout << "achieved :- achieved(0).\n";
	cout << "achieved :- achieved(T+1), not achieved(T), time(T;T+1).\n";
	cout << "achieved(T) :- ";
	for(ListPair::iterator n = $2->begin(); n != $2->end(); ++n) {
		if(n != $2->begin()) cout << ", ";
		cout_neg(n->second);
		cout << "fluent_"+n->first+"(T)";
	}
	cout << ", T=n.\n";
	
	cout << "achieved(T) :- achieved(T+1), ";
	for(ListPair::iterator n = $2->begin(); n != $2->end(); ++n) {
		if(n != $2->begin()) cout << ", ";
		cout_neg(n->second);
		cout << "fluent_"+n->first+"(T)";
	}
	cout << ", time(T;T+1).\n";

	// hides the achieve predicates
	if(debug < 2) cout << "#hide achieved.\n";
	if(debug < 2) cout << "#hide achieved(T).\n";
	
	delete $2;
}

// predict f_1,...,f_n
pred_rule: PRED fluent_list {
	// rules for generating all possible initial states
	for(StringSet::iterator n = all_fluents.begin(); n != all_fluents.end(); ++n) {
		cout << "fluent_"+*n+"(0) :- not "+neg+"fluent_"+*n+"(0).\n";
		cout << neg+"fluent_"+*n+"(0) :- not fluent_"+*n+"(0).\n";
	}
	
	// justify the observations by including for every action:
	for(StringSet::iterator n = all_actions.begin(); n != all_actions.end(); ++n) {
		cout << "action_"+*n+"(T) :- allow_action_"+*n+"(T), not "+neg+"action_"+*n+"(T), time(T), T<n.\n";
		cout << neg+"action_"+*n+"(T) :- not action_"+*n+"(T), time(T), T<n.\n";
	}

	cout << "predicted :- ";
	for(ListPair::iterator n = $2->begin(); n != $2->end(); ++n) {
		if(n != $2->begin()) cout << ", ";
		cout_neg(n->second);
		cout << "fluent_"+n->first+"(T)";
	}
	cout << ", time(T), T>="<<latest_obs<<".\n";
	
	// hides the predicted predicate
	if(debug < 2) cout << "#hide predicted.\n";

	delete $2;
}


// f_1,...,f_n
fluent_list: IDENTIFIER {
	$$ = new ListPair();
	$$->push_back(pair<string, bool>(*$1, true));
	delete $1;
}
| NEG_IDENTI {
	$$ = new ListPair();
	$$->push_back(pair<string, bool>(*$1, false));
	delete $1;
}
| fluent_list COMMA IDENTIFIER {
	$1->push_back(pair<string, bool>(*$3, true));
	$$ = $1;
	delete $3;
}
| fluent_list COMMA NEG_IDENTI {
	$1->push_back(pair<string, bool>(*$3, false));
	$$ = $1;
	delete $3;
}

%%

void CtaidParser::postProcessing() {
	// trig_allow_actions = triggered_actions + allowed_actions
	StringSet trig_allow_actions;
	set_union(triggered_actions.begin(), triggered_actions.end(), allowed_actions.begin(), allowed_actions.end(),
				insert_iterator<StringSet>(trig_allow_actions, trig_allow_actions.begin()) );
	
	// exogenous_actions = all_actions - trig_allow_actions
	set_difference(all_actions.begin(), all_actions.end(), trig_allow_actions.begin(), trig_allow_actions.end(),
				insert_iterator<StringSet>(exogenous_actions, exogenous_actions.begin()) );
	
	// add rules for EXOGENOUS actions (not triggered or allowed)
	cout << "\n% exogenous actions\n";
	for(StringSet::iterator n = exogenous_actions.begin(); n != exogenous_actions.end(); ++n) {
		cout << "allow_action_"+*n+"(T) :- time(T).\n";

		// hides allow_action for every exogenous action
		if(debug < 2) cout << "#hide allow_action_"+*n+"(T).\n";
	}
	
	
	// inertial_fluents = all_fluents - default_fluents
	set_difference(all_fluents.begin(), all_fluents.end(), default_fluents.begin(), default_fluents.end(),
					insert_iterator<StringSet>(inertial_fluents, inertial_fluents.begin()) );
	
	// add rules for INERTIAL fluents (not default)
	cout << "\n% inertial fluents\n";
	for(StringSet::iterator n = inertial_fluents.begin(); n != inertial_fluents.end(); ++n) {
		cout << "fluent_"+*n+"(T+1) :- fluent_"+*n+"(T), not "+neg+"fluent_"+*n+"(T+1), time(T;T+1).\n";
		cout << neg+"fluent_"+*n+"(T+1) :- "+neg+"fluent_"+*n+"(T), not fluent_"+*n+"(T+1), time(T;T+1).\n";
		
		if(neg == "neg_") {
			// relationship between fluent and its negation
			cout << ":- fluent_"+*n+"(T), neg_fluent_"+*n+"(T), time(T).\n";
			
			// hides neg_fluent for every inertial fluent
			if(debug < 2) {
				cout << "#hide neg_fluent_"+*n+"(T).\n";
			}
		}
		if(debug < 1) cout << "#hide fluent_"+*n+"(T).\n";
	}

	// add time
	cout << "\n% time facts\n";
	cout << "time(0..n).\n";
	if(debug < 2) cout << "#hide time(N).\n";
}

bool CtaidParser::cout_neg(bool not_neg) {
	if(not_neg) return true;
	else {
		cout << neg;
		return false;
	}
}
