#include <string>
#include <iostream>
#include <boost/any.hpp>
#include <QPair>
#include <QVariant>

#include "backendconnection.h"
#include "log.h"
#include "searchentry.h"
#include "util.h"
#include <rpcdriver/types.h>

using namespace std;
using namespace rpc;

void BackendConnection::authenticate( const QString & password )
{
	boost::any p = password.toStdString();
	CmdPtr cmd(new anylist);
	cmd->push_back( string("authenticate") );
	cmd->push_back(p);
	driver->queueCommand(-1,cmd);
}

void BackendConnection::createSession( const QString& u )
{
    boost::any url = u.toStdString();
    CmdPtr cmd(new anylist);
    cmd->push_back( string("createSession") );
    cmd->push_back(url);
    driver->queueCommand(-1,cmd);
}

void BackendConnection::sendChat(int session,const QString& msg)
{
    CmdPtr cmd(new anylist);
    cmd->push_back( string("sendChat") );
    cmd->push_back( session );
    cmd->push_back( msg.toStdString() );
    driver->queueCommand(-1,cmd);
}

void BackendConnection::sendSearch(int session, int64 size, int sizeMode, int typeMode, const QString& search)
{
    CmdPtr cmd(new anylist);
    cmd->push_back( string("search") );
    cmd->push_back( session );
    cmd->push_back( search.toStdString() );
    cmd->push_back( size );
    cmd->push_back( sizeMode );
    cmd->push_back( typeMode );
    driver->queueCommand(-1,cmd);
}

void BackendConnection::getSharedDirs()
{
    CmdPtr cmd(new anylist);
    cmd->push_back( string("getSharedDirectories") );
    driver->queueCommand(-1,cmd);
}

void BackendConnection::addShare(const QString& name,const QString& dir)
{
    CmdPtr cmd(new anylist);
    cmd->push_back( string("addShare") );
    cmd->push_back(name.toStdString());
    cmd->push_back(dir.toStdString());
    driver->queueCommand(-1,cmd);
}

void BackendConnection::delShare(const QString& name)
{
    CmdPtr cmd(new anylist);
    cmd->push_back( string("removeShare") );
    cmd->push_back(name.toStdString());
    driver->queueCommand(-1,cmd);
}

void BackendConnection::requestRunningSessions()
{
    CmdPtr cmd(new anylist);
    cmd->push_back( string("requestRunningSessions") );
    driver->queueCommand(-1,cmd);
}

void BackendConnection::downloadFile(const SearchEntry& file)
{
    logger->debug("Downloading " + file.getPath() + file.getFileName());
    CmdPtr cmd(new anylist);
    cmd->push_back( string("downloadFile") );
    cmd->push_back( file.getFile() );
	cmd->push_back( file.getType() == SearchEntry::DIRECTORY ? (int64)-1 : file.getFileSize() );
    cmd->push_back( file.getUserId() );
	cmd->push_back( string("") );
    cmd->push_back( file.getTTH().toStdString() );
    driver->queueCommand(-1,cmd);
}

void BackendConnection::downloadFile( int uId, const QString& file, int64 size, const QString& tth )
{
    logger->debug( "Downloading " + file );
    CmdPtr cmd(new anylist);
    cmd->push_back( string("downloadFile") );
    cmd->push_back( file.toStdString() );
    cmd->push_back( size );
    cmd->push_back( uId );
    cmd->push_back( string("") );
    cmd->push_back( tth.toStdString() );
    driver->queueCommand(-1,cmd);
}


void BackendConnection::downloadFileTo(const SearchEntry& file, const QString& dir )
{
    logger->debug("Downloading " + file.getPath() + file.getFileName());
    CmdPtr cmd(new anylist);
    cmd->push_back( string("downloadFile") );
    cmd->push_back( file.getFile() );
    cmd->push_back( file.getType() == SearchEntry::DIRECTORY ? -1 : file.getFileSize() );
    cmd->push_back( file.getUserId() );
    cmd->push_back(dir.toStdString());
    cmd->push_back( file.getTTH().toStdString() );
    driver->queueCommand(-1,cmd);
}

void BackendConnection::downloadFileTo( int uId, const QString& file, int64 size, const QString& tth, const QString& dir )
{
    logger->debug( "Downloading " + file + " to " + dir );
    CmdPtr cmd(new anylist);
    cmd->push_back( string("downloadFile") );
    cmd->push_back( file.toStdString() );
    cmd->push_back( size );
    cmd->push_back( uId );
    cmd->push_back( dir.toStdString() );
    cmd->push_back( tth.toStdString() );
    driver->queueCommand(-1,cmd);
}


void BackendConnection::getUserFileList( int userId )
{
    logger->debug( "Requesting file list for user with id:" + QString::number(userId) );
    CmdPtr cmd(new anylist);
    cmd->push_back( string("getUserFileList") );
    cmd->push_back( userId );
    driver->queueCommand(-1,cmd);
}

void BackendConnection::tellBackendToDie()
{
    CmdPtr cmd(new anylist);
    cmd->push_back( string("die") );
    driver->queueCommand(-1,cmd);
}

void BackendConnection::getHubList( bool refresh )
{
    CmdPtr cmd(new anylist);
    cmd->push_back( string("getHubList") );
    cmd->push_back( refresh );
    driver->queueCommand(-1,cmd);
}


void BackendConnection::getFavouriteHubs( )
{
    CmdPtr cmd(new anylist);
    cmd->push_back( string("getFavouriteHubs") );
    driver->queueCommand(-1,cmd);
}

void BackendConnection::addFavouriteHub( const rpc_types::FavouriteHub &e )
{
    CmdPtr cmd(new anylist);
    cmd->push_back( string("addFavouriteHub") );
    cmd->push_back( e.nick );
    cmd->push_back( e.userDescription );
    cmd->push_back( e.name );
    cmd->push_back( e.server );
    cmd->push_back( e.description );
    cmd->push_back( e.password );
    cmd->push_back( e.autoConnect );
    driver->queueCommand(-1,cmd);
}

void BackendConnection::removeFavouriteHub( const QString &server )
{
    CmdPtr cmd(new anylist);
    cmd->push_back( string("removeFavouriteHub") );
    cmd->push_back( server.toStdString() );
    driver->queueCommand(-1,cmd);
}

void BackendConnection::closeSession( int session )
{
    CmdPtr cmd(new anylist);
    cmd->push_back( string("closeSession") );
    cmd->push_back( session );
    driver->queueCommand(-1,cmd);
}

void BackendConnection::getSettings( QList<QString> keys )
{
    CmdPtr cmd(new anylist);
    cmd->push_back( string("getSettings") );
    // convert the qlist to an anylist
    anylist k;
    for(int i=0;i<keys.size();i++)
        k.push_back(keys[i].toStdString());
    cmd->push_back(k);
    driver->queueCommand(-1,cmd);
}

void BackendConnection::setSettings(QList<QPair<QString,QVariant> > settings)
{
    CmdPtr cmd(new anylist);
    cmd->push_back( string("setSettings") );
    anylist keys,values;

    for(int i=0;i<settings.size();i++)
    {

        keys.push_back(settings[i].first.toStdString());
        if(settings[i].second.type()==QVariant::String)
            values.push_back( settings[i].second.toString().toStdString() );
        else if(settings[i].second.type()==QVariant::Int)
            values.push_back( settings[i].second.toInt() );
        else if(settings[i].second.type()==QVariant::LongLong)
            values.push_back( settings[i].second.toLongLong() );
    }

    cmd->push_back(keys);
    cmd->push_back(values);
    driver->queueCommand(-1,cmd);
}

void BackendConnection::sendPassword( int session, const QString& pass )
{
    CmdPtr cmd(new anylist);
    cmd->push_back( string("sendPassword") );
    cmd->push_back( session );
    cmd->push_back( pass.toStdString() );
    driver->queueCommand(-1,cmd);
}

void BackendConnection::removeQueueItem( int id )
{
	logger->info(QString("Removing QueueItem with id %1").arg(id));
    CmdPtr cmd(new anylist);
    cmd->push_back( string("removeQueueItem") );
	cmd->push_back( id );
    driver->queueCommand(-1,cmd);
}

void BackendConnection::forceConnectionAttempt( int userId )
{
	logger->info(QString("Forcing attempt on user with id %1").arg(userId));
	CmdPtr cmd(new anylist);
    cmd->push_back( string("connectToUser") );
	cmd->push_back( userId );
    driver->queueCommand(-1,cmd);
}

void BackendConnection::die( )
{
	logger->info("BackendConnection::die()");
	CmdPtr cmd(new anylist);
	cmd->push_back( string("die") );
	driver->queueCommand(-1,cmd);
}

BackendConnection::~ BackendConnection( )
{
	logger->info("Destructing backendconnection");
}

void BackendConnection::removeSource( int qid, int uid )
{
	logger->debug("BackendConnection::removeSource: " + QString::number( qid ) + ", " + QString::number( uid ));
	CmdPtr cmd(new anylist);
	cmd->push_back( string("removeSource") );
	cmd->push_back( qid );
	cmd->push_back( uid );
	driver->queueCommand(-1,cmd);
}
