//
// C++ Interface: searchlistmodel
//
// Description: 
//
//
// Author: Mikael Gransell <mikael.gransell@bredband.net>, (C) 2005
//
// Copyright: See COPYING file that comes with this distribution
//
//

#ifndef SEARCH_TABLE_MODEL_H_
#define SEARCH_TABLE_MODEL_H_

#include <QAbstractTableModel>
#include <QList>
#include <QPointer>
#include <QFileIconProvider>

#include "searchentry.h"

/**
 * This class acts as the model for the search table. Data is added to
 * this model and the view that displays the data is updated according to 
 * the Model/View way of programming.
 */
class SearchTableModel : public QAbstractTableModel
{
	Q_OBJECT
public:
	/**
	 * Constructor.
	*/
	SearchTableModel() 
	{
		// Create an icon provider that will suply the list wiht icons
		m_pIconProvider = new QFileIconProvider();
		filterEnabled = false;
		sortOrder = Qt::AscendingOrder;
	}
	/**
	 * Destructor.
	 */
	virtual ~SearchTableModel() 
	{
		delete m_pIconProvider;
	}
	
	/**
	 * Enumerates the different columns in the list model.
	*/
	enum eSearchColumnHeader {
		FILENAME_COLUMN = 0,
		USER_COLUMN,
		SIZE_COLUMN,
		PATH_COLUMN,
		SLOTS_COLUMN,
		HUBNAME_COLUMN,
		//TYPE_COLUMN,
		TTH_COLUMN,
		NUM_COLUMNS // Make sure this is allways last
	};
	
	/**
	 * Returns the amount of rows that should be displayed by the view.
	 * @param index Index of parent.
	*/
	virtual int rowCount( const QModelIndex& ) const { return filteredResults.size(); }
	/**
	 * Returns the amount of columns that the view shoulf display.
	 * @param index Index of parent.
	*/
	virtual int columnCount( const QModelIndex& ) const { return NUM_COLUMNS; } // Make sure to change this if more parameters are added to the search entry 
	
	/**
	 * Returns the data that should be displayed for the specified index with the specified role.
	 * @param index Index of the item in the model that data is requested for.
	 * @param role The type of data that should be returned.
	 * @see Qt::DisplayRole.
	*/
	virtual QVariant data( const QModelIndex& index, int role = Qt::DisplayRole ) const;
	
	/**
	 * Returns the data that should be displayed in the header for a certain role.
	 * @param section The column
	 * @param orientation If it is horisontal or vertical header information.
	 * @param role The type of data, icon, text etc.
	*/
	virtual QVariant headerData( int section, Qt::Orientation orientation, int role = Qt::DisplayRole ) const;
	
	/**
	 * Returns the SearchEntry specified by index.
	 * @param index The index of the search entry we are interested in.
	 * @throws std::out_of_range
	*/
	const SearchEntry& getEntry( int index ) const;

	virtual void sort ( int column, Qt::SortOrder order = Qt::AscendingOrder );
	
	/**
	 * Clears the model of any search results and emit a layout changed signal.
	 */
	void clear();
	
	//bool isEmpty() { return searchResults.size()==0; }
	
public slots:
	/**
	 * Called when search results have been received. Updates the model by adding the result and 
	 * notifies all views that data has been added.
	*/
	void onSearchResult( int session, const SearchEntry& sr );
	void onSearchResult( int session, const QList<SearchEntry>& sr );
	//! Enable / disable the filter. Currently only zero slot filtering is supported, this will change.
	void enableFilter(bool yes);
	
private:
	
	void sortedInsert(const SearchEntry&);
	
	/**
	 * Contains all the search results.
	*/
	QList<SearchEntry> searchResults;

	//! Contains the results that are displayed.
	QList<SearchEntry> filteredResults;
	
	/**
	 * Provides icons for the items in the list.
	 */
	QFileIconProvider* m_pIconProvider;
	
	bool filterEnabled;
	Qt::SortOrder sortOrder;
};

#endif
