/*
 *  userfiledialog.cpp
 *  ui
 *
 *  Created by Mikael Gransell on 4/26/06.
 *  Copyright 2006 __MyCompanyName__. All rights reserved.
 *
 */

#include "log.h"
#include "userfiledialog.h"
#include "globalusermodel.h"


UserFileDialog::UserFileDialog( const UserFileModelPtr& mdl,
								BackendConnectionPtr backendConn,
								boost::shared_ptr<GlobalUserModel> userModel,
								QWidget* parent )
: QDialog(parent),
  contextMenu(new QMenu(this)),
  backendConnection(backendConn),
  model( mdl )
{
	  ui.setupUi(this);
	  ui.treeView->setModel( model.get() );
	  
	  setWindowTitle(userModel->getUser(model->getUserId())->nick);
	  
	  createMenu();
}

void UserFileDialog::on_treeView_customContextMenuRequested( const QPoint& pos )
{
	// Show the menu
	contextMenu->popup( ui.treeView->mapToGlobal(pos) );
}

void UserFileDialog::on_treeView_doubleClicked()
{
	logger->debug( "Double click download" );
    // Try to get all the selected indexes and issue a download for each one.
	QModelIndexList selected = ui.treeView->selectionModel()->selectedIndexes();
	for( int i = 0; i < selected.size() - 1; i++ ) {
		QModelIndex index = selected[i];
		if( !model->getItem(index)->isDir() ) {
			download( index );
		}
	}
}

void UserFileDialog::onDownload()
{
	logger->debug( "Download triggered" );
    // Try to get all the selected indexes and issue a download for each one.
	QModelIndexList selected = ui.treeView->selectionModel()->selectedIndexes();
	for( int i = 0; i < selected.size() - 1; i++ ) {
		download( selected[i] );
	}	
}

void UserFileDialog::onDownloadTo()
{
	logger->debug( "DownloadTo triggered" );
    QString dir = QFileDialog::getExistingDirectory( this, tr("Choose a download directory") );
	if( !dir.isEmpty() ){
		dir += "/";
		// Try to get all the selected indexes and issue a download for each one.
		QModelIndexList selected = ui.treeView->selectionModel()->selectedIndexes();
		for( int i = 0; i < selected.size() - 1; i++ ) {
			downloadTo( selected[i], dir );
		}
	}
}


void UserFileDialog::download( const QModelIndex& index )
{
	try {
		// Throws std::out_of_range
		UserFileModel::TreeItem* item = model->getItem( index );
		
		if ( item ) {
			QString file = formatFileName( item );
			
			int64 size = item->isDir() ? -1 : item->getSize();
			QString tth = item->getTTH();
			
			logger->debug( "Downloading: " + file + " with size " + QString::number(size) + " and tth:" + tth );
			
			backendConnection->downloadFile( model->getUserId(), 
											 file, 
											 size, 
											 tth );
		}
	}
	catch(const std::out_of_range& e) {
		logger->error("Bad index when trying to download file\n");
	}
}

void UserFileDialog::downloadTo( const QModelIndex& index, const QString& path )
{
	try {
		// Throws std::out_of_range
		UserFileModel::TreeItem* item = model->getItem( index );
		
		if ( item ) {
			QString file = formatFileName( item );
			
			int64 size = item->isDir() ? -1 : item->getSize();
			QString tth = item->getTTH();
			
			logger->debug( "Downloading: " + file + " with size " + QString::number(size) + " and tth:" + tth + " to " + path );
			
			backendConnection->downloadFileTo( model->getUserId(), 
											   file, 
											   size, 
											   tth, 
											   path );
		}
	}
	catch(const std::out_of_range& e) {
		logger->error("Bad index when trying to download file\n");
	}	
}

QString UserFileDialog::formatFileName( const UserFileModel::TreeItem* item ) const
{
	QString file = item->getFullName();
	// Strip the first part of the path since this is the user name
	file = file.section( '/', 2 );
	
	// If this is a directory we need to add the final /
	if ( item->isDir() ) {
		file += '/';
	}
	return file;
}

void UserFileDialog::createMenu()
{
	QAction* downloadAct = contextMenu->addAction( tr("Download") );
	connect(downloadAct, SIGNAL(triggered()), this, SLOT(onDownload()));
	
	QAction* downloadToAct = contextMenu->addAction( tr("Download to...") );
	connect(downloadToAct, SIGNAL(triggered()), this, SLOT(onDownloadTo()));
}

