//
// C++ Implementation: UserFileModel
//
// Description:
//
//
// Author: Mikael Gransell <mikael.gransell@bredband.net>, (C) 2005
//
// Copyright: See COPYING file that comes with this distribution
//
//

#include "userfilemodel.h"

#include <QFileIconProvider>

#include <stdexcept>

QVariant UserFileModel::data( const QModelIndex &index, int role ) const 
{
	QVariant ret;
	
	// If we access an item that is out of range
    if (!index.isValid())
        return QVariant();
	
	// Get the item this concerns
	TreeItem *item = static_cast<TreeItem*>(index.internalPointer());
	
	// Check if it wants the text that should be displayed or the decorator.
	switch (role) {
	case Qt::DisplayRole:
			
		ret = item->data(index.column());		
		break;
		
	case Qt::DecorationRole:
				
		// only display icon for the first column
		if ( index.column() == 0 ) {
			
			// Get an icon from the icon provider
			ret = m_pIconProvider->icon( item->isDir() ? QFileIconProvider::Folder : QFileIconProvider::File );
		}
		
	default:
		break;
	}
	
	return ret;
}


Qt::ItemFlags UserFileModel::flags( const QModelIndex &index ) const 
{
    if (!index.isValid())
        return Qt::ItemIsEnabled;

    return Qt::ItemIsEnabled | Qt::ItemIsSelectable;
}


QVariant UserFileModel::headerData( int section, Qt::Orientation orientation, int role ) const 
{
    if (role != Qt::DisplayRole)
        return QVariant();

    if(orientation==Qt::Horizontal) {
        switch( section ) {
		case TreeItem::FILE_COLUMN:
            return tr("File Name");

		case TreeItem::SIZE_COLUMN:
            return tr("Size");
			
		case TreeItem::TTH_COLUMN:
			return tr("TTH");
        }
    }
    return QVariant();
}

QModelIndex UserFileModel::index( int row, int column, const QModelIndex &parent ) const 
{
    TreeItem *parentItem;

    if (!parent.isValid())
        parentItem = root;
    else
        parentItem = static_cast<TreeItem*>(parent.internalPointer());

    TreeItem *childItem = parentItem->child(row);
    if (childItem)
        return createIndex(row, column, childItem);
    else
        return QModelIndex();
}


QModelIndex UserFileModel::parent( const QModelIndex &index ) const 
{
    if (!index.isValid())
        return QModelIndex();

    TreeItem *childItem = static_cast<TreeItem*>(index.internalPointer());
    TreeItem *parentItem = childItem->parent();

    if (parentItem == root)
        return QModelIndex();

    return createIndex(parentItem->row(), 0, parentItem);
}


int UserFileModel::rowCount( const QModelIndex &parent ) const 
{
    TreeItem *parentItem;

    if (!parent.isValid())
        parentItem = root;
    else
        parentItem = static_cast<TreeItem*>(parent.internalPointer());

    return parentItem->childCount();
}


int UserFileModel::columnCount(const QModelIndex &parent) const 
{
    if (parent.isValid())
        return static_cast<TreeItem*>(parent.internalPointer())->columnCount();
    else
        return root->columnCount();
}

UserFileModel::TreeItem* UserFileModel::getItem( const QModelIndex& index ) const
{
	if ( !index.isValid() )
		throw std::out_of_range("Bad QModelIndex");

	TreeItem *item = static_cast<TreeItem*>(index.internalPointer());

	return item;
}

int UserFileModel::getUserId() const
{
	return userId;
}

bool UserFileModel::setTree( const QString& tree ) 
{
	root = new TreeItem( "", NULL );
    // Start parsing the tree.s
    handleSubTree( tree, root );

    // Calculate the size of each node.
    root->calculateSize();
	
	emit layoutChanged();

    return true;
}


void UserFileModel::handleSubTree( const QString& tree, TreeItem* parent ) const 
{	
	// Check if this is a directory
	int start = 0;
	int endDir = -1;
	
	while ( (start = tree.indexOf( '/', endDir + 1 )) != -1 ) {
		// Retrieve directory name
		QString dirName = tree.mid( endDir + 1, start - (endDir + 1) );
	
		TreeItem* item = new TreeItem( dirName, parent );
		//logger->info( "Adding directory " + dirName );
		
		// Find end of this dir
		endDir = start + 1;
		int level = 1;
		while ( level ) {
			QChar c = tree[endDir];
			// Go down a level if we find a new dir
			if( c == '/' ) level++;
			
			// Go up if we find the end of a dir
			if( c == '\\' ) level--;
			
			// Step forward
			endDir++;
		}
		
		QString dir = tree.mid( start + 1, endDir - start - 1  );
		handleSubTree( dir, item );
	}
	
	// Find the end of this directory
	int endFiles = tree.indexOf( '\\', endDir + 1 );
	// Get the files string
	QString files = tree.mid( endDir + 1, endFiles );
	//logger->info( "files: " + files );
	// Split all files and go through them
	QStringList fileList = files.split( '*' );
	for ( int i = 0; i < fileList.size(); i++ ) {
		addFile( fileList[i], parent );
	}
}

void UserFileModel::addFile( const QString& file, TreeItem* parent ) const 
{
	// Split the file with sep :
	QStringList lst = file.split( '|' );
	
	if ( lst.size() >= 2 ) {
		// Pos 1 is the file name
		QString name = lst[0];
	
		// Pos 2 is the size
		QString size = lst[1];
	
		// Pos 3 is TTH
		QString tth = lst[2];
	
		// Create item
		TreeItem* item = new TreeItem( name, parent );
		item->setSize( size.toLongLong() );
		item->setTTH( tth );
		
		//logger->info( "Added file: " + name + " with size: " + size + " and tth: " + tth );
	}
	else {
		//logger->error( "Bad file string format: " + file );
	}
}
