/* GAdmin-OpenVPN - An easy to use GTK+ frontend for the openvpn client.
 * Copyright (C) 2008 - 2010 Magnus Loef <magnus-swe@telia.com> 
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 3 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.
 * See the GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License along
 * with this program; if not, write to the Free Software Foundation, Inc.,
 * 51 Franklin Street, Fifth Floor, Boston, MA 02110-1301, USA.
 *
*/



#include "../config.h"
#include <gtk/gtk.h>
#include <stdlib.h>
#include <stdio.h>
#include <string.h>
#include <errno.h>
#include <sys/wait.h>
#include <unistd.h>
#include "gettext.h"
#include "widgets.h"
#include "allocate.h"
#include "show_info.h"
#include "commands.h"

extern int MAX_READ_POPEN;


void run_command_show_err(gchar *command)
{
    FILE *fp;
    char *line, *info;
    
    if((fp=popen(command, "r"))==NULL)
    {
	perror("popen");
	return;
    }
    else
      {
	  line = allocate(MAX_READ_POPEN+2);
	  info = allocate(MAX_READ_POPEN+2);
	  while(fgets(line, MAX_READ_POPEN, fp)!=NULL)
	     strcat(info, line);

	  pclose(fp);
	  show_info(info);
	  free(info);
	  free(line);
      }
}


int run_command(gchar *command)
{
    FILE *fp;
    int status=0, exit_status=0;
    
    if((fp=popen(command, "w"))==NULL)
    {
	perror("popen");
	return 0;
    }
    status = pclose(fp);

    exit_status = WEXITSTATUS(status);

    if( exit_status > 0 )
      exit_status = 0;
    else
      exit_status = 1;

    return exit_status;
}


int networkmanager_start()
{
    FILE *fp;
    gchar *upscript_path, *upscript, *cmd, *info;

    /* Handle networkmanager if its installed */
    if( ! file_exists(NETWORKMANAGER_DISPATCH_DIR) )
      return 0;

    /* Write an upscript */
    upscript_path = g_strdup_printf("%s/10-gadmin-openvpn-client", NETWORKMANAGER_DISPATCH_DIR);
    if((fp=fopen(upscript_path, "w+"))==NULL)
    {
	info = g_strdup_printf(_("Can not write the openvpn client networkmanager upscript here: \n%s\n"), upscript_path);
	show_info(info);
	g_free(info);
	g_free(upscript_path);
    	return 0;
    }
    upscript = g_strconcat(
    "#!/bin/bash\n",
    "\n",
    "# Makes NetworkManager start gadmin-openvpn-client and restart\n",
    "# the VPN-tunnel if the network connection goes down then up again.\n",
    "\n",
    "# Must set and export path or networkmanager wont run sbin programs.\n",
    "export PATH=/sbin:/usr/sbin:/bin:/usr/bin\n",
    "\n",
    "if [ \"$2\" = \"down\" ]; then\n",
    "   ", SYSINIT_SCRIPTS_DIR, "/gadmin-openvpn-client stop\n",
    "fi\n",
    "\n",
    "if [ \"$2\" = \"up\" ]; then\n",
    "   ", SYSINIT_SCRIPTS_DIR, "/gadmin-openvpn-client stop\n",
    "   ", SYSINIT_SCRIPTS_DIR, "/gadmin-openvpn-client start\n",
    "fi\n",
    "\n",
    NULL);    
    fputs(upscript, fp);
    fclose(fp);
    g_free(upscript);

    /* Chmod the script to 755 */
    cmd = g_strdup_printf("chmod 755 %s", upscript_path);
    if( ! run_command(cmd) )
    {
        printf("Show popup, error chmodding the networkmanager upscript.\n");
	g_free(cmd);
	g_free(upscript_path);
	return 0;
    }
    g_free(cmd);
    g_free(upscript_path);

    return 1;
}


void networkmanager_stop()
{
    gchar *upscript_path;

    upscript_path = g_strdup_printf("%s/10-gadmin-openvpn-client", NETWORKMANAGER_DISPATCH_DIR);

    /* Handle networkmanager if its installed */
    if( ! file_exists(upscript_path) )
    {
	g_free(upscript_path);
        return;
    }

    unlink(upscript_path);

    g_free(upscript_path);
}


void init_start(struct w *widgets)
{
    gchar *cmd;
    gint active_index = 0;

    /* Start the gadmin-openvpn-client sysinit script */
    cmd = g_strdup_printf("%s/gadmin-openvpn-client start", SYSINIT_SCRIPTS_DIR);
    if( strlen(cmd) > 4 )
    {
	if( ! run_command(cmd) )
	{
	    run_command_show_err(cmd);
	}
    }
    g_free(cmd);

    /* Dont start at boot if not specified */
    active_index = gtk_combo_box_get_active(GTK_COMBO_BOX(widgets->client_set_combo[6]));
    if( active_index < 1 )
      return;


    /* Make the client start at boot using NetworkManager if it exists */
    if( networkmanager_start() )
    {
	/* Start using network manager, not sysinit */	
	return;
    }

    /* Make the client start at boot */
    cmd = g_strdup_printf("%s", SYSINIT_START_CMD);
    if( strlen(cmd) > 4 )
    {
	if( ! run_command(cmd) )
	{
	    run_command_show_err(cmd);
	    g_free(cmd);
	    return;
	}
    }
    g_free(cmd);
}

void init_stop(struct w *widgets)
{
    gchar *cmd;

    /* Remove the networkmanager script if any */
    networkmanager_stop();

    /* Make the client not start at boot */
    cmd = g_strdup_printf("%s", SYSINIT_STOP_CMD);
    if( strlen(cmd) > 4 )
    {
	if( ! run_command(cmd) )
	{
	    run_command_show_err(cmd);
	    /* Dont return */
	}
    }
    g_free(cmd);

    /* Stop the gadmin-openvpn-client sysinit script */
    cmd = g_strdup_printf("%s/gadmin-openvpn-client stop", SYSINIT_SCRIPTS_DIR);
    if( strlen(cmd) > 4 )
    {
	if( ! run_command(cmd) )
	{
	    run_command_show_err(cmd);
	}
    }
    g_free(cmd);
}

int file_exists(char *infile)
{
    FILE *fp;

    if((fp=fopen(infile, "r"))==NULL)
      return 0;

    fclose(fp);
    return 1;
}
