static gchar *exe = NULL;

static void set_gerror (GError **error, GbrInitError errcode);


/** Initialize the BinReloc library (for applications).
 *
 * This function must be called before using any other BinReloc functions.
 * It attempts to locate the application's canonical filename.
 *
 * @note If you want to use BinReloc for a library, then you should call
 *       gbr_init_lib() instead.
 * @note Initialization failure is not fatal. BinReloc functions will just
 *       fallback to the supplied default path.
 *
 * @param error  If BinReloc failed to initialize, then the error report will
 *               be stored in this variable. Set to NULL if you don't want an
 *               error report. See the #GbrInitError for a list of error
 *               codes.
 *
 * @returns TRUE on success, FALSE if BinReloc failed to initialize.
 */
gboolean
gbr_init (GError **error)
{
       GbrInitError errcode = 0;

	/* Shut up compiler warning about uninitialized variable. */
	errcode = GBR_INIT_ERROR_NOMEM;

	/* Locate the application's filename. */
	exe = _br_find_exe (&errcode);
	if (exe != NULL)
		/* Success! */
		return TRUE;
	else {
		/* Failed :-( */
		set_gerror (error, errcode);
		return FALSE;
	}
}


/** Initialize the BinReloc library (for libraries).
 *
 * This function must be called before using any other BinReloc functions.
 * It attempts to locate the calling library's canonical filename.
 *
 * @note The BinReloc source code MUST be included in your library, or this
 *       function won't work correctly.
 * @note Initialization failure is not fatal. BinReloc functions will just
 *       fallback to the supplied default path.
 *
 * @returns TRUE on success, FALSE if a filename cannot be found.
 */
gboolean
gbr_init_lib (GError **error)
{
       GbrInitError errcode = 0;

	/* Shut up compiler warning about uninitialized variable. */
	errcode = GBR_INIT_ERROR_NOMEM;

	exe = _br_find_exe_for_symbol ((const void *) "", &errcode);
	if (exe != NULL)
		/* Success! */
		return TRUE;
	else {
		/* Failed :-( */
		set_gerror (error, errcode);
		return exe != NULL;
	}
}


static void
set_gerror (GError **error, GbrInitError errcode)
{
	gchar *error_message;

	if (error == NULL)
		return;

	switch (errcode) {
	case GBR_INIT_ERROR_NOMEM:
		error_message = "Cannot allocate memory.";
		break;
	case GBR_INIT_ERROR_OPEN_MAPS:
		error_message = "Unable to open /proc/self/maps for reading.";
		break;
	case GBR_INIT_ERROR_READ_MAPS:
		error_message = "Unable to read from /proc/self/maps.";
		break;
	case GBR_INIT_ERROR_INVALID_MAPS:
		error_message = "The file format of /proc/self/maps is invalid.";
		break;
	case GBR_INIT_ERROR_DISABLED:
		error_message = "Binary relocation support is disabled.";
		break;
	default:
		error_message = "Unknown error.";
		break;
	};
	g_set_error (error, g_quark_from_static_string ("GBinReloc"),
		     errcode, "%s", error_message);
}


/** Find the canonical filename of the current application.
 *
 * @param default_exe  A default filename which will be used as fallback.
 * @returns A string containing the application's canonical filename,
 *          which must be freed when no longer necessary. If BinReloc is
 *          not initialized, or if the initialization function failed,
 *          then a copy of default_exe will be returned. If default_exe
 *          is NULL, then NULL will be returned.
 */
gchar *
gbr_find_exe (const gchar *default_exe)
{
	if (exe == NULL) {
		/* BinReloc is not initialized. */
		if (default_exe != NULL)
			return g_strdup (default_exe);
		else
			return NULL;
	}
	return g_strdup (exe);
}


/** Locate the directory in which the current application is installed.
 *
 * The prefix is generated by the following pseudo-code evaluation:
 * \code
 * dirname(exename)
 * \endcode
 *
 * @param default_dir  A default directory which will used as fallback.
 * @return A string containing the directory, which must be freed when no
 *         longer necessary. If BinReloc is not initialized, or if the
 *         initialization function failed, then a copy of default_dir
 *         will be returned. If default_dir is NULL, then NULL will be
 *         returned.
 */
gchar *
gbr_find_exe_dir (const gchar *default_dir)
{
	if (exe == NULL) {
		/* BinReloc not initialized. */
		if (default_dir != NULL)
			return g_strdup (default_dir);
		else
			return NULL;
	}

	return g_path_get_dirname (exe);
}


/** Locate the prefix in which the current application is installed.
 *
 * The prefix is generated by the following pseudo-code evaluation:
 * \code
 * dirname(dirname(exename))
 * \endcode
 *
 * @param default_prefix  A default prefix which will used as fallback.
 * @return A string containing the prefix, which must be freed when no
 *         longer necessary. If BinReloc is not initialized, or if the
 *         initialization function failed, then a copy of default_prefix
 *         will be returned. If default_prefix is NULL, then NULL will be
 *         returned.
 */
gchar *
gbr_find_prefix (const gchar *default_prefix)
{
	gchar *dir1, *dir2;

	if (exe == NULL) {
		/* BinReloc not initialized. */
		if (default_prefix != NULL)
			return g_strdup (default_prefix);
		else
			return NULL;
	}

	dir1 = g_path_get_dirname (exe);
	dir2 = g_path_get_dirname (dir1);
	g_free (dir1);
	return dir2;
}


/** Locate the application's binary folder.
 *
 * The path is generated by the following pseudo-code evaluation:
 * \code
 * prefix + "/bin"
 * \endcode
 *
 * @param default_bin_dir  A default path which will used as fallback.
 * @return A string containing the bin folder's path, which must be freed when
 *         no longer necessary. If BinReloc is not initialized, or if the
 *         initialization function failed, then a copy of default_bin_dir will
 *         be returned. If default_bin_dir is NULL, then NULL will be returned.
 */
gchar *
gbr_find_bin_dir (const gchar *default_bin_dir)
{
	gchar *prefix, *dir;

	prefix = gbr_find_prefix (NULL);
	if (prefix == NULL) {
		/* BinReloc not initialized. */
		if (default_bin_dir != NULL)
			return g_strdup (default_bin_dir);
		else
			return NULL;
	}

	dir = g_build_filename (prefix, "bin", NULL);
	g_free (prefix);
	return dir;
}


/** Locate the application's superuser binary folder.
 *
 * The path is generated by the following pseudo-code evaluation:
 * \code
 * prefix + "/sbin"
 * \endcode
 *
 * @param default_sbin_dir  A default path which will used as fallback.
 * @return A string containing the sbin folder's path, which must be freed when
 *         no longer necessary. If BinReloc is not initialized, or if the
 *         initialization function failed, then a copy of default_sbin_dir will
 *         be returned. If default_bin_dir is NULL, then NULL will be returned.
 */
gchar *
gbr_find_sbin_dir (const gchar *default_sbin_dir)
{
	gchar *prefix, *dir;

	prefix = gbr_find_prefix (NULL);
	if (prefix == NULL) {
		/* BinReloc not initialized. */
		if (default_sbin_dir != NULL)
			return g_strdup (default_sbin_dir);
		else
			return NULL;
	}

	dir = g_build_filename (prefix, "sbin", NULL);
	g_free (prefix);
	return dir;
}


/** Locate the application's data folder.
 *
 * The path is generated by the following pseudo-code evaluation:
 * \code
 * prefix + "/share"
 * \endcode
 *
 * @param default_data_dir  A default path which will used as fallback.
 * @return A string containing the data folder's path, which must be freed when
 *         no longer necessary. If BinReloc is not initialized, or if the
 *         initialization function failed, then a copy of default_data_dir
 *         will be returned. If default_data_dir is NULL, then NULL will be
 *         returned.
 */
gchar *
gbr_find_data_dir (const gchar *default_data_dir)
{
	gchar *prefix, *dir;

	prefix = gbr_find_prefix (NULL);
	if (prefix == NULL) {
		/* BinReloc not initialized. */
		if (default_data_dir != NULL)
			return g_strdup (default_data_dir);
		else
			return NULL;
	}

	dir = g_build_filename (prefix, "share", NULL);
	g_free (prefix);
	return dir;
}


/** Locate the application's localization folder.
 *
 * The path is generated by the following pseudo-code evaluation:
 * \code
 * prefix + "/share/locale"
 * \endcode
 *
 * @param default_locale_dir  A default path which will used as fallback.
 * @return A string containing the localization folder's path, which must be freed when
 *         no longer necessary. If BinReloc is not initialized, or if the
 *         initialization function failed, then a copy of default_locale_dir will be returned.
 *         If default_locale_dir is NULL, then NULL will be returned.
 */
gchar *
gbr_find_locale_dir (const gchar *default_locale_dir)
{
	gchar *data_dir, *dir;

	data_dir = gbr_find_data_dir (NULL);
	if (data_dir == NULL) {
		/* BinReloc not initialized. */
		if (default_locale_dir != NULL)
			return g_strdup (default_locale_dir);
		else
			return NULL;
	}

	dir = g_build_filename (data_dir, "locale", NULL);
	g_free (data_dir);
	return dir;
}


/** Locate the application's library folder.
 *
 * The path is generated by the following pseudo-code evaluation:
 * \code
 * prefix + "/lib"
 * \endcode
 *
 * @param default_lib_dir  A default path which will used as fallback.
 * @return A string containing the library folder's path, which must be freed when
 *         no longer necessary. If BinReloc is not initialized, or if the
 *         initialization function failed, then a copy of default_lib_dir will be returned.
 *         If default_lib_dir is NULL, then NULL will be returned.
 */
gchar *
gbr_find_lib_dir (const gchar *default_lib_dir)
{
	gchar *prefix, *dir;

	prefix = gbr_find_prefix (NULL);
	if (prefix == NULL) {
		/* BinReloc not initialized. */
		if (default_lib_dir != NULL)
			return g_strdup (default_lib_dir);
		else
			return NULL;
	}

	dir = g_build_filename (prefix, "lib", NULL);
	g_free (prefix);
	return dir;
}


/** Locate the application's libexec folder.
 *
 * The path is generated by the following pseudo-code evaluation:
 * \code
 * prefix + "/libexec"
 * \endcode
 *
 * @param default_libexec_dir  A default path which will used as fallback.
 * @return A string containing the libexec folder's path, which must be freed when
 *         no longer necessary. If BinReloc is not initialized, or if the initialization
 *         function failed, then a copy of default_libexec_dir will be returned.
 *         If default_libexec_dir is NULL, then NULL will be returned.
 */
gchar *
gbr_find_libexec_dir (const gchar *default_libexec_dir)
{
	gchar *prefix, *dir;

	prefix = gbr_find_prefix (NULL);
	if (prefix == NULL) {
		/* BinReloc not initialized. */
		if (default_libexec_dir != NULL)
			return g_strdup (default_libexec_dir);
		else
			return NULL;
	}

	dir = g_build_filename (prefix, "libexec", NULL);
	g_free (prefix);
	return dir;
}


/** Locate the application's configuration files folder.
 *
 * The path is generated by the following pseudo-code evaluation:
 * \code
 * prefix + "/etc"
 * \endcode
 *
 * @param default_etc_dir  A default path which will used as fallback.
 * @return A string containing the etc folder's path, which must be freed when
 *         no longer necessary. If BinReloc is not initialized, or if the initialization
 *         function failed, then a copy of default_etc_dir will be returned.
 *         If default_etc_dir is NULL, then NULL will be returned.
 */
gchar *
gbr_find_etc_dir (const gchar *default_etc_dir)
{
	gchar *prefix, *dir;

	prefix = gbr_find_prefix (NULL);
	if (prefix == NULL) {
		/* BinReloc not initialized. */
		if (default_etc_dir != NULL)
			return g_strdup (default_etc_dir);
		else
			return NULL;
	}

	dir = g_build_filename (prefix, "etc", NULL);
	g_free (prefix);
	return dir;
}
