/*
 * This file is part of the Ubuntu TV Media Scanner
 * Copyright (C) 2012-2013 Canonical Ltd.
 *
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU Lesser General Public License version 3 as
 * published by the Free Software Foundation.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public License
 * along with this program.  If not, see <http://www.gnu.org/licenses/>.
 *
 * Contact: Jim Hodapp <jim.hodapp@canonical.com>
 * Authored by: Mathias Hasselmann <mathias@openismus.com>
 */
#ifndef MEDIASCANNER_REFRESHPOLICY_H
#define MEDIASCANNER_REFRESHPOLICY_H

// Media Scanner Library
#include "mediascanner/declarations.h"

namespace mediascanner {

/**
 * @brief Shared pointer holding a RefreshPolicy.
 */
typedef std::shared_ptr<RefreshPolicy> RefreshPolicyPtr;

/**
 * @brief A refresh policy decides when changes to the media index must be
 * re-read. A policy is needed because, to optimize performance,
 * Lucene++ doesn't automatically refresh its index readers upon changes.
 */
class RefreshPolicy {
public:
    virtual ~RefreshPolicy();

    /**
     * @brief The default policy - currently an instance of InstantRefreshPolicy.
     */
    static RefreshPolicyPtr default_policy();

    /**
     * @brief This method is called by the MediaIndex before it starts any
     * read operation. The policy can now check if the media index needs to
     * be reopened, can can do so if needed.
     * @param index
     */
    virtual bool OnBeginReading(MediaIndex *index) = 0;

    /**
     * @brief This method is called by the WritableMediaIndex before it
     * starts any write operation. The policy can now check if the media index
     * needs to be reopened, can can do so if needed.
     * @param index
     */
    virtual bool OnBeginWriting(WritableMediaIndex *index) = 0;
};

class InstantRefreshPolicy : public RefreshPolicy {
public:
    bool OnBeginReading(MediaIndex *index);
    bool OnBeginWriting(WritableMediaIndex *index);
};

} // namespace mediascanner

#endif // MEDIASCANNER_REFRESHPOLICY_H
