/*
 * COPYRIGHT (c) International Business Machines Corp. 2021-2024
 *
 * This program is provided under the terms of the Common Public License,
 * version 1.0 (CPL-1.0). Any use, reproduction or distribution for this
 * software constitutes recipient's acceptance of CPL-1.0 terms which can be
 * found in the file LICENSE file or at
 * https://opensource.org/licenses/cpl1.0.php
 */
#ifndef LIB_KMIPCLIENT_H
#define LIB_KMIPCLIENT_H

#include <stdint.h>
#include <stdbool.h>

#include <openssl/bn.h>

enum kmip_tag {
	KMIP_TAG_ACTIVATION_DATE			= 0x420001,
	KMIP_TAG_APPLICATION_DATA			= 0x420002,
	KMIP_TAG_APPLICATION_NAMESPACE			= 0x420003,
	KMIP_TAG_APPLICATION_SPECIFIC_INFORMATION	= 0x420004,
	KMIP_TAG_ARCHIVE_DATE				= 0x420005, /* deprecated since v1.1 */
	KMIP_TAG_ASYNCHRONOUS_CORRELATION_VALUE		= 0x420006,
	KMIP_TAG_ASYNCHRONOUS_INDICATOR			= 0x420007,
	KMIP_TAG_ATTRIBUTE				= 0x420008,
	KMIP_TAG_ATTRIBUTE_INDEX			= 0x420009, /* v1.x only */
	KMIP_TAG_ATTRIBUTE_NAME				= 0x42000A,
	KMIP_TAG_ATTRIBUTE_VALUE			= 0x42000B,
	KMIP_TAG_AUTHENTICATION				= 0x42000C,
	KMIP_TAG_BATCH_COUNT				= 0x42000D,
	KMIP_TAG_BATCH_ERROR_CONTINUATION_OPTION	= 0x42000E,
	KMIP_TAG_BATCH_ITEM				= 0x42000F,
	KMIP_TAG_BATCH_ORDER_OPTION			= 0x420010,
	KMIP_TAG_BLOCK_CIPHER_MODE			= 0x420011,
	KMIP_TAG_CANCELATION_RESULT			= 0x420012,
	KMIP_TAG_CERTIFICATE				= 0x420013,
	KMIP_TAG_CERTIFICATE_IDENTIFIER			= 0x420014, /* deprecated since v1.1 */
	KMIP_TAG_CERTIFICATE_ISSUER			= 0x420015, /* deprecated since v1.1 */
	KMIP_TAG_CERTIFICATE_ISSUER_ALTERNATIVE_NAME	= 0x420016, /* deprecated since v1.1 */
	KMIP_TAG_CERTIFICATE_ISSUER_DISTINGUISHED_NAME	= 0x420017, /* deprecated since v1.1 */
	KMIP_TAG_CERTIFICATE_REQUEST			= 0x420018,
	KMIP_TAG_CERTIFICATE_REQUEST_TYPE		= 0x420019,
	KMIP_TAG_CERTIFICATE_SUBJECT			= 0x42001A, /* deprecated since v1.1 */
	KMIP_TAG_CERTIFICATE_SUBJECT_ALTERNATIVE_NAME	= 0x42001B, /* deprecated since v1.1 */
	KMIP_TAG_CERTIFICATE_SUBJECT_DISTINGUISHED_NAME	= 0x42001C, /* deprecated since v1.1 */
	KMIP_TAG_CERTIFICATE_TYPE			= 0x42001D,
	KMIP_TAG_CERTIFICATE_VALUE			= 0x42001E,
	KMIP_TAG_COMMON_TEMPLATE_ATTRIBUTE		= 0x42001F, /* v1.x only */
	KMIP_TAG_COMPROMIZE_DATE			= 0x420020,
	KMIP_TAG_COMPROMISE_OCCURRENCE_DATE		= 0x420021,
	KMIP_TAG_CONTACT_INFORMATION			= 0x420022,
	KMIP_TAG_CREDENTIAL				= 0x420023,
	KMIP_TAG_CREDENTIAL_TYPE			= 0x420024,
	KMIP_TAG_CREDENTIAL_VALUE			= 0x420025,
	KMIP_TAG_CRITICALITY_INDICATOR			= 0x420026,
	KMIP_TAG_CRT_Coefficient			= 0x420027,
	KMIP_TAG_CRYPTOGRAPHIC_ALGORITHM		= 0x420028,
	KMIP_TAG_CRYPTOGRAPHIC_DOMAIN_PARAMETERS	= 0x420029,
	KMIP_TAG_CRYPTOGRAPHIC_LENGTH			= 0x42002A,
	KMIP_TAG_CRYPTOGRAPHIC_PARAMETERS		= 0x42002B,
	KMIP_TAG_CRYPTOGRAPHIC_USAGE_MASK		= 0x42002C,
	KMIP_TAG_CUSTOM_ATTRIBUTE			= 0x42002D, /* v1.x only */
	KMIP_TAG_D					= 0x42002E,
	KMIP_TAG_DEACTIVATION_DATE			= 0x42002F,
	KMIP_TAG_DERIVATION_DATE			= 0x420030,
	KMIP_TAG_DERIVATION_DATA			= 0x420031,
	KMIP_TAG_DERIVATION_PARAMETERS			= 0x420032,
	KMIP_TAG_DESTROY_DATE				= 0x420033,
	KMIP_TAG_DIGEST					= 0x420034,
	KMIP_TAG_DIGEST_VALUE				= 0x420035,
	KMIP_TAG_ENCRYPTION_KEY_INFORMATION		= 0x420036,
	KMIP_TAG_G					= 0x420037,
	KMIP_TAG_HASHING_ALGORITHM			= 0x420038,
	KMIP_TAG_INITIAL_DATE				= 0x420039,
	KMIP_TAG_INITIALIZATION_VECTOR			= 0x42003A,
	KMIP_TAG_ISSUER					= 0x42003B, /* deprecated since v1.1 */
	KMIP_TAG_ITERATION_COUNT			= 0x42003C,
	KMIP_TAG_IV_COUNTER_NONCE			= 0x42003D,
	KMIP_TAG_J					= 0x42003E,
	KMIP_TAG_KEY					= 0x42003F,
	KMIP_TAG_KEY_BLOCK				= 0x420040,
	KMIP_TAG_KEY_COMPRESSION_TYPE			= 0x420041,
	KMIP_TAG_KEY_FORMAT_TYPE			= 0x420042,
	KMIP_TAG_KEY_MATERIAL				= 0x420043,
	KMIP_TAG_KEY_PART_IDENTIFIER			= 0x420044,
	KMIP_TAG_KEY_VALUE				= 0x420045,
	KMIP_TAG_KEY_WRAPPING_DATA			= 0x420046,
	KMIP_TAG_KEY_WRAPPING_SPECIFICATION		= 0x420047,
	KMIP_TAG_LAST_CHANGE_DATE			= 0x420048,
	KMIP_TAG_LEASE_TIME				= 0x420049,
	KMIP_TAG_LINK					= 0x42004A,
	KMIP_TAG_LINK_TYPE				= 0x42004B,
	KMIP_TAG_LINKED_OBJECT_IDENTIFIER		= 0x42004C,
	KMIP_TAG_MAC_SIGNATURE				= 0x42004D,
	KMIP_TAG_MAC_SIGNATURE_KEY_INFORMATION		= 0x42004E,
	KMIP_TAG_MAXIMUM_ITEMS				= 0x42004F,
	KMIP_TAG_MAXIMUM_RESPONSE_SIZE			= 0x420050,
	KMIP_TAG_MESSAGE_EXTENSION			= 0x420051,
	KMIP_TAG_MODULUS				= 0x420052,
	KMIP_TAG_NAME					= 0x420053,
	KMIP_TAG_NAME_TYPE				= 0x420054,
	KMIP_TAG_NAME_VALUE				= 0x420055,
	KMIP_TAG_OBJECT_GROUP				= 0x420056,
	KMIP_TAG_OBJECT_TYPE				= 0x420057,
	KMIP_TAG_OFFSET					= 0x420058,
	KMIP_TAG_OPAQUE_DATA_TYPE			= 0x420059,
	KMIP_TAG_OPAQUE_DATA_VALUE			= 0x42005A,
	KMIP_TAG_OPAQUE_OBJECT				= 0x42005B,
	KMIP_TAG_OPERATION				= 0x42005C,
	KMIP_TAG_OPERATION_POLICY_NAME			= 0x42005D, /* deprecated since v1.3 */
	KMIP_TAG_P					= 0x42005E,
	KMIP_TAG_PADDING_METHOD				= 0x42005F,
	KMIP_TAG_PRIME_EXPONENT_P			= 0x420060,
	KMIP_TAG_PRIME_EXPONENT_Q			= 0x420061,
	KMIP_TAG_PRIME_FIELD_SIZE			= 0x420062,
	KMIP_TAG_PRIVATE_EXPONENT			= 0x420063,
	KMIP_TAG_PRIVATE_KEY				= 0x420064,
	KMIP_TAG_PRIVATE_KEY_TEMPLATE_ATTRIBUTE		= 0x420065, /* v1.x only */
	KMIP_TAG_PRIVATE_KEY_UNIQUE_IDENTIFIER		= 0x420066,
	KMIP_TAG_PROCESS_START_DATE			= 0x420067,
	KMIP_TAG_PROTECT_STOP_DATE			= 0x420068,
	KMIP_TAG_PROTOCOL_VERSION			= 0x420069,
	KMIP_TAG_PROTOCOL_VERSION_MAJOR			= 0x42006A,
	KMIP_TAG_PROTOCOL_VERSION_MINOR			= 0x42006B,
	KMIP_TAG_PUBLIC_EXPONENT			= 0x42006C,
	KMIP_TAG_PUBLIC_KEY				= 0x42006D,
	KMIP_TAG_PUBLIC_KEY_TEMPLATE_ATTRIBUTE		= 0x42006E, /* v1.x only */
	KMIP_TAG_PUBLIC_KEY_UNIQUE_IDENTIFIER		= 0x42006F,
	KMIP_TAG_PUT_FUNCTION				= 0x420070,
	KMIP_TAG_Q					= 0x420071,
	KMIP_TAG_Q_STRING				= 0x420072,
	KMIP_TAG_Q_LENGTH				= 0x420073,
	KMIP_TAG_QUERY_FUNCTION				= 0x420074,
	KMIP_TAG_RECOMMENDED_CURVE			= 0x420075,
	KMIP_TAG_REPLACED_UNIQUE_IDENTIFIER		= 0x420076,
	KMIP_TAG_REQUEST_HEADER				= 0x420077,
	KMIP_TAG_REQUEST_MESSAGE			= 0x420078,
	KMIP_TAG_REQUEST_PAYLOAD			= 0x420079,
	KMIP_TAG_RESPONSE_HEADER			= 0x42007A,
	KMIP_TAG_RESPONSE_MESSAGE			= 0x42007B,
	KMIP_TAG_RESPONSE_PAYLOAD			= 0x42007C,
	KMIP_TAG_RESULT_MESSAGE				= 0x42007D,
	KMIP_TAG_RESULT_REASON				= 0x42007E,
	KMIP_TAG_RESULT_STATUS				= 0x42007F,
	KMIP_TAG_REVOCATION_MESSAGE			= 0x420080,
	KMIP_TAG_REVOCATION_REASON			= 0x420081,
	KMIP_TAG_REVOCATION_REASON_CODE			= 0x420082,
	KMIP_TAG_KEY_ROLE_TYPE				= 0x420083,
	KMIP_TAG_SALT					= 0x420084,
	KMIP_TAG_SECRET_DATA				= 0x420085,
	KMIP_TAG_SECRET_DATA_TYPE			= 0x420086,
	KMIP_TAG_SERIAL_NUMBER				= 0x420087, /* deprecated since v1.1 */
	KMIP_TAG_SERVER_INFORMATION			= 0x420088,
	KMIP_TAG_SPLIT_KEY				= 0x420089,
	KMIP_TAG_SPLIT_KEY_METHOD			= 0x42008A,
	KMIP_TAG_SPLIT_KEY_PARTS			= 0x42008B,
	KMIP_TAG_SPLIT_KEY_THRESHOLD			= 0x42008C,
	KMIP_TAG_STATE					= 0x42008D,
	KMIP_TAG_STORAGE_STATUS_MASK			= 0x42008E,
	KMIP_TAG_SYMMETRIC_KEY				= 0x42008F,
	KMIP_TAG_TEMPLATE				= 0x420090, /* v1.x only */
	KMIP_TAG_TEMPLATE_ATTRIBUTE			= 0x420091, /* v1.x only */
	KMIP_TAG_TIME_STAMP				= 0x420092,
	KMIP_TAG_UNIQUE_BATCH_ITEM_ID			= 0x420093,
	KMIP_TAG_UNIQUE_IDENTIFIER			= 0x420094,
	KMIP_TAG_USAGE_LIMITS				= 0x420095,
	KMIP_TAG_USAGE_LIMITS_COUNT			= 0x420096,
	KMIP_TAG_USAGE_LIMITS_TOTAL			= 0x420097,
	KMIP_TAG_USAGE_LIMITS_UNIT			= 0x420098,
	KMIP_TAG_USERNAME				= 0x420099,
	KMIP_TAG_VALIDITY_DATE				= 0x42009A,
	KMIP_TAG_VALIDITY_INDICATOR			= 0x42009B,
	KMIP_TAG_VENDOR_EXTENSION			= 0x42009C,
	KMIP_TAG_VENDOR_IDENTIFICATION			= 0x42009D,
	KMIP_TAG_WRAPPING_METHOD			= 0x42009E,
	KMIP_TAG_X					= 0x42009F,
	KMIP_TAG_Y					= 0x4200A0,
	KMIP_TAG_PASSWORD				= 0x4200A1,
	KMIP_TAG_DEVICE_IDENTIFIER			= 0x4200A2, /* since v1.2 */
	KMIP_TAG_ENCODING_OPTION			= 0x4200A3, /* since v1.2 */
	KMIP_TAG_EXTENSION_INFORMATION			= 0x4200A4, /* since v1.2 */
	KMIP_TAG_EXTENSION_NAME				= 0x4200A5, /* since v1.2 */
	KMIP_TAG_EXTENSION_TAG				= 0x4200A6, /* since v1.2 */
	KMIP_TAG_EXTENSION_TYPE				= 0x4200A7, /* since v1.2 */
	KMIP_TAG_FRESH					= 0x4200A8, /* since v1.2 */
	KMIP_TAG_MACHINE_IDENTIFIER			= 0x4200A9, /* since v1.2 */
	KMIP_TAG_MEDIA_IDENTIFIER			= 0x4200AA, /* since v1.2 */
	KMIP_TAG_NETWORK_IDENTIFIER			= 0x4200AB, /* since v1.2 */
	KMIP_TAG_OBJECT_GROUP_MEMBER			= 0x4200AC, /* since v1.2 */
	KMIP_TAG_CERTIFICATE_LENGTH			= 0x4200AD, /* since v1.2 */
	KMIP_TAG_DIGITAL_SIGNATURE_ALGORITHM		= 0x4200AE, /* since v1.2 */
	KMIP_TAG_CERTIFICATE_SERIAL_NUMBER		= 0x4200AF, /* since v1.2 */
	KMIP_TAG_DEVICE_SERIAL_NUMBER			= 0x4200B0, /* since v1.2 */
	KMIP_TAG_ISSUER_ALTERNATE_NAME			= 0x4200B1, /* since v1.2 */
	KMIP_TAG_ISSUER_DISTINGUISHED_NAME		= 0x4200B2, /* since v1.2 */
	KMIP_TAG_SUBJECT_ALTERNATE_NAME			= 0x4200B3, /* since v1.2 */
	KMIP_TAG_SUBJECT_DISTINGUISHED_NAME		= 0x4200B4, /* since v1.2 */
	KMIP_TAG_X_509_CERTIFICATE_IDENTIFIER		= 0x4200B5, /* since v1.2 */
	KMIP_TAG_X_509_CERTIFICATE_ISSUER		= 0x4200B6, /* since v1.2 */
	KMIP_TAG_X_509_CERTIFICATE_SUBJECT		= 0x4200B7, /* since v1.2 */
	KMIP_TAG_KEY_VALUE_LOCATION			= 0x4200B8, /* since v1.2 */
	KMIP_TAG_KEY_VALUE_LOCATION_VALUE		= 0x4200B9, /* since v1.2 */
	KMIP_TAG_KEY_VALUE_LOCATION_TYPE		= 0x4200BA, /* since v1.2 */
	KMIP_TAG_KEY_VALUE_PRESENT			= 0x4200BB, /* since v1.2 */
	KMIP_TAG_ORIGINAL_CREATION_DATE			= 0x4200BC, /* since v1.2 */
	KMIP_TAG_PGP_KEY				= 0x4200BD, /* since v1.2 */
	KMIP_TAG_PGP_KEY_VERSION			= 0x4200BE, /* since v1.2 */
	KMIP_TAG_ALTERNATE_NAME				= 0x4200BF, /* since v1.2 */
	KMIP_TAG_ALTERNATE_NAME_VALUE			= 0x4200C0, /* since v1.2 */
	KMIP_TAG_ALTERNATE_NAME_TYPE			= 0x4200C1, /* since v1.2 */
	KMIP_TAG_DATA					= 0x4200C2, /* since v1.2 */
	KMIP_TAG_SIGNATURE_DATA				= 0x4200C3, /* since v1.2 */
	KMIP_TAG_DATA_LENGTH				= 0x4200C4, /* since v1.2 */
	KMIP_TAG_RANDOM_IV				= 0x4200C5, /* since v1.2 */
	KMIP_TAG_MAC_DATA				= 0x4200C6, /* since v1.2 */
	KMIP_TAG_ATTESTATION_TYPE			= 0x4200C7, /* since v1.2 */
	KMIP_TAG_NONCE					= 0x4200C8, /* since v1.2 */
	KMIP_TAG_NONCE_ID				= 0x4200C9, /* since v1.2 */
	KMIP_TAG_NONCE_VALUE				= 0x4200CA, /* since v1.2 */
	KMIP_TAG_ATTESTATION_MEASUREMENT		= 0x4200CB, /* since v1.2 */
	KMIP_TAG_ATTESTATION_ASSERTION			= 0x4200CC, /* since v1.2 */
	KMIP_TAG_IV_LENGTH				= 0x4200CD, /* since v1.2 */
	KMIP_TAG_TAG_LENGTH				= 0x4200CE, /* since v1.2 */
	KMIP_TAG_FIXED_FIELD_LENGTH			= 0x4200CF, /* since v1.2 */
	KMIP_TAG_COUNTER_LENGTH				= 0x4200D0, /* since v1.2 */
	KMIP_TAG_INITIAL_COUNTER_VALUE			= 0x4200D1, /* since v1.2 */
	KMIP_TAG_INVOCATION_FIELD_LENGTH		= 0x4200D2, /* since v1.2 */
	KMIP_TAG_ATTESTATION_CAPABLE_INDICATOR		= 0x4200D3, /* since v1.2 */
	KMIP_TAG_OFFSET_ITEMS				= 0x4200D4, /* since v1.3 */
	KMIP_TAG_LOCATED_ITEMS				= 0x4200D5, /* since v1.3 */
	KMIP_TAG_CORRELATION_VALUE			= 0x4200D6, /* since v1.3 */
	KMIP_TAG_INIT_INDICATOR				= 0x4200D7, /* since v1.3 */
	KMIP_TAG_FINAL_INDICATOR			= 0x4200D8, /* since v1.3 */
	KMIP_TAG_RNG_PARAMETERS				= 0x4200D9, /* since v1.3 */
	KMIP_TAG_RNG_ALGORITHM				= 0x4200DA, /* since v1.3 */
	KMIP_TAG_DRBG_ALGORITHM				= 0x4200DB, /* since v1.3 */
	KMIP_TAG_FIPS186_VARIANT			= 0x4200DC, /* since v1.3 */
	KMIP_TAG_PREDICTION_RESISTANCE			= 0x4200DD, /* since v1.3 */
	KMIP_TAG_RANDOM_NUMBER_GENERATOR		= 0x4200DE, /* since v1.3 */
	KMIP_TAG_VALIDATION_INFORMATION			= 0x4200DF, /* since v1.3 */
	KMIP_TAG_VALIDATION_AUTHORITY_TYPE		= 0x4200E0, /* since v1.3 */
	KMIP_TAG_VALIDATION_AUTHORITY_COUNTRY		= 0x4200E1, /* since v1.3 */
	KMIP_TAG_VALIDATION_AUTHORITY_URI		= 0x4200E2, /* since v1.3 */
	KMIP_TAG_VALIDATION_VERSION_MAJOR		= 0x4200E3, /* since v1.3 */
	KMIP_TAG_VALIDATION_VERSION_MINOR		= 0x4200E4, /* since v1.3 */
	KMIP_TAG_VALIDATION_TYPE			= 0x4200E5, /* since v1.3 */
	KMIP_TAG_VALIDATION_LEVEL			= 0x4200E6, /* since v1.3 */
	KMIP_TAG_VALIDATION_CERTIFICATE_IDENTIFIER	= 0x4200E7, /* since v1.3 */
	KMIP_TAG_VALIDATION_CERTIFICATE_URI		= 0x4200E8, /* since v1.3 */
	KMIP_TAG_VALIDATION_VENDOR_URI			= 0x4200E9, /* since v1.3 */
	KMIP_TAG_VALIDATION_PROFILE			= 0x4200EA, /* since v1.3 */
	KMIP_TAG_PROFILE_INFORMATION			= 0x4200EB, /* since v1.3 */
	KMIP_TAG_PROFILE_NAME				= 0x4200EC, /* since v1.3 */
	KMIP_TAG_SERVER_URI				= 0x4200ED, /* since v1.3 */
	KMIP_TAG_SERVER_PORT				= 0x4200EE, /* since v1.3 */
	KMIP_TAG_STREAMING_CAPABILITY			= 0x4200EF, /* since v1.3 */
	KMIP_TAG_ASYNCHRONOUS_CAPABILITY		= 0x4200F0, /* since v1.3 */
	KMIP_TAG_ATTESTATION_CAPABILITY			= 0x4200F1, /* since v1.3 */
	KMIP_TAG_UNWRAP_MODE				= 0x4200F2, /* since v1.3 */
	KMIP_TAG_DESTROY_ACTION				= 0x4200F3, /* since v1.3 */
	KMIP_TAG_SHREDDING_ALGORITHM			= 0x4200F4, /* since v1.3 */
	KMIP_TAG_RNG_MODE				= 0x4200F5, /* since v1.3 */
	KMIP_TAG_CLIENT_REGISTRATION_METHOD		= 0x4200F6, /* since v1.3 */
	KMIP_TAG_CAPABILITY_INFORMATION			= 0x4200F7, /* since v1.3 */
	KMIP_TAG_KEY_WRAP_TYPE				= 0x4200F8, /* since v1.4 */
	KMIP_TAG_BATCH_UNDO_CAPABILITY			= 0x4200F9, /* since v1.4 */
	KMIP_TAG_BATCH_CONTINUE_CAPABILITY		= 0x4200FA, /* since v1.4 */
	KMIP_TAG_PKCS_12_FRIENDLY_NAME			= 0x4200FB, /* since v1.4 */
	KMIP_TAG_DESCRIPTION				= 0x4200FC, /* since v1.4 */
	KMIP_TAG_COMMENT				= 0x4200FD, /* since v1.4 */
	KMIP_TAG_AUTHENTICATED_ENCRYPTION_ADDITIONAL_DATA = 0x4200FE, /* since v1.4 */
	KMIP_TAG_AUTHENTICTAED_ENCRYPTION_TAG		= 0x4200FF, /* since v1.4 */
	KMIP_TAG_SALT_LENGTH				= 0x420100, /* since v1.4 */
	KMIP_TAG_MASK_GENERATOR				= 0x420101, /* since v1.4 */
	KMIP_TAG_MASK_GENERATOR_HASHING_ALGORITHM	= 0x420102, /* since v1.4 */
	KMIP_TAG_P_SOURCE				= 0x420103, /* since v1.4 */
	KMIP_TAG_TRAILER_FIELD				= 0x420104, /* since v1.4 */
	KMIP_TAG_CLIENT_CORRELATION_VALUE		= 0x420105, /* since v1.4 */
	KMIP_TAG_SERVER_CORRELATION_VALUE		= 0x420106, /* since v1.4 */
	KMIP_TAG_DIGESTED_DATA				= 0x420107, /* since v1.4 */
	KMIP_TAG_CERTIFICATE_SUBJECT_CN			= 0x420108, /* since v1.4 */
	KMIP_TAG_CERTIFICATE_SUBJECT_O			= 0x420109, /* since v1.4 */
	KMIP_TAG_CERTIFICATE_SUBJECT_OU			= 0x42010A, /* since v1.4 */
	KMIP_TAG_CERTIFICATE_SUBJECT_EMAIL		= 0x42010B, /* since v1.4 */
	KMIP_TAG_CERTIFICATE_SUBJECT_C			= 0x42010C, /* since v1.4 */
	KMIP_TAG_CERTIFICATE_SUBJECT_ST			= 0x42010D, /* since v1.4 */
	KMIP_TAG_CERTIFICATE_SUBJECT_L			= 0x42010E, /* since v1.4 */
	KMIP_TAG_CERTIFICATE_SUBJECT_UID		= 0x42010F, /* since v1.4 */
	KMIP_TAG_CERTIFICATE_SUBJECT_SERIAL_NUMBER	= 0x420110, /* since v1.4 */
	KMIP_TAG_CERTIFICATE_SUBJECT_TITLE		= 0x420111, /* since v1.4 */
	KMIP_TAG_CERTIFICATE_SUBJECT_DC			= 0x420112, /* since v1.4 */
	KMIP_TAG_CERTIFICATE_SUBJECT_DN_QUALIFIER	= 0x420113, /* since v1.4 */
	KMIP_TAG_CERTIFICATE_ISSUER_CN			= 0x420114, /* since v1.4 */
	KMIP_TAG_CERTIFICATE_ISSUER_O			= 0x420115, /* since v1.4 */
	KMIP_TAG_CERTIFICATE_ISSUER_OU			= 0x420116, /* since v1.4 */
	KMIP_TAG_CERTIFICATE_ISSUER_EMAIL		= 0x420117, /* since v1.4 */
	KMIP_TAG_CERTIFICATE_ISSUER_C			= 0x420118, /* since v1.4 */
	KMIP_TAG_CERTIFICATE_ISSUER_ST			= 0x420119, /* since v1.4 */
	KMIP_TAG_CERTIFICATE_ISSUER_L			= 0x42011A, /* since v1.4 */
	KMIP_TAG_CERTIFICATE_ISSUER_UID			= 0x42011B, /* since v1.4 */
	KMIP_TAG_CERTIFICATE_ISSUER_SERIAL_NUMBER	= 0x42011C, /* since v1.4 */
	KMIP_TAG_CERTIFICATE_ISSUER_TITLE		= 0x42011D, /* since v1.4 */
	KMIP_TAG_CERTIFICATE_ISSUER_DC			= 0x42011E, /* since v1.4 */
	KMIP_TAG_CERTIFICATE_ISSUER_DN_QUALIFIER	= 0x42011F, /* since v1.4 */
	KMIP_TAG_SENSITIVE				= 0x420120, /* since v1.4 */
	KMIP_TAG_ALWAYS_SENSITIVE			= 0x420121, /* since v1.4 */
	KMIP_TAG_EXTRACTABLE				= 0x420122, /* since v1.4 */
	KMIP_TAG_NEVER_EXTRACTABLE			= 0x420123, /* since v1.4 */
	KMIP_TAG_REPLACE_EXISTING			= 0x420124, /* since v1.4 */
	KMIP_TAG_ATTRIBUTES				= 0x420125, /* since v2.0 */
	KMIP_TAG_COMMON_ATTRIBUTES			= 0x420126, /* since v2.0 */
	KMIP_TAG_PRIVATE_KEY_ATTRIBUTES			= 0x420127, /* since v2.0 */
	KMIP_TAG_PUBLIC_KEY_ATTRIBUTES			= 0x420128, /* since v2.0 */
	KMIP_TAG_EXTENSION_ENUMERATION			= 0x420129, /* since v2.0 */
	KMIP_TAG_EXTENSION_ATTRIBUTE			= 0x42012A, /* since v2.0 */
	KMIP_TAG_EXTENSION_PARENT_STRUCTURE_TAG		= 0x42012B, /* since v2.0 */
	KMIP_TAG_EXTENSION_DESCRIPTION			= 0x42012C, /* since v2.0 */
	KMIP_TAG_SERVER_NAME				= 0x42012D, /* since v2.0 */
	KMIP_TAG_SERVER_SERIAL_NUMBER			= 0x42012E, /* since v2.0 */
	KMIP_TAG_SERVER_VERSION				= 0x42012F, /* since v2.0 */
	KMIP_TAG_SERVER_LOAD				= 0x420130, /* since v2.0 */
	KMIP_TAG_PRODUCT_NAME				= 0x420131, /* since v2.0 */
	KMIP_TAG_BUILD_LEVEL				= 0x420132, /* since v2.0 */
	KMIP_TAG_BUILD_DATE				= 0x420133, /* since v2.0 */
	KMIP_TAG_CLUSTER_INFO				= 0x420134, /* since v2.0 */
	KMIP_TAG_ALTERNATE_FAILOVER_ENDPOINTS		= 0x420135, /* since v2.0 */
	KMIP_TAG_SHORT_UNIQUE_IDENTIFIER		= 0x420136, /* since v2.0 */
	KMIP_TAG_TAG					= 0x420138, /* since v2.0 */
	KMIP_TAG_CERTIFICATE_REQUEST_UNIQUE_IDENTIFIER = 0x420139, /* since v2.0 */
	KMIP_TAG_NIST_KEY_TYPE				= 0x42013A, /* since v2.0 */
	KMIP_TAG_ATTRIBUTE_REFERENCE			= 0x42013B, /* since v2.0 */
	KMIP_TAG_CURRENT_ATTRIBUTE			= 0x42013C, /* since v2.0 */
	KMIP_TAG_NEW_ATTRIBUTE				= 0x42013D, /* since v2.0 */
	KMIP_TAG_CERTIFICATE_REQUEST_VALUE		= 0x420140, /* since v2.0 */
	KMIP_TAG_LOG_MESSAGE				= 0x420141, /* since v2.0 */
	KMIP_TAG_PROFILE_VERSION			= 0x420142, /* since v2.0 */
	KMIP_TAG_PROFILE_VERSION_MAJOR			= 0x420143, /* since v2.0 */
	KMIP_TAG_PROFILE_VERSION_MINOR			= 0x420144, /* since v2.0 */
	KMIP_TAG_PROTECTION_LEVEL			= 0x420145, /* since v2.0 */
	KMIP_TAG_PROTECTION_PERIOD			= 0x420146, /* since v2.0 */
	KMIP_TAG_QUANTUM_SAFE				= 0x420147, /* since v2.0 */
	KMIP_TAG_QUANTUM_SAFE_CAPABILITY		= 0x420148, /* since v2.0 */
	KMIP_TAG_TICKET					= 0x420149, /* since v2.0 */
	KMIP_TAG_TICKET_TYPE				= 0x42014A, /* since v2.0 */
	KMIP_TAG_TICKET_VALUE				= 0x42014B, /* since v2.0 */
	KMIP_TAG_REQUEST_COUNT				= 0x42014C, /* since v2.0 */
	KMIP_TAG_RIGHTS					= 0x42014D, /* since v2.0 */
	KMIP_TAG_OBJECTS				= 0x42014E, /* since v2.0 */
	KMIP_TAG_OPERATIONS				= 0x42014F, /* since v2.0 */
	KMIP_TAG_RIGHT					= 0x420150, /* since v2.0 */
	KMIP_TAG_ENDPOINT_ROLE				= 0x420151, /* since v2.0 */
	KMIP_TAG_DEFAULTS_INFORMATION			= 0x420152, /* since v2.0 */
	KMIP_TAG_OBJECT_DEFAULTS			= 0x420153, /* since v2.0 */
	KMIP_TAG_EPHEMERAL				= 0x420154, /* since v2.0 */
	KMIP_TAG_SERVER_HASHED_PASSWORD			= 0x420155, /* since v2.0 */
	KMIP_TAG_ONE_TIME_PASSWORD			= 0x420156, /* since v2.0 */
	KMIP_TAG_HASHED_PASSWORD			= 0x420157, /* since v2.0 */
	KMIP_TAG_ADJUSTMENT_TYPE			= 0x420158, /* since v2.0 */
	KMIP_TAG_PKCS_11_INTERFACE			= 0x420159, /* since v2.0 */
	KMIP_TAG_PKCS_11_FUNCTION			= 0x42015A, /* since v2.0 */
	KMIP_TAG_PKCS_11_INPUT_PARAMETERS		= 0x42015B, /* since v2.0 */
	KMIP_TAG_PKCS_11_OUTPUT_PARAMETERS		= 0x42015C, /* since v2.0 */
	KMIP_TAG_PKCS_11_RETURN_CODE			= 0x42015D, /* since v2.0 */
	KMIP_TAG_PROTECTION_STORAGE_MASK		= 0x42015E, /* since v2.0 */
	KMIP_TAG_PROTECTION_STORAGE_MASKS		= 0x42015F, /* since v2.0 */
	KMIP_TAG_INTEROP_FUNCTION			= 0x420160, /* since v2.0 */
	KMIP_TAG_INTEROP_IDENTIFIER			= 0x420161, /* since v2.0 */
	KMIP_TAG_ADJUSTMENT_VALUE			= 0x420162, /* since v2.0 */
	KMIP_TAG_COMMON_PROTECTION_STORAGE_MASKS	= 0x420163, /* since v2.0 */
	KMIP_TAG_PRIVATE_PROTECTION_STORAGE_MASKS	= 0x420164, /* since v2.0 */
	KMIP_TAG_PUBLIC_PROTECTION_STORAGE_MASKS	= 0x420165, /* since v2.0 */
	KMIP_TAG_OBJECT_GROUPS				= 0x420166, /* since v2.1 */
	KMIP_TAG_OBJECT_TYPES				= 0x420167, /* since v2.1 */
	KMIP_TAG_CONSTRAINTS				= 0x420168, /* since v2.1 */
	KMIP_TAG_CONSTRAINT				= 0x420169, /* since v2.1 */
	KMIP_TAG_ROTATE_INTERVAL			= 0x42016A, /* since v2.1 */
	KMIP_TAG_ROTATE_AUTOMATIC			= 0x42016B, /* since v2.1 */
	KMIP_TAG_ROTATE_OFFSET				= 0x42016C, /* since v2.1 */
	KMIP_TAG_ROTATE_DATE				= 0x42016D, /* since v2.1 */
	KMIP_TAG_ROTATE_GENERATION			= 0x42016E, /* since v2.1 */
	KMIP_TAG_ROTATE_NAME				= 0x42016F, /* since v2.1 */
	KMIP_TAG_ROTATE_NAME_VALUE			= 0x420170, /* since v2.1 */
	KMIP_TAG_ROTATE_NAME_TYPE			= 0x420171, /* since v2.1 */
	KMIP_TAG_ROTATE_LATEST				= 0x420172, /* since v2.1 */
	KMIP_TAG_ASYNCHRONOUS_REQUEST			= 0x420173, /* since v2.1 */
	KMIP_TAG_SUBMISSION_DATE			= 0x420174, /* since v2.1 */
	KMIP_TAG_PROCESSING_STAGE			= 0x420175, /* since v2.1 */
	KMIP_TAG_ASYNCHRONOUS_CORRELATION_VALUES	= 0x420176, /* since v2.1 */
};

enum kmip_type {
	KMIP_TYPE_STRUCTURE		= 0x01,
	KMIP_TYPE_INTEGER		= 0x02,
	KMIP_TYPE_LONG_INTEGER		= 0x03,
	KMIP_TYPE_BIG_INTEGER		= 0x04,
	KMIP_TYPE_ENUMERATION		= 0x05,
	KMIP_TYPE_BOOLEAN		= 0x06,
	KMIP_TYPE_TEXT_STRING		= 0x07,
	KMIP_TYPE_BYTE_STRING		= 0x08,
	KMIP_TYPE_DATE_TIME		= 0x09,
	KMIP_TYPE_INTERVAL		= 0x0A,
	KMIP_TYPE_DATE_TIME_EXTENDED	= 0x0B, /* Since v2.0 */
};

enum kmip_operation {
	KMIP_OPERATION_CREATE			= 0x01,
	KMIP_OPERATION_CREATE_KEY_PAIR		= 0x02,
	KMIP_OPERATION_REGISTER			= 0x03,
	KMIP_OPERATION_RE_KEY			= 0x04,
	KMIP_OPERATION_DERIVE_KEY		= 0x05,
	KMIP_OPERATION_CERTIFY			= 0x06,
	KMIP_OPERATION_RE_CERTIFY		= 0x07,
	KMIP_OPERATION_LOCATE			= 0x08,
	KMIP_OPERATION_CHECK			= 0x09,
	KMIP_OPERATION_GET			= 0x0A,
	KMIP_OPERATION_GET_ATTRIBUTES		= 0x0B,
	KMIP_OPERATION_GET_ATTRIBUTE_LIST	= 0x0C,
	KMIP_OPERATION_ADD_ATTRIBUTE		= 0x0D,
	KMIP_OPERATION_MODIFY_ATTRIBUTE		= 0x0E,
	KMIP_OPERATION_DELETE_ATTRIBUTE		= 0x0F,
	KMIP_OPERATION_OBTAIN_LEASE		= 0x10,
	KMIP_OPERATION_GET_USAGE_ALLOCATION	= 0x11,
	KMIP_OPERATION_ACTIVATE			= 0x12,
	KMIP_OPERATION_REVOKE			= 0x13,
	KMIP_OPERATION_DESTROY			= 0x14,
	KMIP_OPERATION_ARCHIVE			= 0x15,
	KMIP_OPERATION_RECOVER			= 0x16,
	KMIP_OPERATION_VALIDATE			= 0x17,
	KMIP_OPERATION_QUERY			= 0x18,
	KMIP_OPERATION_CANCEL			= 0x19,
	KMIP_OPERATION_POLL			= 0x1A,
	KMIP_OPERATION_NOTIFY			= 0x1B,
	KMIP_OPERATION_PUT			= 0x1C,
	KMIP_OPERATION_RE_KEY_KEY_PAIR		= 0x1D, /* since v1.2 */
	KMIP_OPERATION_DISCOVER_VERSIONS	= 0x1E, /* since v1.2 */
	KMIP_OPERATION_ENCRYPT			= 0x1F, /* since v1.2 */
	KMIP_OPERATION_DECRYPT			= 0x20, /* since v1.2 */
	KMIP_OPERATION_SIGN			= 0x21, /* since v1.2 */
	KMIP_OPERATION_SIGNATURE_VERIFY		= 0x22, /* since v1.2 */
	KMIP_OPERATION_MAC			= 0x23, /* since v1.2 */
	KMIP_OPERATION_MAC_VERIFY		= 0x24, /* since v1.2 */
	KMIP_OPERATION_RNG_RETRIEVE		= 0x25, /* since v1.2 */
	KMIP_OPERATION_RNG_SEED			= 0x26, /* since v1.2 */
	KMIP_OPERATION_HASH			= 0x27, /* since v1.2 */
	KMIP_OPERATION_CREATE_SPLIT_KEY		= 0x28, /* since v1.2 */
	KMIP_OPERATION_JOIN_SPLIT_KEY		= 0x29, /* since v1.2 */
	KMIP_OPERATION_IMPORT			= 0x2A, /* since v1.4 */
	KMIP_OPERATION_EXPORT			= 0x2B, /* since v1.4 */
	KMIP_OPERATION_LOG			= 0x2C, /* since v2.0 */
	KMIP_OPERATION_LOGIN			= 0x2D, /* since v2.0 */
	KMIP_OPERATION_LOGOUT			= 0x2E, /* since v2.0 */
	KMIP_OPERATION_DELEGATE_LOGIN		= 0x2F, /* since v2.0 */
	KMIP_OPERATION_ADJUST_ATTRIBUTE		= 0x30, /* since v2.0 */
	KMIP_OPERATION_SET_ATTRIBUTE		= 0x31, /* since v2.0 */
	KMIP_OPERATION_SET_ENDPOINT_ROLE	= 0x32, /* since v2.0 */
	KMIP_OPERATION_PKS_11			= 0x33, /* since v2.0 */
	KMIP_OPERATION_INTEROP			= 0x34, /* since v2.0 */
	KMIP_OPERATION_RE_PROVISION		= 0x35, /* since v2.0 */
	KMIP_OPERATION_SET_DEFAULTS		= 0x36, /* since v2.1 */
	KMIP_OPERATION_SET_CONSTRAINTS		= 0x37, /* since v2.1 */
	KMIP_OPERATION_GET_CONSTRAINTS		= 0x38, /* since v2.1 */
	KMIP_OPERATION_QUERY_ASYNCHRONOUS_REQUESTS = 0x39, /* since v2.1 */
	KMIP_OPERATION_PROCESS			= 0x3A, /* since v2.1 */
	KMIP_OPERATION_PING			= 0x3B, /* since v2.1 */
};

enum kmip_batch_error_cont_option {
	KMIP_BATCH_ERR_CONT_CONTINUE		= 0x01,
	KMIP_BATCH_ERR_CONT_STOP		= 0x02,
	KMIP_BATCH_ERR_CONT_UNDO		= 0x03,
};

enum kmip_crypto_usage_mask {
	KMIP_CRY_USAGE_MASK_SIGN		= 0x00000001,
	KMIP_CRY_USAGE_MASK_VERIFY		= 0x00000002,
	KMIP_CRY_USAGE_MASK_ENCRYPT		= 0x00000004,
	KMIP_CRY_USAGE_MASK_DECRYPT		= 0x00000008,
	KMIP_CRY_USAGE_MASK_WRAP_KEY		= 0x00000010,
	KMIP_CRY_USAGE_MASK_UNWRAP_KEY		= 0x00000020,
	KMIP_CRY_USAGE_MASK_EXPORT		= 0x00000040, /* v1.x only */
	KMIP_CRY_USAGE_MASK_MAC_GENERATE	= 0x00000080,
	KMIP_CRY_USAGE_MASK_MAC_VERIFY		= 0x00000100,
	KMIP_CRY_USAGE_MASK_DERIVE_KEY		= 0x00000200,
	KMIP_CRY_USAGE_MASK_CONTENT_COMMITMENT	= 0x00000400, /* v1.x only */
	KMIP_CRY_USAGE_MASK_KEY_AGREEMENT	= 0x00000800,
	KMIP_CRY_USAGE_MASK_CERTIFICATE_SIGN	= 0x00001000,
	KMIP_CRY_USAGE_MASK_CLR_SIGN		= 0x00002000,
	KMIP_CRY_USAGE_MASK_GENERATE_CRYPTOGRAM	= 0x00004000, /* v1.x only */
	KMIP_CRY_USAGE_MASK_VALIDATE_CRYPTOGRAM	= 0x00008000, /* v1.x only */
	KMIP_CRY_USAGE_MASK_TRANSLATE_ENCRYPT	= 0x00010000, /* v1.x only */
	KMIP_CRY_USAGE_MASK_TRANSLATE_DECRYPT	= 0x00020000, /* v1.x only */
	KMIP_CRY_USAGE_MASK_TRANSLATE_WRAP	= 0x00040000, /* v1.x only */
	KMIP_CRY_USAGE_MASK_TRANSLATE_UNWRAP	= 0x00080000, /* v1.x only */
	KMIP_CRY_USAGE_MASK_AUTHENTICATE	= 0x00100000, /* since v2.0 */
	KMIP_CRY_USAGE_MASK_UNRESTRICTED	= 0x00200000, /* since v2.0 */
	KMIP_CRY_USAGE_MASK_FPE_ENCRYPT		= 0x00400000, /* since v2.0 */
	KMIP_CRY_USAGE_MASK_FPE_DECRYPT		= 0x00800000, /* since v2.0 */
};

enum kmip_result_status {
	KMIP_RESULT_STATUS_SUCCESS			= 0x00,
	KMIP_RESULT_STATUS_OPERATION_FAILED		= 0x01,
	KMIP_RESULT_STATUS_OPERATION_PENDING		= 0x02,
	KMIP_RESULT_STATUS_OPERATION_UNDONE		= 0x03,
};

enum kmip_result_reason {
	KMIP_RESULT_REASON_ITEM_NOT_FOUND		= 0x01,
	KMIP_RESULT_REASON_RESPONSE_TOO_LARGE		= 0x02,
	KMIP_RESULT_REASON_AUTH_NOT_SUCCESSFUL		= 0x03,
	KMIP_RESULT_REASON_INVALID_MESSAGE		= 0x04,
	KMIP_RESULT_REASON_OPERATION_NOT_SUCCESSFUL	= 0x05,
	KMIP_RESULT_REASON_MISSING_DATA			= 0x06,
	KMIP_RESULT_REASON_INVALIUD_FIELD		= 0x07,
	KMIP_RESULT_REASON_FEATURE_NOT_SUPPORTED	= 0x08,
	KMIP_RESULT_REASON_OP_CANCELED_BY_REQUESTOR	= 0x09,
	KMIP_RESULT_REASON_CRYPTOGRAPHIC_FAILURE	= 0x0A,
	KMIP_RESULT_REASON_ILLEGAL_OPERATION		= 0x0B, /* v 1.x only */
	KMIP_RESULT_REASON_PERMISSION_DENIED		= 0x0C,
	KMIP_RESULT_REASON_OBJECT_ARCHIVED		= 0x0D,
	KMIP_RESULT_REASON_INDEX_OUT_OF_BOUNDS		= 0x0E, /* v 1.x only */
	KMIP_RESULT_REASON_APP_NAMESPACE_NOT_SUPPORTED	= 0x0F,
	KMIP_RESULT_REASON_KEY_FORMAT_TYPE_NOT_SUPPORTED = 0x10,
	KMIP_RESULT_REASON_KEY_COMPRESSION_TYPE_NOT_SUPPORTED = 0x11,
	KMIP_RESULT_REASON_ENCODING_OPTION_ERROR	= 0x12, /* since v1.2 */
	KMIP_RESULT_REASON_KEY_VALUE_NOT_PRESENT	= 0x13, /* since v1.2 */
	KMIP_RESULT_REASON_ATTESTATION_REQUIRED		= 0x14, /* since v1.2 */
	KMIP_RESULT_REASON_ATTESTATION_FAILED		= 0x15, /* since v1.2 */
	KMIP_RESULT_REASON_SENSITIVE			= 0x16, /* since v1.4 */
	KMIP_RESULT_REASON_NOT_EXTRACTABLE		= 0x17, /* since v1.4 */
	KMIP_RESULT_REASON_OBJECT_ALREADY_EXISTS	= 0x18, /* since v1.4 */
	KMIP_RESULT_REASON_INVALID_TICKET		= 0x19, /* since v2.0 */
	KMIP_RESULT_REASON_USAGE_LIMIT_EXCEEDED		= 0x1A, /* since v2.0 */
	KMIP_RESULT_REASON_NUMERIC_RANGE		= 0x1B, /* since v2.0 */
	KMIP_RESULT_REASON_INVALID_DATA_TYPE		= 0x1C, /* since v2.0 */
	KMIP_RESULT_REASON_READ_ONLY_ATTRIBUTE		= 0x1D, /* since v2.0 */
	KMIP_RESULT_REASON_MULTI_VALUED_ATTRIBUTE	= 0x1E, /* since v2.0 */
	KMIP_RESULT_REASON_UNSUPPORTED_ATTRIBUTE	= 0x1F, /* since v2.0 */
	KMIP_RESULT_REASON_ATTRIBUTE_INSTANCE_NOT_FOUND	= 0x20, /* since v2.0 */
	KMIP_RESULT_REASON_ATTRIBUTE_NOT_FOUND		= 0x21, /* since v2.0 */
	KMIP_RESULT_REASON_ATTRIBUTE_READ_ONLY		= 0x22, /* since v2.0 */
	KMIP_RESULT_REASON_ATTRIBUTE_SINGLE_VALUED	= 0x23, /* since v2.0 */
	KMIP_RESULT_REASON_BAD_CRYPTOGRAPHIC_PARAMETERS	= 0x24, /* since v2.0 */
	KMIP_RESULT_REASON_BAD_PASSWORD			= 0x25, /* since v2.0 */
	KMIP_RESULT_REASON_CODEC_ERROR			= 0x26, /* since v2.0 */
	KMIP_RESULT_REASON_ILLEGAL_OBJECT_TYPE		= 0x28, /* since v2.0 */
	KMIP_RESULT_REASON_INCOMPATIBLE_CRYPTO_USAGE_MASK = 0x29, /* since v2.0 */
	KMIP_RESULT_REASON_INTERNAL_SERVER_ERROR	= 0x2A, /* since v2.0 */
	KMIP_RESULT_REASON_INVALID_ASYNC_CORRELATION_VALUE = 0x2B, /* since v2.0 */
	KMIP_RESULT_REASON_INVALID_ATTRIBUTE		= 0x2C, /* since v2.0 */
	KMIP_RESULT_REASON_INVALID_ATTRIBUTE_VALUE	= 0x2D, /* since v2.0 */
	KMIP_RESULT_REASON_INVALID_CORRELATION_VALUE	= 0x2E, /* since v2.0 */
	KMIP_RESULT_REASON_INVALID_CSR			= 0x2F, /* since v2.0 */
	KMIP_RESULT_REASON_INVALID_OBJECT_TYPE		= 0x30, /* since v2.0 */
	KMIP_RESULT_REASON_KEY_WRAP_TYPE_NOT_SUPPORTED	= 0x32, /* since v2.0 */
	KMIP_RESULT_REASON_MISSING_INITIALIZATION_VECTOR = 0x34, /* since v2.0 */
	KMIP_RESULT_REASON_NOT_UNIQUE_NAME_ATTRIBUTE	= 0x35, /* since v2.0 */
	KMIP_RESULT_REASON_OBJECT_DESTROYED		= 0x36, /* since v2.0 */
	KMIP_RESULT_REASON_OBJECT_NOT_FOUND		= 0x37, /* since v2.0 */
	KMIP_RESULT_REASON_NOT_AUTHORISED		= 0x39, /* since v2.0 */
	KMIP_RESULT_REASON_SERVER_LIMIT_EXCEEDED	= 0x3A, /* since v2.0 */
	KMIP_RESULT_REASON_UNKNOWN_ENUMERATION		= 0x3B, /* since v2.0 */
	KMIP_RESULT_REASON_UNKNOWN_MESSAGE_EXTENSION	= 0x3C, /* since v2.0 */
	KMIP_RESULT_REASON_UNKNOWN_TAG			= 0x3D, /* since v2.0 */
	KMIP_RESULT_REASON_UNSUPPORTED_CRYPTO_PARAMETERS = 0x3E, /* since v2.0 */
	KMIP_RESULT_REASON_UNSUPPORTED_PROTOCOL_VERSION	= 0x3F, /* since v2.0 */
	KMIP_RESULT_REASON_WRAPPING_OBJECT_ARCHIVED	= 0x40, /* since v2.0 */
	KMIP_RESULT_REASON_WRAPPING_OBJECT_DESTROYED	= 0x41, /* since v2.0 */
	KMIP_RESULT_REASON_WRAPPING_OBJECT_NOT_FOUND	= 0x42, /* since v2.0 */
	KMIP_RESULT_REASON_WRONG_KEY_LIFECYCLE_STATE	= 0x43, /* since v2.0 */
	KMIP_RESULT_REASON_PROTECTION_STORAGE_UNAVAILABLE = 0x44, /* since v2.0 */
	KMIP_RESULT_REASON_PKCS_11_CODE_ERROR		= 0x45, /* since v2.0 */
	KMIP_RESULT_REASON_PKCS_11_INVALID_FUNCTION	= 0x46, /* since v2.0 */
	KMIP_RESULT_REASON_PKCS_11_INVALID_INTERFACE	= 0x47, /* since v2.0 */
	KMIP_RESULT_REASON_PRIVATE_PROT_STORAGE_UNAVAILABLE = 0x48, /* since v2.0 */
	KMIP_RESULT_REASON_PUBLIC_PROT_STORAGE_UNAVAILABLE = 0x49, /* since v2.0 */
	KMIP_RESULT_REASON_UNKNOWN_OBJECT_GROUP		= 0x4A, /* since v2.1 */
	KMIP_RESULT_REASON_CONSTRAINT_VIOLATION		= 0x4B, /* since v2.1 */
	KMIP_RESULT_REASON_DUPLICATE_PROCESS_REQUEST	= 0x4C, /* since v2.1 */
	KMIP_RESULT_REASON_GENERAL_FAILURE		= 0x100,
};

enum kmip_query_function {
	KMIP_QUERY_OPERATIONS			= 0x01,
	KMIP_QUERY_OBJECTS			= 0x02,
	KMIP_QUERY_SERVER_INFORMATION		= 0x03,
	KMIP_QUERY_APPLICATION_NAMESPACES	= 0x04, /* since v1.2 */
	KMIP_QUERY_EXTENSION_LIST		= 0x05, /* since v1.2 */
	KMIP_QUERY_EXTENSION_MAP		= 0x06, /* since v1.2 */
	KMIP_QUERY_ATTESTATION_TYPES		= 0x07, /* since v1.2 */
	KMIP_QUERY_QUERY_RNGS			= 0x08, /* since v1.3 */
	KMIP_QUERY_VALIDATIONS			= 0x09, /* since v1.3 */
	KMIP_QUERY_PROFILES			= 0x0A, /* since v1.3 */
	KMIP_QUERY_CAPABILITIES			= 0x0B, /* since v1.3 */
	KMIP_QUERY_CLIENT_REGISTRATION_METHODS	= 0x0C, /* since v1.3 */
	KMIP_QUERY_DEFAULTS_INFORMATION		= 0x0D, /* since v2.0 */
	KMIP_QUERY_STORAGE_PROTECTION_MASKS	= 0x0E, /* since v2.0 */
};

enum kmip_name_type {
	KMIP_NAME_TYPE_UNINTERPRETED_TEXT_STRING	= 0x01,
	KMIP_NAME_TYPE_URI				= 0x02,
};

enum kmip_alternative_name_type {
	KMIP_ALT_NAME_TYPE_UNINTERPRETED_TEXT_STRING	= 0x01,
	KMIP_ALT_NAME_TYPE_URI				= 0x02,
	KMIP_ALT_NAME_TYPE_OBJECT_SERIAL_NUMBER		= 0x03,
	KMIP_ALT_NAME_TYPE_EMAIL_ADDRESS		= 0x04,
	KMIP_ALT_NAME_TYPE_DNS_NAME			= 0x05,
	KMIP_ALT_NAME_TYPE_X_500_DISTINGUISHED_NAME	= 0x06,
	KMIP_ALT_NAME_TYPE_IP_ADDRESS			= 0x07,
};

enum kmip_unique_identifier {
	KMIP_UNIQUE_ID_ID_PLACEHOLDER		= 0x01, /* since v2.0 */
	KMIP_UNIQUE_ID_CERTIFY			= 0x02, /* since v2.0 */
	KMIP_UNIQUE_ID_CREATE			= 0x03, /* since v2.0 */
	KMIP_UNIQUE_ID_CREATE_KEY_PAIR		= 0x04, /* since v2.0 */
	KMIP_UNIQUE_ID_CREATE_KEY_PAIR_PRIVATE	= 0x05, /* since v2.0 */
	KMIP_UNIQUE_ID_CREATE_KEY_PAIR_PUBLIC	= 0x06, /* since v2.0 */
	KMIP_UNIQUE_ID_CREATE_SPLIT_KEY		= 0x07, /* since v2.0 */
	KMIP_UNIQUE_ID_DERIVE_KEY		= 0x08, /* since v2.0 */
	KMIP_UNIQUE_ID_IMPORT			= 0x09, /* since v2.0 */
	KMIP_UNIQUE_ID_JOIN_SPLIT_KEY		= 0x0A, /* since v2.0 */
	KMIP_UNIQUE_ID_LOCATE			= 0x0B, /* since v2.0 */
	KMIP_UNIQUE_ID_REGISTER			= 0x0C, /* since v2.0 */
	KMIP_UNIQUE_ID_RE_KEY			= 0x0D, /* since v2.0 */
	KMIP_UNIQUE_ID_RE_CERTIFY		= 0x0E, /* since v2.0 */
	KMIP_UNIQUE_ID_RE_KEY_KEY_PAIR		= 0x0F, /* since v2.0 */
	KMIP_UNIQUE_ID_RE_KEY_KEY_PAIR_PRIVATE	= 0x10, /* since v2.0 */
	KMIP_UNIQUE_ID_RE_KEY_KEY_PAIR_PUBLIC	= 0x11, /* since v2.0 */
};

enum kmip_object_type {
	KMIP_OBJECT_TYPE_CERTIFICATE		= 0x01,
	KMIP_OBJECT_TYPE_SYMMETRIC_KEY		= 0x02,
	KMIP_OBJECT_TYPE_PUBLIC_KEY		= 0x03,
	KMIP_OBJECT_TYPE_PRIVATE_KEY		= 0x04,
	KMIP_OBJECT_TYPE_SPLIT_KEY		= 0x05,
	KMIP_OBJECT_TYPE_TEMPLATE		= 0x06, /* v1.x only */
	KMIP_OBJECT_TYPE_SECRET_DATA		= 0x07,
	KMIP_OBJECT_TYPE_OPAQUE_OBJECT		= 0x08,
	KMIP_OBJECT_TYPE_PGP_KEY		= 0x09, /* since v1.2 */
	KMIP_OBJECT_TYPE_CERTIFICATE_REQUEST	= 0x0A, /* since v2.0 */
};

enum kmip_crypto_algo {
	KMIP_CRYPTO_ALGO_DES			= 0x01,
	KMIP_CRYPTO_ALGO_3DES			= 0x02,
	KMIP_CRYPTO_ALGO_AES			= 0x03,
	KMIP_CRYPTO_ALGO_RSA			= 0x04,
	KMIP_CRYPTO_ALGO_DSA			= 0x05,
	KMIP_CRYPTO_ALGO_ECDSA			= 0x06,
	KMIP_CRYPTO_ALGO_HMAC_SHA1		= 0x07,
	KMIP_CRYPTO_ALGO_HMAC_SHA224		= 0x08,
	KMIP_CRYPTO_ALGO_HMAC_SHA256		= 0x09,
	KMIP_CRYPTO_ALGO_HMAC_SHA384		= 0x0A,
	KMIP_CRYPTO_ALGO_HMAC_SHA512		= 0x0B,
	KMIP_CRYPTO_ALGO_HMAC_MD5		= 0x0C,
	KMIP_CRYPTO_ALGO_DH			= 0x0D,
	KMIP_CRYPTO_ALGO_ECDH			= 0x0E,
	KMIP_CRYPTO_ALGO_ECMQV			= 0x0F,
	KMIP_CRYPTO_ALGO_BLOWFISH		= 0x10,
	KMIP_CRYPTO_ALGO_CAMELLIA		= 0x11,
	KMIP_CRYPTO_ALGO_CAST5			= 0x12,
	KMIP_CRYPTO_ALGO_IDEA			= 0x13,
	KMIP_CRYPTO_ALGO_MARS			= 0x14,
	KMIP_CRYPTO_ALGO_RC2			= 0x15,
	KMIP_CRYPTO_ALGO_RC4			= 0x16,
	KMIP_CRYPTO_ALGO_RC5			= 0x17,
	KMIP_CRYPTO_ALGO_SKIPJACK		= 0x18,
	KMIP_CRYPTO_ALGO_TWOFISH		= 0x19,
	KMIP_CRYPTO_ALGO_EC			= 0x1A, /* since v1.2 */
	KMIP_CRYPTO_ALGO_ONE_TIME_PAD		= 0x1B, /* since v1.3 */
	KMIP_CRYPTO_ALGO_CHACHA20		= 0x1C, /* since v1.4 */
	KMIP_CRYPTO_ALGO_POLY1305		= 0x1D, /* since v1.4 */
	KMIP_CRYPTO_ALGO_CHACHA20_POLY1305	= 0x1E, /* since v1.4 */
	KMIP_CRYPTO_ALGO_SHA3_224		= 0x1F, /* since v1.4 */
	KMIP_CRYPTO_ALGO_SHA3_256		= 0x20, /* since v1.4 */
	KMIP_CRYPTO_ALGO_SHA3_384		= 0x21, /* since v1.4 */
	KMIP_CRYPTO_ALGO_SHA3_512		= 0x22, /* since v1.4 */
	KMIP_CRYPTO_ALGO_HMAC_SHA3_224		= 0x23, /* since v1.4 */
	KMIP_CRYPTO_ALGO_HMAC_SHA3_256		= 0x24, /* since v1.4 */
	KMIP_CRYPTO_ALGO_HMAC_SHA3_384		= 0x25, /* since v1.4 */
	KMIP_CRYPTO_ALGO_HMAC_SHA3_512		= 0x26, /* since v1.4 */
	KMIP_CRYPTO_ALGO_SHAKE_128		= 0x27, /* since v1.4 */
	KMIP_CRYPTO_ALGO_SHAKE_256		= 0x28, /* since v1.4 */
	KMIP_CRYPTO_ALGO_ARIA			= 0x29, /* since v2.0 */
	KMIP_CRYPTO_ALGO_SEED			= 0x2A, /* since v2.0 */
	KMIP_CRYPTO_ALGO_SM2			= 0x2B, /* since v2.0 */
	KMIP_CRYPTO_ALGO_SM3			= 0x2C, /* since v2.0 */
	KMIP_CRYPTO_ALGO_SM4			= 0x2D, /* since v2.0 */
	KMIP_CRYPTO_ALGO_GOST_R34_10_2012	= 0x2E, /* since v2.0 */
	KMIP_CRYPTO_ALGO_GOST_R34_11_2012	= 0x2F, /* since v2.0 */
	KMIP_CRYPTO_ALGO_GOST_R34_13_2015	= 0x30, /* since v2.0 */
	KMIP_CRYPTO_ALGO_GOST_28147_89		= 0x31, /* since v2.0 */
	KMIP_CRYPTO_ALGO_XMSS			= 0x32, /* since v2.0 */
	KMIP_CRYPTO_ALGO_SPHINCS_256		= 0x33, /* since v2.0 */
	KMIP_CRYPTO_ALGO_MCELIECE		= 0x34, /* since v2.0 */
	KMIP_CRYPTO_ALGO_MCELIECE_6960119	= 0x35, /* since v2.0 */
	KMIP_CRYPTO_ALGO_MCELIECE_8192128	= 0x36, /* since v2.0 */
	KMIP_CRYPTO_ALGO_ED25519		= 0x37, /* since v2.0 */
	KMIP_CRYPTO_ALGO_ED448			= 0x38, /* since v2.0 */
};

enum kmip_certificate_type {
	KMIP_CERTIFICATE_TYPE_X_509		= 0x01,
	KMIP_CERTIFICATE_TYPE_PGP		= 0x02,
};

enum kmip_state {
	KMIP_STATE_PRE_ACTIVE			= 0x01,
	KMIP_STATE_ACTIVE			= 0x02,
	KMIP_STATE_DEACTIVATED			= 0x03,
	KMIP_STATE_COMPROMISED			= 0x04,
	KMIP_STATE_DESTROYED			= 0x05,
	KMIP_STATE_DESTROYED_COMPROMISED	= 0x06,
};

enum kmip_protection_storage_mask {
	KMIP_PROT_STORAGE_MASK_SOFTWARE			= 0x00000001,
	KMIP_PROT_STORAGE_MASK_HARDWARE			= 0x00000002,
	KMIP_PROT_STORAGE_MASK_ON_PROCESSOR		= 0x00000004,
	KMIP_PROT_STORAGE_MASK_ON_SYSTEM		= 0x00000008,
	KMIP_PROT_STORAGE_MASK_OFF_SYSTEM		= 0x00000010,
	KMIP_PROT_STORAGE_MASK_HYPERVISOR		= 0x00000020,
	KMIP_PROT_STORAGE_MASK_OPERATING_SYSTEM		= 0x00000040,
	KMIP_PROT_STORAGE_MASK_CONTAINER		= 0x00000080,
	KMIP_PROT_STORAGE_MASK_ON_PREMISES		= 0x00000100,
	KMIP_PROT_STORAGE_MASK_OFF_PREMISES		= 0x00000200,
	KMIP_PROT_STORAGE_MASK_SELF_MANAGED		= 0x00000400,
	KMIP_PROT_STORAGE_MASK_OUTSOURCED		= 0x00000800,
	KMIP_PROT_STORAGE_MASK_VALIDATED		= 0x00001000,
	KMIP_PROT_STORAGE_MASK_SAME_JURISDICATION	= 0x00002000,
};

enum kmip_revoke_reason {
	KMIP_REVOK_RSN_UNSPECIFIED		= 0x01,
	KMIP_REVOK_RSN_KEY_COMPROMISE		= 0x02,
	KMIP_REVOK_RSN_CA_COMPROMISE		= 0x03,
	KMIP_REVOK_RSN_AFFILIATION_CHANGED	= 0x04,
	KMIP_REVOK_RSN_SUPERSEDED		= 0x05,
	KMIP_REVOK_RSN_CESSATION_OF_OPERATION	= 0x06,
	KMIP_REVOK_RSN_PRIVILEGE_WITHDRAWN	= 0x07,
};

enum kmip_object_group_member {
	KMIP_OBJ_GROUP_MEMBER_FRESH		= 0x01,
	KMIP_OBJ_GROUP_MEMBER_DEFAULT		= 0x02,
};

enum kmip_storage_status_mask {
	KMIP_STORAGE_STATUS_MASK_ONLINE		= 0x01,
	KMIP_STORAGE_STATUS_MASK_ARCHIVAL	= 0x02,
	KMIP_STORAGE_STATUS_MASK_DESTTROYED	= 0x04,
};

enum kmip_key_format_type {
	KMIP_KEY_FORMAT_TYPE_RAW			= 0x01,
	KMIP_KEY_FORMAT_TYPE_OPAQUE			= 0x02,
	KMIP_KEY_FORMAT_TYPE_PKCS_1			= 0x03,
	KMIP_KEY_FORMAT_TYPE_PKCS_8			= 0x04,
	KMIP_KEY_FORMAT_TYPE_X_509			= 0x05,
	KMIP_KEY_FORMAT_TYPE_EC_PRIVATE_KEY		= 0x06,
	KMIP_KEY_FORMAT_TYPE_TRANSPARENT_SYMMETRIC_KEY	= 0x07,
	KMIP_KEY_FORMAT_TYPE_TRANSPARENT_DSA_PRIVATE_KEY = 0x08,
	KMIP_KEY_FORMAT_TYPE_TRANSPARENT_DSA_PUBLIC_KEY = 0x09,
	KMIP_KEY_FORMAT_TYPE_TRANSPARENT_RSA_PRIVATE_KEY = 0x0A,
	KMIP_KEY_FORMAT_TYPE_TRANSPARENT_RSA_PUBLIC_KEY	= 0x0B,
	KMIP_KEY_FORMAT_TYPE_TRANSPARENT_DH_PRIVATE_KEY	= 0x0C,
	KMIP_KEY_FORMAT_TYPE_TRANSPARENT_DH_PUBLIC_KEY	= 0x0D,
	KMIP_KEY_FORMAT_TYPE_TRANSPARENT_ECDSA_PRIVATE_KEY = 0x0E, /* deprecated since v1.3 */
	KMIP_KEY_FORMAT_TYPE_TRANSPARENT_ECDSA_PUBLIC_KEY = 0x0F, /* deprecated since v1.3 */
	KMIP_KEY_FORMAT_TYPE_TRANSPARENT_ECDH_PRIVATE_KEY = 0x10, /* deprecated since v1.3 */
	KMIP_KEY_FORMAT_TYPE_TRANSPARENT_ECDH_PUBLIC_KEY = 0x11, /* deprecated since v1.3 */
	KMIP_KEY_FORMAT_TYPE_TRANSPARENT_ECMQV_PRIVATE_KEY = 0x12, /* deprecated since v1.3 */
	KMIP_KEY_FORMAT_TYPE_TRANSPARENT_ECMQV_PUBLIC_KEY = 0x13, /* deprecated since v1.3 */
	KMIP_KEY_FORMAT_TYPE_TRANSPARENT_EC_PRIVATE_KEY	= 0x14, /* since v1.3 */
	KMIP_KEY_FORMAT_TYPE_TRANSPARENT_EC_PUBLIC_KEY	= 0x15, /* since v1.3 */
	KMIP_KEY_FORMAT_TYPE_PKCS_12			= 0x16, /* since v1.4 */
	KMIP_KEY_FORMAT_TYPE_PKCS_10			= 0x17, /* since v2.0 */
};

enum kmip_key_compression_type {
	KMIP_KEY_COMPRESSION_TYPE_EC_PUBKEY_UNCOMPRESSED	= 0x01,
	KMIP_KEY_COMPRESSION_TYPE_EC_PUBKEY_COMPRESSED_PRIME	= 0x02,
	KMIP_KEY_COMPRESSION_TYPE_EC_PUBKEY_COMPRESSED_CHAR2	= 0x03,
	KMIP_KEY_COMPRESSION_TYPE_EC_PUBKEY_HYBID		= 0x04,
};

enum kmip_wrapping_method {
	KMIP_WRAPPING_METHOD_ENCRYPT			= 0x01,
	KMIP_WRAPPING_METHOD_MAC_SIGN			= 0x02,
	KMIP_WRAPPING_METHOD_ENCRYPT_THEN_MAC_SIGN	= 0x03,
	KMIP_WRAPPING_METHOD_MAC_SIGN_THEN_ENCRYPT	= 0x04,
	KMIP_WRAPPING_METHOD_TR_31			= 0x05,
};

enum kmip_key_wrap_type {
	KMIP_KEY_WRAP_TYPE_NOT_WRAPPED			= 0x01,
	KMIP_KEY_WRAP_TYPE_AS_REGISTERED		= 0x02,
};

enum kmip_block_cipher_mode {
	KMIP_BLOCK_CIPHER_MODE_CBC			= 0x01,
	KMIP_BLOCK_CIPHER_MODE_ECB			= 0x02,
	KMIP_BLOCK_CIPHER_MODE_PCBC			= 0x03,
	KMIP_BLOCK_CIPHER_MODE_CFB			= 0x04,
	KMIP_BLOCK_CIPHER_MODE_OFB			= 0x05,
	KMIP_BLOCK_CIPHER_MODE_CTR			= 0x06,
	KMIP_BLOCK_CIPHER_MODE_CMAC			= 0x07,
	KMIP_BLOCK_CIPHER_MODE_CCM			= 0x08,
	KMIP_BLOCK_CIPHER_MODE_GCM			= 0x09,
	KMIP_BLOCK_CIPHER_MODE_CBC_MAC			= 0x0A,
	KMIP_BLOCK_CIPHER_MODE_XTS			= 0x0B,
	KMIP_BLOCK_CIPHER_MODE_AES_KEY_WRAP_PADDING	= 0x0C,
	KMIP_BLOCK_CIPHER_MODE_NIST_KEY_WRAP		= 0x0D,
	KMIP_BLOCK_CIPHER_MODE_X9_102_AESKW		= 0x0E,
	KMIP_BLOCK_CIPHER_MODE_X9_102_TDKW		= 0x0F,
	KMIP_BLOCK_CIPHER_MODE_X9_102_AKW1		= 0x10,
	KMIP_BLOCK_CIPHER_MODE_X9_102_AKW2		= 0x11,
	KMIP_BLOCK_CIPHER_MODE_AEAD			= 0x12, /* since v1.4 */
};

enum kmip_padding_method {
	KMIP_PADDING_METHOD_NONE			= 0x01,
	KMIP_PADDING_METHOD_OAEP			= 0x02,
	KMIP_PADDING_METHOD_PKCS5			= 0x03,
	KMIP_PADDING_METHOD_SSL3			= 0x04,
	KMIP_PADDING_METHOD_ZEROS			= 0x05,
	KMIP_PADDING_METHOD_ANSI_X9_23			= 0x06,
	KMIP_PADDING_METHOD_ISO_10126			= 0x07,
	KMIP_PADDING_METHOD_PKCS_1_5			= 0x08,
	KMIP_PADDING_METHOD_X9_31			= 0x09,
	KMIP_PADDING_METHOD_PSS				= 0x0A,
};

enum kmip_hashing_algo {
	KMIP_HASHING_ALGO_MD2				= 0x01,
	KMIP_HASHING_ALGO_MD4				= 0x02,
	KMIP_HASHING_ALGO_MD5				= 0x03,
	KMIP_HASHING_ALGO_SHA_1				= 0x04,
	KMIP_HASHING_ALGO_SHA_224			= 0x05,
	KMIP_HASHING_ALGO_SHA_256			= 0x06,
	KMIP_HASHING_ALGO_SHA_384			= 0x07,
	KMIP_HASHING_ALGO_SHA_512			= 0x08,
	KMIP_HASHING_ALGO_RIPEMD_160			= 0x09,
	KMIP_HASHING_ALGO_TIGER				= 0x0A,
	KMIP_HASHING_ALGO_WIRLPOOL			= 0x0B,
	KMIP_HASHING_ALGO_SHA_512_224			= 0x0C, /* since v1.2 */
	KMIP_HASHING_ALGO_SHA_512_256			= 0x0D, /* since v1.2 */
	KMIP_HASHING_ALGO_SHA_3_224			= 0x0E, /* since v1.4 */
	KMIP_HASHING_ALGO_SHA_3_256			= 0x0F, /* since v1.4 */
	KMIP_HASHING_ALGO_SHA_3_384			= 0x10, /* since v1.4 */
	KMIP_HASHING_ALGO_SHA_3_512			= 0x11, /* since v1.4 */
};

enum kmip_key_role_type {
	KMIP_KEY_ROLE_TYPE_BDK				= 0x01,
	KMIP_KEY_ROLE_TYPE_CVK				= 0x02,
	KMIP_KEY_ROLE_TYPE_DEK				= 0x03,
	KMIP_KEY_ROLE_TYPE_KMAC				= 0x04,
	KMIP_KEY_ROLE_TYPE_MKSMC			= 0x05,
	KMIP_KEY_ROLE_TYPE_MKSMI			= 0x06,
	KMIP_KEY_ROLE_TYPE_MKDAC			= 0x07,
	KMIP_KEY_ROLE_TYPE_MKDN				= 0x08,
	KMIP_KEY_ROLE_TYPE_MKCP				= 0x09,
	KMIP_KEY_ROLE_TYPE_MKOTH			= 0x0A,
	KMIP_KEY_ROLE_TYPE_KEK				= 0x0B,
	KMIP_KEY_ROLE_TYPE_MAC16609			= 0x0C,
	KMIP_KEY_ROLE_TYPE_MAC97971			= 0x0D,
	KMIP_KEY_ROLE_TYPE_MAC97972			= 0x0E,
	KMIP_KEY_ROLE_TYPE_MAC97973			= 0x0F,
	KMIP_KEY_ROLE_TYPE_MAC97974			= 0x10,
	KMIP_KEY_ROLE_TYPE_MAC97975			= 0x11,
	KMIP_KEY_ROLE_TYPE_ZPK				= 0x12,
	KMIP_KEY_ROLE_TYPE_PVKIBM			= 0x13,
	KMIP_KEY_ROLE_TYPE_PVKPVV			= 0x14,
	KMIP_KEY_ROLE_TYPE_PVKOTH			= 0x15,
	KMIP_KEY_ROLE_TYPE_DUKPT			= 0x16, /* since v1.4 */
	KMIP_KEY_ROLE_TYPE_IV				= 0x17, /* since v1.4 */
	KMIP_KEY_ROLE_TYPE_TRKBK			= 0x18, /* since v1.4 */
};

enum kmip_signature_algo {
	KMIP_SIGNATURE_ALGO_MD2_WITH_RSA_ENCRYPTION	= 0x01, /* since v1.2 */
	KMIP_SIGNATURE_ALGO_MD5_WITH_RSA_ENCRYPTION	= 0x02, /* since v1.2 */
	KMIP_SIGNATURE_ALGO_SHA_1_WITH_RSA_ENCRYPTION	= 0x03, /* since v1.2 */
	KMIP_SIGNATURE_ALGO_SHA_224_WITH_RSA_ENCRYPTION	= 0x04, /* since v1.2 */
	KMIP_SIGNATURE_ALGO_SHA_256_WITH_RSA_ENCRYPTION	= 0x05, /* since v1.2 */
	KMIP_SIGNATURE_ALGO_SHA_384_WITH_RSA_ENCRYPTION	= 0x06, /* since v1.2 */
	KMIP_SIGNATURE_ALGO_SHA_512_WITH_RSA_ENCRYPTION	= 0x07, /* since v1.2 */
	KMIP_SIGNATURE_ALGO_RSASSA_PSS			= 0x08, /* since v1.2 */
	KMIP_SIGNATURE_ALGO_DSA_WITH_SHA_1		= 0x09, /* since v1.2 */
	KMIP_SIGNATURE_ALGO_DSA_WITH_SHA_244		= 0x0A, /* since v1.2 */
	KMIP_SIGNATURE_ALGO_DSA_WITH_SHA_256		= 0x0B, /* since v1.2 */
	KMIP_SIGNATURE_ALGO_ECDSA_WITH_SHA_1		= 0x0C, /* since v1.2 */
	KMIP_SIGNATURE_ALGO_ECDSA_WITH_SHA_224		= 0x0D, /* since v1.2 */
	KMIP_SIGNATURE_ALGO_ECDSA_WITH_SHA_256		= 0x0E, /* since v1.2 */
	KMIP_SIGNATURE_ALGO_ECDSA_WITH_SHA_384		= 0x0F, /* since v1.2 */
	KMIP_SIGNATURE_ALGO_ECDSA_WITH_SHA_512		= 0x10, /* since v1.2 */
	KMIP_SIGNATURE_ALGO_SHA3_256_WITH_RSA_ENCRYPTION = 0x11, /* since v1.4 */
	KMIP_SIGNATURE_ALGO_SHA3_385_WITH_RSA_ENCRYPTION = 0x12, /* since v1.4 */
	KMIP_SIGNATURE_ALGO_SHA3_512_WITH_RSA_ENCRYPTION = 0x13, /* since v1.4 */
};

enum kmip_mask_generator {
	KMIP_MASK_GENERATOR_MGF1	= 0x01, /* since v1.4 */
};

enum kmip_encoding_option {
	KMIP_ENCODING_OPTION_NO		= 0x01, /* since v1.2 */
	KMIP_ENCODING_OPTION_TTLV	= 0x02, /* since v1.2 */
};

enum kmip_recommended_curve {
	KMIP_REC_CURVE_P_192		= 0x01,
	KMIP_REC_CURVE_K_163		= 0x02,
	KMIP_REC_CURVE_B_163		= 0x03,
	KMIP_REC_CURVE_P_224		= 0x04,
	KMIP_REC_CURVE_K_223		= 0x05,
	KMIP_REC_CURVE_B_223		= 0x06,
	KMIP_REC_CURVE_P_256		= 0x07,
	KMIP_REC_CURVE_K_283		= 0x08,
	KMIP_REC_CURVE_B_283		= 0x09,
	KMIP_REC_CURVE_P_384		= 0x0A,
	KMIP_REC_CURVE_K_409		= 0x0B,
	KMIP_REC_CURVE_B_409		= 0x0C,
	KMIP_REC_CURVE_P_521		= 0x0D,
	KMIP_REC_CURVE_K_571		= 0x0E,
	KMIP_REC_CURVE_B_571		= 0x0F,
	KMIP_REC_CURVE_SECP112R1	= 0x10, /* since v1.2 */
	KMIP_REC_CURVE_SECP112R2	= 0x11, /* since v1.2 */
	KMIP_REC_CURVE_SECP128R1	= 0x12, /* since v1.2 */
	KMIP_REC_CURVE_SECP128R2	= 0x13, /* since v1.2 */
	KMIP_REC_CURVE_SECP160K1	= 0x14, /* since v1.2 */
	KMIP_REC_CURVE_SECP160R1	= 0x15, /* since v1.2 */
	KMIP_REC_CURVE_SECP160R2	= 0x16, /* since v1.2 */
	KMIP_REC_CURVE_SECP192K1	= 0x17, /* since v1.2 */
	KMIP_REC_CURVE_SECP224K1	= 0x18, /* since v1.2 */
	KMIP_REC_CURVE_SECP256K1	= 0x19, /* since v1.2 */
	KMIP_REC_CURVE_SECT113R1	= 0x1A, /* since v1.2 */
	KMIP_REC_CURVE_SECT113R2	= 0x1B, /* since v1.2 */
	KMIP_REC_CURVE_SECT131R1	= 0x1C, /* since v1.2 */
	KMIP_REC_CURVE_SECT131R2	= 0x1D, /* since v1.2 */
	KMIP_REC_CURVE_SECT163R1	= 0x1E, /* since v1.2 */
	KMIP_REC_CURVE_SECT193R1	= 0x1F, /* since v1.2 */
	KMIP_REC_CURVE_SECT193R2	= 0x20, /* since v1.2 */
	KMIP_REC_CURVE_SECT239K1	= 0x21, /* since v1.2 */
	KMIP_REC_CURVE_ANSIX9P192V2	= 0x22, /* since v1.2 */
	KMIP_REC_CURVE_ANSIX9P192V3	= 0x23, /* since v1.2 */
	KMIP_REC_CURVE_ANSIX9P239V1	= 0x24, /* since v1.2 */
	KMIP_REC_CURVE_ANSIX9P239V2	= 0x25, /* since v1.2 */
	KMIP_REC_CURVE_ANSIX9P239V3	= 0x26, /* since v1.2 */
	KMIP_REC_CURVE_ANSIX9C2PNB163V1	= 0x27, /* since v1.2 */
	KMIP_REC_CURVE_ANSIX9C2PNB163V2	= 0x28, /* since v1.2 */
	KMIP_REC_CURVE_ANSIX9C2PNB163V3	= 0x29, /* since v1.2 */
	KMIP_REC_CURVE_ANSIX9C2PNB176V1	= 0x2A, /* since v1.2 */
	KMIP_REC_CURVE_ANSIX9C2TNB191V1	= 0x2B, /* since v1.2 */
	KMIP_REC_CURVE_ANSIX9C2TNB191V2	= 0x2C, /* since v1.2 */
	KMIP_REC_CURVE_ANSIX9C2TNB191V3	= 0x2D, /* since v1.2 */
	KMIP_REC_CURVE_ANSIX9C2PNB208W1	= 0x2E, /* since v1.2 */
	KMIP_REC_CURVE_ANSIX9C2TNB239V1	= 0x2F, /* since v1.2 */
	KMIP_REC_CURVE_ANSIX9C2TNB239V2	= 0x30, /* since v1.2 */
	KMIP_REC_CURVE_ANSIX9C2TNB239V3	= 0x31, /* since v1.2 */
	KMIP_REC_CURVE_ANSIX9C2PNB272W1	= 0x32, /* since v1.2 */
	KMIP_REC_CURVE_ANSIX9C2PNB304W1	= 0x33, /* since v1.2 */
	KMIP_REC_CURVE_ANSIX9C2TNB359V1	= 0x34, /* since v1.2 */
	KMIP_REC_CURVE_ANSIX9C2PNB368W1	= 0x35, /* since v1.2 */
	KMIP_REC_CURVE_ANSIX9C2TNB431R1	= 0x36, /* since v1.2 */
	KMIP_REC_CURVE_BRAINPOOLP160R1	= 0x37, /* since v1.2 */
	KMIP_REC_CURVE_BRAINPOOLP160T1	= 0x38, /* since v1.2 */
	KMIP_REC_CURVE_BRAINPOOLP192R1	= 0x39, /* since v1.2 */
	KMIP_REC_CURVE_BRAINPOOLP192T1	= 0x3A, /* since v1.2 */
	KMIP_REC_CURVE_BRAINPOOLP224R1	= 0x3B, /* since v1.2 */
	KMIP_REC_CURVE_BRAINPOOLP224T1	= 0x3C, /* since v1.2 */
	KMIP_REC_CURVE_BRAINPOOLP256R1	= 0x3D, /* since v1.2 */
	KMIP_REC_CURVE_BRAINPOOLP256T1	= 0x3E, /* since v1.2 */
	KMIP_REC_CURVE_BRAINPOOLP320R1	= 0x3F, /* since v1.2 */
	KMIP_REC_CURVE_BRAINPOOLP320T1	= 0x40, /* since v1.2 */
	KMIP_REC_CURVE_BRAINPOOLP384R1	= 0x41, /* since v1.2 */
	KMIP_REC_CURVE_BRAINPOOLP384T1	= 0x42, /* since v1.2 */
	KMIP_REC_CURVE_BRAINPOOLP512R1	= 0x43, /* since v1.2 */
	KMIP_REC_CURVE_BRAINPOOLP512T1	= 0x44, /* since v1.2 */
	KMIP_REC_CURVE_CURVE25519	= 0x45, /* since v2.0 */
	KMIP_REC_CURVE_CURVE448		= 0x46, /* since v2.0 */
};

enum kmip_protection_level {
	KMIP_PROTECTION_LEVEL_HIGH	= 0x01, /* since v2.0 */
	KMIP_PROTECTION_LEVEL_LOW	= 0x02, /* since v2.0 */
};

enum kmip_key_value_location_type {
	KMIP_KEY_VAL_LOC_TYPE_UNINTERPRETED_TEXT_STRING	= 0x01,
	KMIP_KEY_VAL_LOC_TYPE_URI			= 0x02,
};

enum kmip_link_type {
	KMIP_LINK_TYPE_CERTIFICATE		= 0x0101,
	KMIP_LINK_TYPE_PUBLIC_KEY		= 0x0102,
	KMIP_LINK_TYPE_PRIVATE_KEY		= 0x0103,
	KMIP_LINK_TYPE_DERIVATION_BASE_OBJECT	= 0x0104,
	KMIP_LINK_TYPE_DERIVED_KEY		= 0x0105,
	KMIP_LINK_TYPE_REPLACEMENT_OBJECT	= 0x0106,
	KMIP_LINK_TYPE_REPLACED_OBJECT		= 0x0107,
	KMIP_LINK_TYPE_PARENT			= 0x0108, /* since v1.2 */
	KMIP_LINK_TYPE_CHILD			= 0x0109, /* since v1.2 */
	KMIP_LINK_TYPE_PREVIOUS			= 0x010A, /* since v1.2 */
	KMIP_LINK_TYPE_NEXT			= 0x010B, /* since v1.2 */
	KMIP_LINK_TYPE_PKCS_12_CERTIFICATE	= 0x010C, /* since v1.4 */
	KMIP_LINK_TYPE_PKCS_12_PASSWORD		= 0x010D, /* since v1.4 */
	KMIP_LINK_TYPE_WRAPPING_KEY		= 0x010E, /* since v2.0 */
};

enum kmip_client_registration_method {
	KMIP_CLIENT_REG_METH_UNSPECIFIED		= 0x01, /* since v1.3 */
	KMIP_CLIENT_REG_METH_SERVER_PRE_GENERATED	= 0x02, /* since v1.3 */
	KMIP_CLIENT_REG_METH_SERVER_ON_DEMAND		= 0x03, /* since v1.3 */
	KMIP_CLIENT_REG_METH_CLIENT_GENERATED		= 0x04, /* since v1.3 */
	KMIP_CLIENT_REG_METH_CLIENT_REGISTERED		= 0x05, /* since v1.3 */
};

enum kmip_rng_algorithm {
	KMIP_RNG_ALGO_UNSPECIFIED	= 0x01, /* since v1.3 */
	KMIP_RNG_ALGO_FIPS_186_2	= 0x02, /* since v1.3 */
	KMIP_RNG_ALGO_DRBG		= 0x03, /* since v1.3 */
	KMIP_RNG_ALGO_NRBG		= 0x04, /* since v1.3 */
	KMIP_RNG_ALGO_ANSI_X9_31	= 0x05, /* since v1.3 */
	KMIP_RNG_ALGO_ANSI_X9_62	= 0x06, /* since v1.3 */
};

enum kmip_drbg_algorithm {
	KMIP_DRBG_ALGO_UNSPECIFIED	= 0x01, /* since v1.3 */
	KMIP_DRBG_ALGO_DUAL_EC		= 0x02, /* since v1.3 */
	KMIP_DRBG_ALGO_HASH		= 0x03, /* since v1.3 */
	KMIP_DRBG_ALGO_HMAC		= 0x04, /* since v1.3 */
	KMIP_DRBG_ALGO_CTR		= 0x05, /* since v1.3 */
};

enum kmip_fips186_variation {
	KMIP_FIPS186_VARI_UNSPECIFIED		= 0x01, /* since v1.3 */
	KMIP_FIPS186_VARI_GP_X_ORIGINAL		= 0x02, /* since v1.3 */
	KMIP_FIPS186_VARI_GP_X_CHANGE_NOTICE	= 0x03, /* since v1.3 */
	KMIP_FIPS186_VARI_X_ORIGINAL		= 0x04, /* since v1.3 */
	KMIP_FIPS186_VARI_X_CHANGE_NOTICE	= 0x05, /* since v1.3 */
	KMIP_FIPS186_VARI_K_ORIGINAL		= 0x06, /* since v1.3 */
	KMIP_FIPS186_VARI_K_CHANGE_NOTICE	= 0x07, /* since v1.3 */
};

enum kmip_validation_authority_type {
	KMIP_VALIDATION_AUTH_TYPE_UNSPECIFIED		= 0x01, /* since v1.3 */
	KMIP_VALIDATION_AUTH_TYPE_NIST_CMVP		= 0x02, /* since v1.3 */
	KMIP_VALIDATION_AUTH_TYPE_COMMON_CRITERIA	= 0x03, /* since v1.3 */
};

enum kmip_validation_type {
	KMIP_VALIDATION_TYPE_UNSPECIFIED	= 0x01, /* since v1.3 */
	KMIP_VALIDATION_TYPE_HARDWARE		= 0x02, /* since v1.3 */
	KMIP_VALIDATION_TYPE_SOFTWARE		= 0x03, /* since v1.3 */
	KMIP_VALIDATION_TYPE_FIRMWARE		= 0x04, /* since v1.3 */
	KMIP_VALIDATION_TYPE_HYBRID		= 0x05, /* since v1.3 */
};

enum kmip_unwrap_mode {
	KMIP_UNWRAP_MODE_UNSPECIFIED		= 0x01, /* since v1.3 */
	KMIP_UNWRAP_MODE_PROCESSED		= 0x02, /* since v1.3 */
	KMIP_UNWRAP_MODE_NOT_PROCESSED		= 0x03, /* since v1.3 */
};

enum kmip_destroy_action {
	KMIP_DESTROY_ACTION_UNSPECIFIED			= 0x01, /* since v1.3 */
	KMIP_DESTROY_ACTION_KEY_MATERIAL_DELETED	= 0x02, /* since v1.3 */
	KMIP_DESTROY_ACTION_KEY_MATERIAL_SHREDDED	= 0x03, /* since v1.3 */
	KMIP_DESTROY_ACTION_META_DATA_DELETED		= 0x04, /* since v1.3 */
	KMIP_DESTROY_ACTION_META_DATA_SHREDDED		= 0x05, /* since v1.3 */
	KMIP_DESTROY_ACTION_DELETED			= 0x06, /* since v1.3 */
	KMIP_DESTROY_ACTION_SHREDDED			= 0x07, /* since v1.3 */
};

enum kmip_shredding_algorithm {
	KMIP_SHREDDING_ALGO_UNSPECIFIED		= 0x01, /* since v1.3 */
	KMIP_SHREDDING_ALGO_CRYPTOGRAPHIC	= 0x02, /* since v1.3 */
	KMIP_SHREDDING_ALGO_UNSUPPORTED		= 0x03, /* since v1.3 */
};

enum kmip_rng_mode {
	KMIP_RNG_MODE_UNSPECIFIED		= 0x01, /* since v1.3 */
	KMIP_RNG_MODE_SHARED_INSTANTIATION	= 0x02, /* since v1.3 */
	KMIP_RNG_MODE_NON_SHARED_INSTANCIATION	= 0x03, /* since v1.3 */
};

enum kmip_KMIP_PROFILE_s {
	KMIP_PROFILE_BASELINE_SERVER_BASIC_KMIP_V1_2		= 0x0001,
	KMIP_PROFILE_BASELINE_SERVER_TLS_V1_2_KMIP_V1_2		= 0x0002,
	KMIP_PROFILE_BASELINE_CLIENT_BASIC_KMIP_V1_2		= 0x0003,
	KMIP_PROFILE_BASELINE_CLIENT_TLS_V1_2_KMIP_V1_2		= 0x0004,
	KMIP_PROFILE_COMPLETE_SERVER_BASIC_KMIP_V1_2		= 0x0005,
	KMIP_PROFILE_COMPLETE_SERVER_TLS_V1_2_KMIP_V1_2		= 0x0006,
	KMIP_PROFILE_TAPE_LIBRARY_CLIENT_KMIP_V1_0		= 0x0007,
	KMIP_PROFILE_TAPE_LIBRARY_CLIENT_KMIP_V1_1		= 0x0008,
	KMIP_PROFILE_TAPE_LIBRARY_CLIENT_KMIP_V1_2		= 0x0009,
	KMIP_PROFILE_TAPE_LIBRARY_SERVER_KMIP_V1_0		= 0x000A,
	KMIP_PROFILE_TAPE_LIBRARY_SERVER_KMIP_V1_1		= 0x000B,
	KMIP_PROFILE_TAPE_LIBRARY_SERVER_KMIP_V1_2		= 0x000C,
	KMIP_PROFILE_SYMMETRIC_KEY_LIFECYCLE_CLIENT_KMIP_V1_0	= 0x000D,
	KMIP_PROFILE_SYMMETRIC_KEY_LIFECYCLE_CLIENT_KMIP_V1_1	= 0x000E,
	KMIP_PROFILE_SYMMETRIC_KEY_LIFECYCLE_CLIENT_KMIP_V1_2	= 0x000F,
	KMIP_PROFILE_SYMMETRIC_KEY_LIFECYCLE_SERVER_KMIP_V1_0	= 0x0010,
	KMIP_PROFILE_SYMMETRIC_KEY_LIFECYCLE_SERVER_KMIP_V1_1	= 0x0011,
	KMIP_PROFILE_SYMMETRIC_KEY_LIFECYCLE_SERVER_KMIP_V1_2	= 0x0012,
	KMIP_PROFILE_ASYMMETRIC_KEY_LIFECYCLE_CLIENT_KMIP_V1_0	= 0x0013,
	KMIP_PROFILE_ASYMMETRIC_KEY_LIFECYCLE_CLIENT_KMIP_V1_1	= 0x0014,
	KMIP_PROFILE_ASYMMETRIC_KEY_LIFECYCLE_CLIENT_KMIP_V1_2	= 0x0015,
	KMIP_PROFILE_ASYMMETRIC_KEY_LIFECYCLE_SERVER_KMIP_V1_0	= 0x0016,
	KMIP_PROFILE_ASYMMETRIC_KEY_LIFECYCLE_SERVER_KMIP_V1_1	= 0x0017,
	KMIP_PROFILE_ASYMMETRIC_KEY_LIFECYCLE_SERVER_KMIP_V1_2	= 0x0018,
	KMIP_PROFILE_BASIC_CRYPTOGRAPHIC_CLIENT_KMIP_V1_2	= 0x0019,
	KMIP_PROFILE_BASIC_CRYPTOGRAPHIC_SERVER_KMIP_V1_2	= 0x001A,
	KMIP_PROFILE_ADVANCED_CRYPTOGRAPHIC_CLIENT_KMIP_V1_2	= 0x001B,
	KMIP_PROFILE_ADVANCED_CRYPTOGRAPHIC_SERVER_KMIP_V1_2	= 0x001C,
	KMIP_PROFILE_RNG_CRYPTOGRAPHIC_CLIENT_KMIP_V1_2		= 0x001D,
	KMIP_PROFILE_RNG_CRYPTOGRAPHIC_SERVER_KMIP_V1_2		= 0x001E,
	KMIP_PROFILE_BASIC_SYMMETRIC_KEY_FOUNDRY_CLIENT_KMIP_V1_0 = 0x001F,
	KMIP_PROFILE_INTERMEDIATE_SYMMETRIC_KEY_FOUNDRY_CLIENT_KMIP_V1_0
								= 0x0020,
	KMIP_PROFILE_ADVANCED_SYMMETRIC_KEY_FOUNDRY_CLIENT_KMIP_V1_0
								= 0x0021,
	KMIP_PROFILE_BASIC_SYMMETRIC_KEY_FOUNDRY_CLIENT_KMIP_V1_1 = 0x0022,
	KMIP_PROFILE_INTERMEDIATE_SYMMETRIC_KEY_FOUNDRY_CLIENT_KMIP_V1_1
								= 0x0023,
	KMIP_PROFILE_ADVANCED_SYMMETRIC_KEY_FOUNDRY_CLIENT_KMIP_V1_1
								= 0x0024,
	KMIP_PROFILE_BASIC_SYMMETRIC_KEY_FOUNDRY_CLIENT_KMIP_V1_2 = 0x0025,
	KMIP_PROFILE_INTERMEDIATE_SYMMETRIC_KEY_FOUNDRY_CLIENT_KMIP_V1_2
								= 0x0026,
	KMIP_PROFILE_ADVANCED_SYMMETRIC_KEY_FOUNDRY_CLIENT_KMIP_V1_2
								= 0x0027,
	KMIP_PROFILE_SYMMETRIC_KEY_FOUNDRY_SERVER_KMIP_V1_0	= 0x0028,
	KMIP_PROFILE_SYMMETRIC_KEY_FOUNDRY_SERVER_KMIP_V1_1	= 0x0029,
	KMIP_PROFILE_SYMMETRIC_KEY_FOUNDRY_SERVER_KMIP_V1_2	= 0x002A,
	KMIP_PROFILE_OPAQUE_MANAGED_OBJECT_STORE_CLIENT_KMIP_V1_0 = 0x002B,
	KMIP_PROFILE_OPAQUE_MANAGED_OBJECT_STORE_CLIENT_KMIP_V1_1 = 0x002C,
	KMIP_PROFILE_OPAQUE_MANAGED_OBJECT_STORE_CLIENT_KMIP_V1_2 = 0x002D,
	KMIP_PROFILE_OPAQUE_MANAGED_OBJECT_STORE_SERVER_KMIP_V1_0 = 0x002E,
	KMIP_PROFILE_OPAQUE_MANAGED_OBJECT_STORE_SERVER_KMIP_V1_1 = 0x002F,
	KMIP_PROFILE_OPAQUE_MANAGED_OBJECT_STORE_SERVER_KMIP_V1_2 = 0x0030,
	KMIP_PROFILE_SUITE_B_MINLOS_128_CLIENT_KMIP_V1_0	= 0x0031,
	KMIP_PROFILE_SUITE_B_MINLOS_128_CLIENT_KMIP_V1_1	= 0x0032,
	KMIP_PROFILE_SUITE_B_MINLOS_128_CLIENT_KMIP_V1_2	= 0x0033,
	KMIP_PROFILE_SUITE_B_MINLOS_128_SERVER_KMIP_V1_0	= 0x0034,
	KMIP_PROFILE_SUITE_B_MINLOS_128_SERVER_KMIP_V1_1	= 0x0035,
	KMIP_PROFILE_SUITE_B_MINLOS_128_SERVER_KMIP_V1_2	= 0x0036,
	KMIP_PROFILE_SUITE_B_MINLOS_192_CLIENT_KMIP_V1_0	= 0x0037,
	KMIP_PROFILE_SUITE_B_MINLOS_192_CLIENT_KMIP_V1_1	= 0x0038,
	KMIP_PROFILE_SUITE_B_MINLOS_192_CLIENT_KMIP_V1_2	= 0x0039,
	KMIP_PROFILE_SUITE_B_MINLOS_192_SERVER_KMIP_V1_0	= 0x003A,
	KMIP_PROFILE_SUITE_B_MINLOS_192_SERVER_KMIP_V1_1	= 0x003B,
	KMIP_PROFILE_SUITE_B_MINLOS_192_SERVER_KMIP_V1_2	= 0x003C,
	KMIP_PROFILE_STORAGE_ARRAY_WITH_SELF_ENCRYPTING_DRIVE_CLIENT_KMIP_V1_0
								= 0x003D,
	KMIP_PROFILE_STORAGE_ARRAY_WITH_SELF_ENCRYPTING_DRIVE_CLIENT_KMIP_V1_1
								= 0x003E,
	KMIP_PROFILE_STORAGE_ARRAY_WITH_SELF_ENCRYPTING_DRIVE_CLIENT_KMIP_V1_2
								= 0x003F,
	KMIP_PROFILE_STORAGE_ARRAY_WITH_SELF_ENCRYPTING_DRIVE_SERVER_KMIP_V1_0
								= 0x0040,
	KMIP_PROFILE_STORAGE_ARRAY_WITH_SELF_ENCRYPTING_DRIVE_SERVER_KMIP_V1_1
								= 0x0041,
	KMIP_PROFILE_STORAGE_ARRAY_WITH_SELF_ENCRYPTING_DRIVE_SERVER_KMIP_V1_2
								= 0x0042,
	KMIP_PROFILE_HTTPS_CLIENT_KMIP_V1_0			= 0x0043,
	KMIP_PROFILE_HTTPS_CLIENT_KMIP_V1_1			= 0x0044,
	KMIP_PROFILE_HTTPS_CLIENT_KMIP_V1_2			= 0x0045,
	KMIP_PROFILE_HTTPS_SERVER_KMIP_V1_0			= 0x0046,
	KMIP_PROFILE_HTTPS_SERVER_KMIP_V1_1			= 0x0047,
	KMIP_PROFILE_HTTPS_SERVER_KMIP_V1_2			= 0x0048,
	KMIP_PROFILE_JSON_CLIENT_KMIP_V1_0			= 0x0049,
	KMIP_PROFILE_JSON_CLIENT_KMIP_V1_1			= 0x004A,
	KMIP_PROFILE_JSON_CLIENT_KMIP_V1_2			= 0x004B,
	KMIP_PROFILE_JSON_SERVER_KMIP_V1_0			= 0x004C,
	KMIP_PROFILE_JSON_SERVER_KMIP_V1_1			= 0x004D,
	KMIP_PROFILE_JSON_SERVER_KMIP_V1_2			= 0x004E,
	KMIP_PROFILE_XML_CLIENT_KMIP_V1_0			= 0x004F,
	KMIP_PROFILE_XML_CLIENT_KMIP_V1_1			= 0x0050,
	KMIP_PROFILE_XML_CLIENT_KMIP_V1_2			= 0x0051,
	KMIP_PROFILE_XML_SERVER_KMIP_V1_0			= 0x0052,
	KMIP_PROFILE_XML_SERVER_KMIP_V1_1			= 0x0053,
	KMIP_PROFILE_XML_SERVER_KMIP_V1_2			= 0x0054,
	KMIP_PROFILE_BASELINE_SERVER_BASIC_KMIP_V1_3		= 0x0055,
	KMIP_PROFILE_BASELINE_SERVER_TLS_V1_2_KMIP_V1_3		= 0x0056,
	KMIP_PROFILE_BASELINE_CLIENT_BASIC_KMIP_V1_3		= 0x0057,
	KMIP_PROFILE_BASELINE_CLIENT_TLS_V1_2_KMIP_V1_3		= 0x0058,
	KMIP_PROFILE_COMPLETE_SERVER_BASIC_KMIP_V1_3		= 0x0059,
	KMIP_PROFILE_COMPLETE_SERVER_TLS_V1_2_KMIP_V1_3		= 0x005A,
	KMIP_PROFILE_TAPE_LIBRARY_CLIENT_KMIP_V1_3		= 0x005B,
	KMIP_PROFILE_TAPE_LIBRARY_SERVER_KMIP_V1_3		= 0x005C,
	KMIP_PROFILE_SYMMETRIC_KEY_LIFECYCLE_CLIENT_KMIP_V1_3	= 0x005D,
	KMIP_PROFILE_SYMMETRIC_KEY_LIFECYCLE_SERVER_KMIP_V1_3	= 0x005E,
	KMIP_PROFILE_ASYMMETRIC_KEY_LIFECYCLE_CLIENT_KMIP_V1_3	= 0x005F,
	KMIP_PROFILE_ASYMMETRIC_KEY_LIFECYCLE_SERVER_KMIP_V1_3	= 0x0060,
	KMIP_PROFILE_BASIC_CRYPTOGRAPHIC_CLIENT_KMIP_V1_3	= 0x0061,
	KMIP_PROFILE_BASIC_CRYPTOGRAPHIC_SERVER_KMIP_V1_3	= 0x0062,
	KMIP_PROFILE_ADVANCED_CRYPTOGRAPHIC_CLIENT_KMIP_V1_3	= 0x0063,
	KMIP_PROFILE_ADVANCED_CRYPTOGRAPHIC_SERVER_KMIP_V1_3	= 0x0064,
	KMIP_PROFILE_RNG_CRYPTOGRAPHIC_CLIENT_KMIP_V1_3		= 0x0065,
	KMIP_PROFILE_RNG_CRYPTOGRAPHIC_SERVER_KMIP_V1_3		= 0x0066,
	KMIP_PROFILE_BASIC_SYMMETRIC_KEY_FOUNDRY_CLIENT_KMIP_V1_3 = 0x0067,
	KMIP_PROFILE_INTERMEDIATE_SYMMETRIC_KEY_FOUNDRY_CLIENT_KMIP_V1_3
								= 0x0068,
	KMIP_PROFILE_ADVANCED_SYMMETRIC_KEY_FOUNDRY_CLIENT_KMIP_V1_3
								= 0x0069,
	KMIP_PROFILE_SYMMETRIC_KEY_FOUNDRY_SERVER_KMIP_V1_3	= 0x006A,
	KMIP_PROFILE_OPAQUE_MANAGED_OBJECT_STORE_CLIENT_KMIP_V1_3 = 0x006B,
	KMIP_PROFILE_OPAQUE_MANAGED_OBJECT_STORE_SERVER_KMIP_V1_3 = 0x006C,
	KMIP_PROFILE_SUITE_B_MINLOS_128_CLIENT_KMIP_V1_3	= 0x006D,
	KMIP_PROFILE_SUITE_B_MINLOS_128_SERVER_KMIP_V1_3	= 0x006E,
	KMIP_PROFILE_SUITE_B_MINLOS_192_CLIENT_KMIP_V1_3	= 0x006F,
	KMIP_PROFILE_SUITE_B_MINLOS_192_SERVER_KMIP_V1_3	= 0x0070,
	KMIP_PROFILE_STORAGE_ARRAY_WITH_SELF_ENCRYPTING_DRIVE_CLIENT_KMIP_V1_3
								= 0x0071,
	KMIP_PROFILE_STORAGE_ARRAY_WITH_SELF_ENCRYPTING_DRIVE_SERVER_KMIP_V1_3
								= 0x0072,
	KMIP_PROFILE_HTTPS_CLIENT_KMIP_V1_3			= 0x0073,
	KMIP_PROFILE_HTTPS_SERVER_KMIP_V1_3			= 0x0074,
	KMIP_PROFILE_JSON_CLIENT_KMIP_V1_3			= 0x0075,
	KMIP_PROFILE_JSON_SERVER_KMIP_V1_3			= 0x0076,
	KMIP_PROFILE_XML_CLIENT_KMIP_V1_3			= 0x0077,
	KMIP_PROFILE_XML_SERVER_KMIP_V1_3			= 0x0078,
	KMIP_PROFILE_BASELINE_SERVER_BASIC_KMIP_V1_4		= 0x0079,
	KMIP_PROFILE_BASELINE_SERVER_TLS_V1_2_KMIP_V1_4		= 0x007A,
	KMIP_PROFILE_BASELINE_CLIENT_BASIC_KMIP_V1_4		= 0x007B,
	KMIP_PROFILE_BASELINE_CLIENT_TLS_V1_2_KMIP_V1_4		= 0x007C,
	KMIP_PROFILE_COMPLETE_SERVER_BASIC_KMIP_V1_4		= 0x007D,
	KMIP_PROFILE_COMPLETE_SERVER_TLS_V1_2_KMIP_V1_4		= 0x007E,
	KMIP_PROFILE_TAPE_LIBRARY_CLIENT_KMIP_V1_4		= 0x007F,
	KMIP_PROFILE_TAPE_LIBRARY_SERVER_KMIP_V1_4		= 0x0080,
	KMIP_PROFILE_SYMMETRIC_KEY_LIFECYCLE_CLIENT_KMIP_V1_4	= 0x0081,
	KMIP_PROFILE_SYMMETRIC_KEY_LIFECYCLE_SERVER_KMIP_V1_4	= 0x0082,
	KMIP_PROFILE_ASYMMETRIC_KEY_LIFECYCLE_CLIENT_KMIP_V1_4	= 0x0083,
	KMIP_PROFILE_ASYMMETRIC_KEY_LIFECYCLE_SERVER_KMIP_V1_4	= 0x0084,
	KMIP_PROFILE_BASIC_CRYPTOGRAPHIC_CLIENT_KMIP_V1_4	= 0x0085,
	KMIP_PROFILE_BASIC_CRYPTOGRAPHIC_SERVER_KMIP_V1_4	= 0x0086,
	KMIP_PROFILE_ADVANCED_CRYPTOGRAPHIC_CLIENT_KMIP_V1_4	= 0x0087,
	KMIP_PROFILE_ADVANCED_CRYPTOGRAPHIC_SERVER_KMIP_V1_4	= 0x0088,
	KMIP_PROFILE_RNG_CRYPTOGRAPHIC_CLIENT_KMIP_V1_4		= 0x0089,
	KMIP_PROFILE_RNG_CRYPTOGRAPHIC_SERVER_KMIP_V1_4		= 0x008A,
	KMIP_PROFILE_BASIC_SYMMETRIC_KEY_FOUNDRY_CLIENT_KMIP_V1_4 = 0x008B,
	KMIP_PROFILE_INTERMEDIATE_SYMMETRIC_KEY_FOUNDRY_CLIENT_KMIP_V1_4
								= 0x008C,
	KMIP_PROFILE_ADVANCED_SYMMETRIC_KEY_FOUNDRY_CLIENT_KMIP_V1_4
								= 0x008D,
	KMIP_PROFILE_SYMMETRIC_KEY_FOUNDRY_SERVER_KMIP_V1_4	= 0x008E,
	KMIP_PROFILE_OPAQUE_MANAGED_OBJECT_STORE_CLIENT_KMIP_V1_4 = 0x008F,
	KMIP_PROFILE_OPAQUE_MANAGED_OBJECT_STORE_SERVER_KMIP_V1_4 = 0x0090,
	KMIP_PROFILE_SUITE_B_MINLOS_128_CLIENT_KMIP_V1_4	= 0x0091,
	KMIP_PROFILE_SUITE_B_MINLOS_128_SERVER_KMIP_V1_4	= 0x0092,
	KMIP_PROFILE_SUITE_B_MINLOS_192_CLIENT_KMIP_V1_4	= 0x0093,
	KMIP_PROFILE_SUITE_B_MINLOS_192_SERVER_KMIP_V1_4	= 0x0094,
	KMIP_PROFILE_STORAGE_ARRAY_WITH_SELF_ENCRYPTING_DRIVE_CLIENT_KMIP_V1_4
								= 0x0095,
	KMIP_PROFILE_STORAGE_ARRAY_WITH_SELF_ENCRYPTING_DRIVE_SERVER_KMIP_V1_4
								= 0x0096,
	KMIP_PROFILE_HTTPS_CLIENT_KMIP_V1_4			= 0x0097,
	KMIP_PROFILE_HTTPS_SERVER_KMIP_V1_4			= 0x0098,
	KMIP_PROFILE_JSON_CLIENT_KMIP_V1_4			= 0x0099,
	KMIP_PROFILE_JSON_SERVER_KMIP_V1_4			= 0x009A,
	KMIP_PROFILE_XML_CLIENT_KMIP_V1_4			= 0x009B,
	KMIP_PROFILE_XML_SERVER_KMIP_V1_4			= 0x009C,
	KMIP_PROFILE_COMPLETE_SERVER_BASIC			= 0x0104,
	KMIP_PROFILE_COMPLETE_SERVER_TLS_V1_2			= 0x0105,
	KMIP_PROFILE_TAPE_LIBRARY_CLIENT			= 0x0106,
	KMIP_PROFILE_TAPE_LIBRARY_SERVER			= 0x0107,
	KMIP_PROFILE_SYMMETRIC_KEY_LIFECYCLE_CLIENT		= 0x0108,
	KMIP_PROFILE_SYMMETRIC_KEY_LIFECYCLE_SERVER		= 0x0109,
	KMIP_PROFILE_ASYMMETRIC_KEY_LIFECYCLE_CLIENT		= 0x010A,
	KMIP_PROFILE_ASYMMETRIC_KEY_LIFECYCLE_SERVER		= 0x010B,
	KMIP_PROFILE_BASIC_CRYPTOGRAPHIC_CLIENT			= 0x010C,
	KMIP_PROFILE_BASIC_CRYPTOGRAPHIC_SERVER			= 0x010D,
	KMIP_PROFILE_ADVANCED_CRYPTOGRAPHIC_CLIENT		= 0x010E,
	KMIP_PROFILE_ADVANCED_CRYPTOGRAPHIC_SERVER		= 0x010F,
	KMIP_PROFILE_RNG_CRYPTOGRAPHIC_CLIENT			= 0x0110,
	KMIP_PROFILE_RNG_CRYPTOGRAPHIC_SERVER			= 0x0111,
	KMIP_PROFILE_BASIC_SYMMETRIC_KEY_FOUNDRY_CLIENT		= 0x0112,
	KMIP_PROFILE_INTERMEDIATE_SYMMETRIC_KEY_FOUNDRY_CLIENT	= 0x0113,
	KMIP_PROFILE_ADVANCED_SYMMETRIC_KEY_FOUNDRY_CLIENT	= 0x0114,
	KMIP_PROFILE_SYMMETRIC_KEY_FOUNDRY_SERVER		= 0x0115,
	KMIP_PROFILE_OPAQUE_MANAGED_OBJECT_STORE_CLIENT		= 0x0116,
	KMIP_PROFILE_OPAQUE_MANAGED_OBJECT_STORE_SERVER		= 0x0117,
	KMIP_PROFILE_STORAGE_ARRAY_WITH_SELF_ENCRYPTING_DRIVE_CLIENT
								= 0x011C,
	KMIP_PROFILE_STORAGE_ARRAY_WITH_SELF_ENCRYPTING_DRIVE_SERVER
								= 0x011D,
	KMIP_PROFILE_HTTPS_CLIENT				= 0x011E,
	KMIP_PROFILE_HTTPS_SERVER				= 0x011F,
	KMIP_PROFILE_JSON_CLIENT				= 0x0120,
	KMIP_PROFILE_JSON_SERVER				= 0x0121,
	KMIP_PROFILE_XML_CLIENT					= 0x0122,
	KMIP_PROFILE_XML_SERVER					= 0x0123,
	KMIP_PROFILE_AES_XTS_CLIENT				= 0x0124,
	KMIP_PROFILE_AES_XTS_SERVER				= 0x0125,
	KMIP_PROFILE_QUANTUM_SAFE_CLIENT			= 0x0126,
	KMIP_PROFILE_QUANTUM_SAFE_SERVER			= 0x0127,
	KMIP_PROFILE_PKCS_11_CLIENT				= 0x0128,
	KMIP_PROFILE_PKCS_11_SERVER				= 0x0129,
	KMIP_PROFILE_BASELINE_CLIENT				= 0x012A,
	KMIP_PROFILE_BASELINE_SERVER				= 0x012B,
	KMIP_PROFILE_COMPLETE_SERVER				= 0x012C,
};

struct kmip_version {
	int32_t major;
	int32_t minor;
};

enum kmip_encoding {
	KMIP_ENCODING_TTLV		= 1,
#ifdef HAVE_LIBCURL
#ifdef HAVE_LIBJSONC
	KMIP_ENCODING_JSON		= 2, /* Only via HTTPS transport */
#endif
#ifdef HAVE_LIBXML2
	KMIP_ENCODING_XML		= 3, /* Only via HTTPS transport */
#endif
#endif
};

enum kmip_transport {
	KMIP_TRANSPORT_PLAIN_TLS	= 1,
#ifdef HAVE_LIBCURL
	KMIP_TRANSPORT_HTTPS		= 2,
#endif
};

#define KMIP_DEFAULT_PLAIN_TLS_PORT	"5696"
#define KMIP_DEFAULT_PLAIN_TLS_PORT_NUM	5696
#ifdef HAVE_LIBCURL
#define KMIP_DEFAULT_HTTPS_PORT		"5696"
#define KMIP_DEFAULT_HTTPS_PORT_NUM	5696
#endif

struct kmip_conn_config {
	/** Encoding used for the KMIP messages */
	enum kmip_encoding encoding;
	/** Transport method used to deliver KMIP messages */
	enum kmip_transport transport;
	/**
	 * The KMIP server.
	 * For Plain-TLS transport, only the hostname and optional port number.
	 * For HTTPS transport, an URL in the form
	 * 'https://hostname[:port]/uri'
	 */
	const char *server;
	/** The client key as an OpenSSL PKEY object. */
	EVP_PKEY *tls_client_key;
	/** File name of the client certificate PEM file */
	const char *tls_client_cert;
	/**
	 * Optional: File name of the CA bundle PEM file, or a name of a
	 * directory the multiple CA certificates. If this is NULL, then the
	 * default system path for CA certificates is used
	 */
	const char *tls_ca;
	/**
	 * Optional: File name of a PEM file holding a CA certificate of the
	 * issuer
	 */
	const char *tls_issuer_cert;
	/**
	 * Optional: File name of a PEM file containing the servers pinned
	 * public key. Public key pinning requires that verify_peer or
	 * verify_host (or both) is true.
	 */
	const char *tls_pinned_pubkey;
	/**
	 * Optional: File name of a PEM file containing the server's
	 * certificate. This can be used to allow peer verification with
	 * self-signed server certificates
	 */
	const char *tls_server_cert;
	/** If true, the peer certificate is verified */
	bool tls_verify_peer;
	/**
	 * If true, that the server certificate is for the server it is known
	 * as (i.e. the hostname in the url)
	 */
	bool tls_verify_host;
	/**
	 * Optional: A list of ciphers for TLSv1.2 and below. This is a colon
	 * separated list of cipher strings. The format of the string is
	 * described in
	 * https://www.openssl.org/docs/man1.1.1/man1/ciphers.html
	 */
	const char *tls_cipher_list;
	/**
	 * Optional: A list of ciphers for TLSv1.3. This is a colon separated
	 * list of TLSv1.3 ciphersuite names in order of preference. Valid
	 * TLSv1.3 ciphersuite names are:
	 * - TLS_AES_128_GCM_SHA256
	 * - TLS_AES_256_GCM_SHA384
	 * - TLS_CHACHA20_POLY1305_SHA256
	 * - TLS_AES_128_CCM_SHA256
	 * - TLS_AES_128_CCM_8_SHA256
	 */
	const char *tls13_cipher_list;
};

/* Opaque KMIP node and connection structures */
struct kmip_connection;
struct kmip_node;

/* Generic KMIP node constructors/destructors and getters */
struct kmip_node *kmip_node_clone(const struct kmip_node *node);
void kmip_node_upref(struct kmip_node *node);
void kmip_node_free(struct kmip_node *node);
enum kmip_tag kmip_node_get_tag(const struct kmip_node *node);
enum kmip_type kmip_node_get_type(const struct kmip_node *node);
char *kmip_node_get_name(const struct kmip_node *node);
void kmip_node_dump(struct kmip_node *node, bool debug);

struct kmip_node *kmip_node_new_structure(enum kmip_tag tag, const char *name,
					  unsigned int num_elements,
					  struct kmip_node **elements);
struct kmip_node *kmip_node_new_structure_va(enum kmip_tag tag,
					     const char *name,
					     unsigned int num_elements, ...);
unsigned int kmip_node_get_structure_element_count(
					const struct kmip_node *node);
struct kmip_node *kmip_node_get_structure_element_by_index(
					const struct kmip_node *node,
					unsigned int index);
unsigned int kmip_node_get_structure_element_by_tag_count(
					const struct kmip_node *node,
					enum kmip_tag tag);
struct kmip_node *kmip_node_get_structure_element_by_tag(
					const struct kmip_node *node,
					enum kmip_tag tag, unsigned int index);
int kmip_node_add_structure_element(struct kmip_node *node,
				    struct kmip_node *element);
int kmip_node_add_structure_elements(struct kmip_node *node,
				     unsigned int num_elements,
				     struct kmip_node **elements);

struct kmip_node *kmip_node_new_integer(enum kmip_tag tag, const char *name,
					int32_t value);
int32_t kmip_node_get_integer(const struct kmip_node *node);

struct kmip_node *kmip_node_new_long(enum kmip_tag tag, const char *name,
				     int64_t value);
int64_t kmip_node_get_long(const struct kmip_node *node);

struct kmip_node *kmip_node_new_bigint(enum kmip_tag tag, const char *name,
				       const BIGNUM *value);
const BIGNUM *kmip_node_get_bigint(const struct kmip_node *node);

struct kmip_node *kmip_node_new_enumeration(enum kmip_tag tag, const char *name,
					    uint32_t enumeration);
uint32_t kmip_node_get_enumeration(const struct kmip_node *node);

struct kmip_node *kmip_node_new_boolean(enum kmip_tag tag, const char *name,
					bool value);
bool kmip_node_get_boolean(const struct kmip_node *node);

struct kmip_node *kmip_node_new_text_string(enum kmip_tag tag, const char *name,
					    const char *value);
const char *kmip_node_get_text_string(const struct kmip_node *node);

struct kmip_node *kmip_node_new_byte_string(enum kmip_tag tag, const char *name,
					    const unsigned char *value,
					    uint32_t length);
const unsigned char *kmip_node_get_byte_string(const struct kmip_node *node,
					 uint32_t *length);

struct kmip_node *kmip_node_new_date_time(enum kmip_tag tag, const char *name,
					  int64_t value);
int64_t kmip_node_get_date_time(const struct kmip_node *node);

struct kmip_node *kmip_node_new_interval(enum kmip_tag tag, const char *name,
					 uint32_t value);
uint32_t kmip_node_get_interval(const struct kmip_node *node);

struct kmip_node *kmip_node_new_date_time_ext(enum kmip_tag tag,
					      const char *name,
					      int64_t value);
int64_t kmip_node_get_date_time_ext(const struct kmip_node *node);

/* Generic functions */
void kmip_set_default_protocol_version(const struct kmip_version *version);
const struct kmip_version *kmip_get_default_protocol_version(void);

/* Request related functions */
struct kmip_node *kmip_new_protocol_version(const struct kmip_version *version);
struct kmip_node *kmip_new_profile_version(const struct kmip_version *version);
struct kmip_node *kmip_new_request_header(const struct kmip_version *version,
					  int32_t max_response_size,
					  const char *client_corr_value,
					  const char *server_corr_value,
					  bool asynchronous,
					  struct kmip_node *authentication,
				enum kmip_batch_error_cont_option batch_err_opt,
					  bool batch_order_option,
					  int32_t batch_count);
struct kmip_node *kmip_new_request_batch_item(enum kmip_operation operation,
					      unsigned char *batch_id,
					      uint32_t batch_id_length,
					      struct kmip_node *payload);
struct kmip_node *kmip_new_request(struct kmip_node *request_header,
				   int32_t batch_count,
				   struct kmip_node **batch_items);
struct kmip_node *kmip_new_request_va(struct kmip_node *request_header,
				      int32_t batch_count, ...);
struct kmip_node *kmip_new_query_request_payload(unsigned int query_count,
				const enum kmip_query_function *functions);
struct kmip_node *kmip_new_query_request_payload_va(unsigned int query_count,
						    ...);
struct kmip_node *kmip_new_discover_versions_payload(int version_count,
					const struct kmip_version *versions);
struct kmip_node *kmip_new_discover_versions_payload_va(int version_count, ...);
struct kmip_node *kmip_new_protection_storage_masks(unsigned int masks_count,
						    int32_t *masks);
struct kmip_node *kmip_new_protection_storage_masks_va(unsigned int masks_count,
						       ...);
struct kmip_node *kmip_new_create_request_payload(
					const struct kmip_version *version,
					enum kmip_object_type obj_type,
					struct kmip_node *prot_storage_masks,
					unsigned int attrs_count,
					struct kmip_node **attrs);
struct kmip_node *kmip_new_create_request_payload_va(
					const struct kmip_version *version,
					enum kmip_object_type obj_type,
					struct kmip_node *prot_storage_masks,
					unsigned int attrs_count, ...);
struct kmip_node *kmip_new_get_attribute_list_request_payload(
					struct kmip_node *unique_id);
struct kmip_node *kmip_new_get_attributes_request_payload(
					const struct kmip_version *version,
					struct kmip_node *unique_id,
					unsigned int num_attrs,
					struct kmip_node **attr_refs);
struct kmip_node *kmip_new_get_attributes_request_payload_va(
					const struct kmip_version *version,
					struct kmip_node *unique_id,
					unsigned int num_attrs, ...);
struct kmip_node *kmip_new_add_attribute_request_payload(
					const struct kmip_version *version,
					struct kmip_node *unique_id,
					struct kmip_node *v2_attr);
struct kmip_node *kmip_new_modify_attribute_request_payload(
					const struct kmip_version *version,
					struct kmip_node *unique_id,
					struct kmip_node *v2_current,
					struct kmip_node *v2_attr);
struct kmip_node *kmip_new_set_attribute_v2_request_payload(
					struct kmip_node *unique_id,
					struct kmip_node *v2_attr);
struct kmip_node *kmip_new_delete_attribute_request_payload(
					const struct kmip_version *version,
					struct kmip_node *unique_id,
					struct kmip_node *v2_current,
					struct kmip_node *attr_ref);
struct kmip_node *kmip_new_activate_request_payload(
						struct kmip_node *unique_id);
struct kmip_node *kmip_new_destroy_request_payload(struct kmip_node *unique_id);
struct kmip_node *kmip_new_archive_request_payload(struct kmip_node *unique_id);
struct kmip_node *kmip_new_recover_request_payload(struct kmip_node *unique_id);
struct kmip_node *kmip_new_revoke_request_payload(struct kmip_node *unique_id,
						  enum kmip_revoke_reason rsn,
						  const char *message,
						  uint64_t compromise_date);
struct kmip_node *kmip_new_locate_request_payload(
				const struct kmip_version *version,
				int32_t max_items, int32_t offset_items,
				enum kmip_storage_status_mask storage_status,
				enum kmip_object_group_member obj_group,
				unsigned int attrs_count,
				struct kmip_node **attrs);
struct kmip_node *kmip_new_locate_request_payload_va(
				const struct kmip_version *version,
				int32_t max_items, int32_t offset_items,
				enum kmip_storage_status_mask storage_status,
				enum kmip_object_group_member obj_group,
				unsigned int attrs_count, ...);
struct kmip_node *kmip_new_register_request_payload(
					const struct kmip_version *version,
					enum kmip_object_type obj_type,
					struct kmip_node *object,
					struct kmip_node *prot_storage_masks,
					unsigned int attrs_count,
					struct kmip_node **attrs);
struct kmip_node *kmip_new_register_request_payload_va(
					const struct kmip_version *version,
					enum kmip_object_type obj_type,
					struct kmip_node *object,
					struct kmip_node *prot_storage_masks,
					unsigned int attrs_count, ...);
struct kmip_node *kmip_new_get_request_payload(
					const struct kmip_version *version,
					struct kmip_node *unique_id,
					enum kmip_key_format_type format_type,
					enum kmip_key_wrap_type wrap_type,
				enum kmip_key_compression_type compr_type,
					struct kmip_node *wrap_specification);

/* Response related functions */
int kmip_get_protocol_version(const struct kmip_node *node,
			      struct kmip_version *version);
int kmip_get_profile_version(const struct kmip_node *node,
			     struct kmip_version *version);
int kmip_get_response_header(const struct kmip_node *node,
			     struct kmip_version *version,
			     int64_t *time_stamp,
			     const char **client_corr_value,
			     const char **server_corr_value,
			     int32_t *batch_count);
int kmip_get_response_batch_item(const struct kmip_node *node,
				 enum kmip_operation *operation,
				 const unsigned char **batch_id,
				 uint32_t *batch_id_length,
				 enum kmip_result_status *status,
				 enum kmip_result_reason *reason,
				 const char **message,
				 const unsigned char **async_corr_value,
				 uint32_t *async_corr_value_len,
				 struct kmip_node **payload);
int kmip_get_response(const struct kmip_node *node,
		      struct kmip_node **response_header,
		      unsigned int batch_index,
		      struct kmip_node **batch_item);
int kmip_get_query_response_payload(const struct kmip_node *node,
				    enum kmip_query_function query_function,
				    unsigned int *num_results,
				    unsigned int result_index,
				    struct kmip_node **result);
int kmip_get_discover_versions_response_payload(const struct kmip_node *node,
						unsigned int *num_versions,
						unsigned int index,
						struct kmip_version *version);
int kmip_get_create_response_payload(const struct kmip_node *node,
				     enum kmip_object_type *obj_type,
				     struct kmip_node **unique_id,
				     unsigned int *num_attrs,
				     unsigned int attr_index,
				     struct kmip_node **attribute);
int kmip_get_get_attribute_list_response_payload(const struct kmip_node *node,
				     struct kmip_node **unique_id,
				     unsigned int *num_attr_refs,
				     unsigned int index,
				     struct kmip_node **attr_ref);
int kmip_get_get_attributes_response_payload(const struct kmip_node *node,
					     struct kmip_node **unique_id,
					     unsigned int *num_attrs,
					     unsigned int index,
					     struct kmip_node **v2_attr);
int kmip_get_add_attribute_response_payload(const struct kmip_node *node,
					     struct kmip_node **unique_id,
					     struct kmip_node **v2_attr);
int kmip_get_modify_attribute_response_payload(const struct kmip_node *node,
					       struct kmip_node **unique_id,
					       struct kmip_node **v2_attr);
int kmip_get_set_attribute_v2_response_payload(const struct kmip_node *node,
					       struct kmip_node **unique_id);
int kmip_get_delete_attribute_response_payload(const struct kmip_node *node,
					       struct kmip_node **unique_id,
					       struct kmip_node **v2_attr);
int kmip_get_activate_response_payload(const struct kmip_node *node,
				       struct kmip_node **unique_id);
int kmip_get_destroy_response_payload(const struct kmip_node *node,
				      struct kmip_node **unique_id);
int kmip_get_archive_response_payload(const struct kmip_node *node,
				      struct kmip_node **unique_id);
int kmip_get_recover_response_payload(const struct kmip_node *node,
				      struct kmip_node **unique_id);
int kmip_get_revoke_response_payload(const struct kmip_node *node,
				     struct kmip_node **unique_id);
int kmip_get_activate_response_payload(const struct kmip_node *node,
				       struct kmip_node **unique_id);
int kmip_get_locate_response_payload(const struct kmip_node *node,
				     int32_t *located_items,
				     unsigned int *num_items,
				     unsigned int index,
				     struct kmip_node **unique_id);
int kmip_get_register_response_payload(const struct kmip_node *node,
				       struct kmip_node **unique_id,
				       unsigned int *num_attrs,
				       unsigned int attr_index,
				       struct kmip_node **attribute);
int kmip_get_get_response_payload(const struct kmip_node *node,
				     enum kmip_object_type *obj_type,
				     struct kmip_node **unique_id,
				     struct kmip_node **object);

/* Attribute related functions */
struct kmip_node *kmip_new_attributes(const struct kmip_version *version,
				      enum kmip_tag v2_tag,
				      unsigned int attrs_count,
				      struct kmip_node **v2_attrs);
struct kmip_node *kmip_new_attributes_va(const struct kmip_version *version,
					 enum kmip_tag v2_tag,
					 unsigned int attrs_count, ...);
int kmip_get_attributes(const struct kmip_node *node, unsigned int *num_attrs,
			unsigned int attr_index, struct kmip_node **attr);

struct kmip_node *kmip_new_vendor_attribute(const char *vendor_id,
					    const char *name,
					    struct kmip_node *value);
int kmip_get_vendor_attribute(const struct kmip_node *node,
			      const char **vendor_id, const char **name,
			      struct kmip_node **value);

struct kmip_node *kmip_new_attribute_reference(enum kmip_tag attr_tag,
					       const char *vendor_id,
					       const char *name);
int kmip_get_attribute_reference(const struct kmip_node *node,
				 enum kmip_tag *attr_tag,
				 const char **vendor_id, const char **name);

struct kmip_node *kmip_new_current_new_attribute(bool new_attr,
						 struct kmip_node *attr);

struct kmip_node *kmip_new_unique_identifier(const char *text_id,
					enum kmip_unique_identifier enum_id,
					int32_t int_id);
int kmip_get_unique_identifier(const struct kmip_node *node,
					   const char **text_id,
					   enum kmip_unique_identifier *enum_id,
					   int32_t *int_id);

struct kmip_node *kmip_new_name(const char *value, enum kmip_name_type type);
int kmip_get_name(const struct kmip_node *node,
		  const char **value, enum kmip_name_type *type);

struct kmip_node *kmip_new_alternative_name(const char *value,
					enum kmip_alternative_name_type type);
int kmip_get_alternative_name(const struct kmip_node *node,
			      const char **value,
			      enum kmip_alternative_name_type *type);

struct kmip_node *kmip_new_object_type(enum kmip_object_type obj_type);
int kmip_get_object_type(const struct kmip_node *node,
			       enum kmip_object_type *obj_type);

struct kmip_node *kmip_new_cryptographic_algorithm(enum kmip_crypto_algo algo);
int kmip_get_cryptographic_algorithm(const struct kmip_node *node,
				     enum kmip_crypto_algo *algo);

struct kmip_node *kmip_new_cryptographic_length(int32_t length);
int kmip_get_cryptographic_length(const struct kmip_node *node,
				  int32_t *length);

struct kmip_node *kmip_new_certificate_type(enum kmip_certificate_type type);
int kmip_get_certificate_type(const struct kmip_node *node,
			      enum kmip_certificate_type *type);

struct kmip_node *kmip_new_cryptographic_usage_mask(int32_t usage_mask);
int kmip_get_cryptographic_usage_mask(const struct kmip_node *node,
				      int32_t *usage_mask);

struct kmip_node *kmip_new_state(enum kmip_state state);
int kmip_get_state(const struct kmip_node *node, enum kmip_state *state);

struct kmip_node *kmip_new_initial_date(int64_t date);
int kmip_get_initial_date(const struct kmip_node *node, int64_t *date);

struct kmip_node *kmip_new_activation_date(int64_t date);
int kmip_get_activation_date(const struct kmip_node *node, int64_t *date);

struct kmip_node *kmip_new_deactivation_date(int64_t date);
int kmip_get_deactivation_date(const struct kmip_node *node, int64_t *date);

struct kmip_node *kmip_new_destroy_date(int64_t date);
int kmip_get_destroy_date(const struct kmip_node *node, int64_t *date);

struct kmip_node *kmip_new_compromise_date(int64_t date);
int kmip_get_compromise_date(const struct kmip_node *node, int64_t *date);

struct kmip_node *kmip_new_compromise_occurrence_date(int64_t date);
int kmip_get_compromise_occurrence_date(const struct kmip_node *node,
					int64_t *date);

struct kmip_node *kmip_new_last_change_date(int64_t date);
int kmip_get_last_change_date(const struct kmip_node *node, int64_t *date);

struct kmip_node *kmip_new_original_creation_date(int64_t date);
int kmip_get_original_creation_date(const struct kmip_node *node,
				    int64_t *date);

struct kmip_node *kmip_new_archive_date(int64_t date);
int kmip_get_archive_date(const struct kmip_node *node, int64_t *date);

struct kmip_node *kmip_new_process_start_date(int64_t date);
int kmip_get_process_start_date(const struct kmip_node *node, int64_t *date);

struct kmip_node *kmip_new_protect_stop_date(int64_t date);
int kmip_get_protect_stop_date(const struct kmip_node *node, int64_t *date);

struct kmip_node *kmip_new_cryptographic_parameters(
					const struct kmip_version *version,
					enum kmip_block_cipher_mode mode,
					enum kmip_padding_method padding,
					enum kmip_hashing_algo hash_algo,
					enum kmip_key_role_type key_role,
					enum kmip_signature_algo signature_algo,
					enum kmip_crypto_algo crypto_algo,
					bool *random_iv,
					int32_t *iv_length,
					int32_t *tag_length,
					int32_t *fixed_field_length,
					int32_t *invoc_field_length,
					int32_t *counter_length,
					int32_t *init_counter_value,
					int32_t *salt_length,
					enum kmip_mask_generator mgf,
					enum kmip_hashing_algo mgf_hash_algo,
					int32_t *trailer_field);
int kmip_get_cryptographic_parameter(const struct kmip_node *node,
				     enum kmip_block_cipher_mode *mode,
				     enum kmip_padding_method *padding,
				     enum kmip_hashing_algo *hash_algo,
				     enum kmip_key_role_type *key_role,
				     enum kmip_signature_algo *signature_algo,
				     enum kmip_crypto_algo *crypto_algo,
				     bool *random_iv,
				     int32_t *iv_length,
				     int32_t *tag_length,
				     int32_t *fixed_field_length,
				     int32_t *invoc_field_length,
				     int32_t *counter_length,
				     int32_t *init_counter_value,
				     int32_t *salt_length,
				     enum kmip_mask_generator *mgf,
				     enum kmip_hashing_algo *mgf_hash_algo,
				     int32_t *trailer_field);

struct kmip_node *kmip_new_cryptographic_domain_parameters(
					int32_t qlength,
					enum kmip_recommended_curve curve);
int kmip_get_cryptographic_domain_parameters(const struct kmip_node *node,
					     int32_t *qlength,
					enum kmip_recommended_curve *curve);

struct kmip_node *kmip_new_digital_signature_algorithm(
				enum kmip_signature_algo signature_algo);
int kmip_get_digital_signature_algorithm(const struct kmip_node *node,
				enum kmip_signature_algo *signature_algo);

struct kmip_node *kmip_new_object_group(const char *group);
int kmip_get_object_group(const struct kmip_node *node, const char **group);

struct kmip_node *kmip_new_revocation_reason(enum kmip_revoke_reason reason,
					     const char *message);
int kmip_get_revocation_reason(const struct kmip_node *node,
			       enum kmip_revoke_reason *reason,
			       const char **message);

struct kmip_node *kmip_new_contact_information(const char *contact);
int kmip_get_contact_information(const struct kmip_node *node,
					 const char **contact);

struct kmip_node *kmip_new_description(const char *description);
int kmip_get_description(const struct kmip_node *node,
			 const char **description);

struct kmip_node *kmip_new_comment(const char *comment);
int kmip_get_comment(const struct kmip_node *node, const char **comment);

struct kmip_node *kmip_new_key_format_type(enum kmip_key_format_type type);
int kmip_get_key_format_type(const struct kmip_node *node,
					  enum kmip_key_format_type *type);

struct kmip_node *kmip_new_protection_level(enum kmip_protection_level level);
int kmip_get_protection_level(const struct kmip_node *node,
			      enum kmip_protection_level *level);

struct kmip_node *kmip_new_protection_period(uint32_t period);
int kmip_get_protection_period(const struct kmip_node *node, uint32_t *period);

struct kmip_node *kmip_new_protection_storage_mask(int32_t protection_mask);
int kmip_get_protection_storage_mask(const struct kmip_node *node,
				     int32_t *protection_mask);

struct kmip_node *kmip_new_fresh(bool fresh);
int kmip_get_fresh(const struct kmip_node *node, bool *fresh);

struct kmip_node *kmip_new_key_value_present(bool present);
int kmip_get_key_value_present(const struct kmip_node *node, bool *present);

struct kmip_node *kmip_new_short_unique_identifier(
				const unsigned char *short_uid,
				uint32_t short_uid_len);
int kmip_get_short_unique_identifier(const struct kmip_node *node,
				     const unsigned char **short_uid,
				     uint32_t *short_uid_len);

struct kmip_node *kmip_new_application_specific_information(
				const char *name_space, const char *data);
int kmip_get_application_specific_information(const struct kmip_node *node,
					      const char **name_space,
					      const char **data);

struct kmip_node *kmip_new_key_value_location(const char *value,
					enum kmip_key_value_location_type type);
int kmip_get_key_value_location(const struct kmip_node *node,
		const char **value, enum kmip_key_value_location_type *type);

struct kmip_node *kmip_new_digest(enum kmip_hashing_algo hash_algo,
				  const unsigned char *digest,
				  uint32_t digest_len);
int kmip_get_digest(const struct kmip_node *node,
		    enum kmip_hashing_algo *hash_algo,
		    const unsigned char **digest, uint32_t *digest_len);

struct kmip_node *kmip_new_sensitive(bool sensitive);
int kmip_get_sensitive(const struct kmip_node *node, bool *sensitive);

struct kmip_node *kmip_new_always_sensitive(bool sensitive);
int kmip_get_always_sensitive(const struct kmip_node *node, bool *sensitive);

struct kmip_node *kmip_new_extractable(bool extractable);
int kmip_get_extractable(const struct kmip_node *node, bool *extractable);

struct kmip_node *kmip_new_never_extractable(bool extractable);
int kmip_get_never_extractable(const struct kmip_node *node, bool *extractable);

struct kmip_node *kmip_new_link(enum kmip_link_type type,
				struct kmip_node *obj_id);
int kmip_get_link(const struct kmip_node *node, enum kmip_link_type *type,
		  struct kmip_node **obj_id);

struct kmip_node *kmip_new_linked_object_identifier(const char *text_id,
					enum kmip_unique_identifier enum_id,
					int32_t int_id);
int kmip_get_linked_object_identifier(const struct kmip_node *node,
				      const char **text_id,
				      enum kmip_unique_identifier *enum_id,
				      int32_t *int_id);

struct kmip_node *kmip_new_operation_policy_name(const char *policy);
int kmip_get_operation_policy_name(const struct kmip_node *node,
				   const char **policy);

struct kmip_node *kmip_new_lease_time(uint32_t lease_time);
int kmip_get_lease_time(const struct kmip_node *node, uint32_t *lease_time);

/* Key related functions */
struct kmip_node *kmip_new_key_block(enum kmip_key_format_type format_type,
				     enum kmip_key_compression_type compr_type,
				     struct kmip_node *key_value,
				     enum kmip_crypto_algo algorithm,
				     int32_t length,
				     struct kmip_node *wrappig_data);
int kmip_get_key_block(const struct kmip_node *node,
		       enum kmip_key_format_type *format_type,
		       enum kmip_key_compression_type *compr_type,
		       struct kmip_node **key_value,
		       enum kmip_crypto_algo *algorithm,
		       int32_t *length,
		       struct kmip_node **wrappig_data);

struct kmip_node *kmip_new_key_value(const struct kmip_version *version,
				     struct kmip_node *key_material,
				     unsigned int attrs_count,
				     struct kmip_node **v2_attrs);
struct kmip_node *kmip_new_key_value_va(const struct kmip_version *version,
					struct kmip_node *key_material,
					unsigned int attrs_count, ...);
int kmip_get_key_value(const struct kmip_node *node,
		       struct kmip_node **key_material,
		       unsigned int *num_attrs, unsigned int index,
		       struct kmip_node **v2_attr);

struct kmip_node *kmip_new_key_wrapping_data(
					const struct kmip_version *version,
					enum kmip_wrapping_method wrap_method,
					struct kmip_node *encr_key_info,
					struct kmip_node *mac_sign_key_info,
					const unsigned char *mac_signature,
					uint32_t mac_signature_len,
					const unsigned char *iv_counter_nonce,
					uint32_t iv_counter_nonce_len,
					enum kmip_encoding_option encoding);
int kmip_get_key_wrapping_data(const struct kmip_node *node,
			       enum kmip_wrapping_method *wrap_method,
			       struct kmip_node **encr_key_info,
			       struct kmip_node **mac_sign_key_info,
			       const unsigned char **mac_signature,
			       uint32_t *mac_signature_len,
			       const unsigned char **iv_counter_nonce,
			       uint32_t *iv_counter_nonce_len,
			       enum kmip_encoding_option *encoding);

struct kmip_node *kmip_new_key_wrapping_specification(
					const struct kmip_version *version,
					enum kmip_wrapping_method wrap_method,
					struct kmip_node *encr_key_info,
					struct kmip_node *mac_sign_key_info,
					enum kmip_encoding_option encoding,
					unsigned int attr_name_count,
					const char **attr_names);
struct kmip_node *kmip_new_key_wrapping_specification_va(
					const struct kmip_version *version,
					enum kmip_wrapping_method wrap_method,
					struct kmip_node *encr_key_info,
					struct kmip_node *mac_sign_key_info,
					enum kmip_encoding_option encoding,
					unsigned int attr_name_count, ...);
int kmip_get_key_wrapping_specification(const struct kmip_node *node,
			       enum kmip_wrapping_method *wrap_method,
			       struct kmip_node **encr_key_info,
			       struct kmip_node **mac_sign_key_info,
			       enum kmip_encoding_option *encoding,
			       unsigned int *num_attr_names,
			       unsigned int attr_name_index,
			       const char **attr_name);

struct kmip_node *kmip_new_key_info(bool mac_sign, struct kmip_node *unique_id,
				    struct kmip_node *crypto_params);
int kmip_get_key_info(const struct kmip_node *node,
		      struct kmip_node **unique_id,
		      struct kmip_node **crypto_params);

struct kmip_node *kmip_new_transparent_symmetric_key(const unsigned char *key,
						     uint32_t key_length);
int kmip_get_transparent_symmetric_key(const struct kmip_node *node,
				       const unsigned char **key,
				       uint32_t *key_length);

struct kmip_node *kmip_new_transparent_rsa_public_key(const BIGNUM *modulus,
						      const BIGNUM *pub_exp);
int kmip_get_transparent_rsa_public_key(const struct kmip_node *node,
					const BIGNUM **modulus,
					const BIGNUM **pub_exp);

struct kmip_node *kmip_new_pkcs1_public_key(EVP_PKEY *pub_key);
int kmip_get_pkcs1_public_key(const struct kmip_node *node,
			      enum kmip_crypto_algo algo,
			      EVP_PKEY **pub_key);

struct kmip_node *kmip_new_pkcs8_public_key(EVP_PKEY *pub_key);
int kmip_get_pkcs8_public_key(const struct kmip_node *node,
			      EVP_PKEY **pub_key);

struct kmip_node *kmip_new_raw_key(const unsigned char *key, uint32_t key_len);
int kmip_get_raw_key(const struct kmip_node *node, const unsigned char **key,
		     uint32_t *key_len);

struct kmip_node *kmip_new_symmetric_key(struct kmip_node *keyblock);
int kmip_get_symmetric_key(const struct kmip_node *node,
			   struct kmip_node **keyblock);

struct kmip_node *kmip_new_public_key(struct kmip_node *keyblock);
int kmip_get_public_key(const struct kmip_node *node,
			struct kmip_node **keyblock);

/* Connection related functions */
int kmip_connection_new(const struct kmip_conn_config *config,
			struct kmip_connection **connection,
			bool debug);
int kmip_connection_perform(struct kmip_connection *connection,
			    struct kmip_node *request,
			    struct kmip_node **response,
			    bool debug);
void kmip_connection_free(struct kmip_connection *connection);
int kmip_connection_get_server_cert(const char *server,
				    enum kmip_transport transport,
				    const char *ca,
				    EVP_PKEY *client_key,
				    const char *client_cert,
				    const char *server_cert_pem,
				    const char *server_pubkey_pem,
				    const char *cert_chain_pem,
				    bool *verified,
				    bool debug);

#endif
