require 'puppet'
require 'spec_helper'
require 'puppet/provider/keystone_identity_provider/openstack'

describe Puppet::Type.type(:keystone_identity_provider).provider(:openstack) do
  let(:set_env) do
    ENV['OS_USERNAME']     = 'test'
    ENV['OS_PASSWORD']     = 'abc123'
    ENV['OS_SYSTEM_SCOPE'] = 'all'
    ENV['OS_AUTH_URL']     = 'http://127.0.0.1:5000/v3'
  end

  let(:id_provider_attrs) do
    {
      :name         => 'idp_one',
      :enabled      => true,
      :description  => 'Nice id provider',
      :remote_ids   => ['entityid_idp1', 'http://entityid_idp2/saml/meta', 3],
      :ensure       => :present
    }
  end

  let(:resource) do
    Puppet::Type::Keystone_identity_provider.new(id_provider_attrs)
  end

  let(:provider) { described_class.new(resource) }

  before(:example) { set_env }
  describe '#create success' do
    it 'creates an identity provider' do
      expect(described_class).to receive(:openstack)
        .with(
              'identity provider', 'create',
              '--format', 'shell', [
                '--remote-id', 'entityid_idp1',
                '--remote-id', 'http://entityid_idp2/saml/meta',
                '--remote-id', '3',
                '--enable',
                '--description', 'Nice id provider',
                'idp_one'
              ]
             )
        .exactly(1).times
        .and_return(
        <<-EOR
description="Nice id provider"
enabled="True"
id="idp_one"
remote_ids="entityid_idp1, http://entityid_idp2/saml/meta, 3"
EOR
      )
      provider.create
      expect(provider.exists?).to be_truthy
    end
  end
  describe '#create failure' do
    it 'fails with an helpful message when hitting remote-id duplicate.' do
      expect(described_class).to receive(:openstack)
        .with(
              'identity provider', 'create',
              '--format', 'shell', [
                '--remote-id', 'entityid_idp1',
                '--remote-id', 'http://entityid_idp2/saml/meta',
                '--remote-id', '3',
                '--enable',
                '--description', 'Nice id provider',
                'idp_one'
              ]
             )
        .exactly(1).times
        .and_raise(Puppet::ExecutionFailure,
                   'openstack Conflict occurred attempting to' \
                   ' store identity_provider')
      expect { provider.create }
        .to raise_error(Puppet::Error::OpenstackDuplicateRemoteId)
    end
  end

  describe '#create with a remote-id-file' do
    let(:id_provider_attrs) do
      {
        :name           => 'idp_one',
        :enabled        => true,
        :description    => 'Nice id provider',
        :remote_id_file => '/tmp/remoteids',
        :ensure         => :present
      }
    end
    it 'create a resource whit remote id in a file' do
      expect(described_class).to receive(:openstack)
        .with(
              'identity provider', 'create',
              '--format', 'shell', [
                '--remote-id-file', '/tmp/remoteids',
                '--enable',
                '--description', 'Nice id provider',
                'idp_one'
              ]
             )
        .exactly(1).times
        .and_return(
        <<-EOR
description="Nice id provider"
enabled="True"
id="idp_one"
remote_ids="entityid_idp1, http://entityid_idp2/saml/meta, 3"
EOR
      )
      provider.create
      expect(provider.exists?).to be_truthy

    end
  end

  describe '#destroy' do
    it 'destroy an identity provider' do
      provider.instance_variable_get('@property_hash')[:id] = 'idp_one'
      expect(described_class).to receive(:openstack)
        .with(
              'identity provider', 'delete', 'idp_one'
             )
      provider.destroy
      expect(provider.exists?).to be_falsy
    end
  end

  describe '#instances' do
    it 'finds every identity provider' do
      expect(described_class).to receive(:openstack)
        .with(
              'identity provider', 'list',
              '--quiet', '--format', 'csv', []
             )
        .exactly(1).times
        .and_return(
        <<-EOR
"ID","Enabled","Description"
"idp_one",True,""
"idp_two",False,"Idp two description"
EOR
      )
      expect(described_class).to receive(:openstack)
        .with(
              'identity provider', 'show',
              '--format', 'shell', 'idp_one'
             )
        .exactly(1).times
        .and_return(
        <<-EOR
description="None"
enabled="True"
id="idp_one"
remote_ids="entityid_idp1, http://entityid_idp2/saml/meta, 3"
EOR
      )
      expect(described_class).to receive(:openstack)
        .with(
              'identity provider', 'show',
              '--format', 'shell', 'idp_two'
             )
        .exactly(1).times
        .and_return(
        <<-EOR
description="Idp two description"
enabled="False"
id="idp_two"
remote_ids=""
EOR
      )
      instances =
        Puppet::Type::Keystone_identity_provider::ProviderOpenstack.instances
      expect(instances.count).to eq(2)
      expect(instances[0].description).to be_empty
      expect(instances[1].enabled).to be_falsy
    end
  end

  describe '#update' do
    context 'remote_ids' do
      it 'changes the remote_ids' do
        expect(provider).to receive(:id).and_return('1234')
        expect(described_class).to receive(:openstack)
          .with(
                'identity provider', 'set',
                [
                  '--remote-id', 'entityid_idp1',
                  '--remote-id', 'http://entityid_idp2/saml/meta',
                  '1234'
                ]
               )
          .exactly(1).times
        provider.remote_ids = ['entityid_idp1', 'http://entityid_idp2/saml/meta']
      end
    end
    context 'with remote_id_file' do
      it 'changes the remote_id_file' do
        expect(provider).to receive(:id).and_return('1234')
        expect(described_class).to receive(:openstack)
          .with(
                'identity provider', 'set',
                ['--remote-id-file', '/tmp/new_file', '1234']
               )
          .exactly(1).times
        provider.remote_id_file = '/tmp/new_file'
      end
    end
    context 'enabled' do
      it 'changes the enable to true' do
        expect(provider).to receive(:id).and_return('1234')
        expect(described_class).to receive(:openstack)
          .with(
                'identity provider', 'set',
                ['--enable', '1234']
               )
          .exactly(1).times
        provider.enabled = :true
      end
      it 'changes the enable to false' do
        expect(provider).to receive(:id).and_return('1234')
        expect(described_class).to receive(:openstack)
          .with(
                'identity provider', 'set',
                ['--disable', '1234']
               )
          .exactly(1).times
        provider.enabled = :false
      end
    end
  end

  describe '#prefetch' do
    let(:resources_catalog) { { 'idp_one' => provider } }
    let(:found_resource) do
      existing = described_class.new
      existing.instance_variable_set('@property_hash',
        :name        => 'idp_one',
        :id          => 'idp_one',
        :description => '',
        :enabled     => true,
        :remote_ids  => [
          'entityid_idp1',
          'http://entityid_idp2/saml/meta',
          '3'],
        :ensure      => :present
      )
      existing
    end
    it 'fill the resource with the right provider' do
      expect(described_class).to receive(:instances)
        .exactly(1).times
        .and_return([found_resource])
      expect(resources_catalog['idp_one'].provider).to be_absent
      described_class.prefetch(resources_catalog)
      expect(resources_catalog['idp_one'].provider).not_to be_absent
    end
  end

  describe '#clean_remote_ids' do
    let(:remote_ids) do
      [
        "http://remoteid?id=idp_one&name=ldap, http://remoteid_2?id='idp'",
        ['http://remoteid?id=idp_one&name=ldap', "http://remoteid_2?id='idp'"]
      ]
    end
    it 'should handle the new output' do
      expect(described_class.clean_remote_ids(remote_ids[0])).to eq(remote_ids[1])
    end
  end
end
