/***************************************************************************
    Copyright (C) 2011 Robby Stephenson <robby@periapsis.org>
 ***************************************************************************/

/***************************************************************************
 *                                                                         *
 *   This program is free software; you can redistribute it and/or         *
 *   modify it under the terms of the GNU General Public License as        *
 *   published by the Free Software Foundation; either version 2 of        *
 *   the License or (at your option) version 3 or any later version        *
 *   accepted by the membership of KDE e.V. (or its successor approved     *
 *   by the membership of KDE e.V.), which shall act as a proxy            *
 *   defined in Section 14 of version 3 of the license.                    *
 *                                                                         *
 *   This program is distributed in the hope that it will be useful,       *
 *   but WITHOUT ANY WARRANTY; without even the implied warranty of        *
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the         *
 *   GNU General Public License for more details.                          *
 *                                                                         *
 *   You should have received a copy of the GNU General Public License     *
 *   along with this program.  If not, see <http://www.gnu.org/licenses/>. *
 *                                                                         *
 ***************************************************************************/

#undef QT_NO_CAST_FROM_ASCII

#include "filtertest.h"

#include "../filter.h"
#include "../entry.h"
#include "../collections/bookcollection.h"

#include <QTest>

QTEST_GUILESS_MAIN( FilterTest )

void FilterTest::initTestCase() {
}

void FilterTest::testFilter() {
  Tellico::Data::CollPtr coll(new Tellico::Data::Collection(true, QStringLiteral("TestCollection")));
  Tellico::Data::EntryPtr entry(new Tellico::Data::Entry(coll));
  entry->setField(QStringLiteral("title"), QStringLiteral("Star Wars"));

  Tellico::FilterRule* rule1 = new Tellico::FilterRule(QStringLiteral("title"),
                                                       QStringLiteral("Star Wars"),
                                                       Tellico::FilterRule::FuncEquals);
  QCOMPARE(rule1->fieldName(), QStringLiteral("title"));
  QCOMPARE(rule1->pattern(), QStringLiteral("Star Wars"));
  QCOMPARE(rule1->function(), Tellico::FilterRule::FuncEquals);

  Tellico::Filter filter(Tellico::Filter::MatchAny);
  filter.append(rule1);
  QVERIFY(filter.matches(entry));
  rule1->setFunction(Tellico::FilterRule::FuncNotEquals);
  QVERIFY(!filter.matches(entry));

  rule1->setFunction(Tellico::FilterRule::FuncContains);
  QVERIFY(filter.matches(entry));

  rule1->setFieldName(QStringLiteral("author"));
  QVERIFY(!filter.matches(entry));
  rule1->setFunction(Tellico::FilterRule::FuncNotContains);
  QVERIFY(filter.matches(entry));

  rule1->setFieldName(QString());
  rule1->setFunction(Tellico::FilterRule::FuncEquals);
  QVERIFY(filter.matches(entry));

  Tellico::FilterRule* rule2 = new Tellico::FilterRule(QStringLiteral("title"),
                                                       QStringLiteral("Star"),
                                                       Tellico::FilterRule::FuncEquals);
  filter.clear();
  filter.append(rule2);
  QVERIFY(!filter.matches(entry));

  rule2->setFunction(Tellico::FilterRule::FuncContains);
  QVERIFY(filter.matches(entry));
  rule2->setFunction(Tellico::FilterRule::FuncNotContains);
  QVERIFY(!filter.matches(entry));

  rule2->setFieldName(QStringLiteral("author"));
  rule2->setFunction(Tellico::FilterRule::FuncContains);
  QVERIFY(!filter.matches(entry));

  rule2->setFieldName(QString());
  QVERIFY(filter.matches(entry));

  Tellico::FilterRule* rule3 = new Tellico::FilterRule(QStringLiteral("title"),
                                                       QStringLiteral("Sta[rt]"),
                                                       Tellico::FilterRule::FuncRegExp);
  QCOMPARE(rule3->pattern(), QStringLiteral("Sta[rt]"));
  filter.clear();
  filter.append(rule3);
  QVERIFY(filter.matches(entry));

  rule3->setFunction(Tellico::FilterRule::FuncNotRegExp);
  QVERIFY(!filter.matches(entry));

  rule3->setFieldName(QStringLiteral("author"));
  QVERIFY(filter.matches(entry));

  rule3->setFieldName(QString());
  rule3->setFunction(Tellico::FilterRule::FuncRegExp);
  QVERIFY(filter.matches(entry));

  entry->setField(QStringLiteral("title"), QStringLiteral("Tmavomodrý Svět"));

  Tellico::FilterRule* rule4 = new Tellico::FilterRule(QStringLiteral("title"),
                                                       QStringLiteral("Tmavomodrý Svět"),
                                                       Tellico::FilterRule::FuncEquals);
  filter.clear();
  filter.append(rule4);
  QVERIFY(filter.matches(entry));

  rule4->setFunction(Tellico::FilterRule::FuncContains);
  QVERIFY(filter.matches(entry));

  rule4->setFunction(Tellico::FilterRule::FuncRegExp);
  QVERIFY(filter.matches(entry));

  Tellico::FilterRule* rule5 = new Tellico::FilterRule(QStringLiteral("title"),
                                                       QLatin1String("Tmavomodry Svet"),
                                                       Tellico::FilterRule::FuncEquals);

  filter.clear();
  filter.append(rule5);
  QVERIFY(!filter.matches(entry));

  rule5->setFunction(Tellico::FilterRule::FuncContains);
  QVERIFY(filter.matches(entry));

  rule5->setFunction(Tellico::FilterRule::FuncRegExp);
  QVERIFY(!filter.matches(entry));

  Tellico::Data::FieldPtr date(new Tellico::Data::Field(QStringLiteral("date"),
                                                        QStringLiteral("Date"),
                                                        Tellico::Data::Field::Date));
  coll->addField(date);

  Tellico::FilterRule* rule6 = new Tellico::FilterRule(QStringLiteral("date"),
                                                       QStringLiteral("2011-01-24"),
                                                       Tellico::FilterRule::FuncAfter);
  QCOMPARE(rule6->pattern(), QStringLiteral("2011-01-24"));
  filter.clear();
  filter.append(rule6);
  // test Bug 361625
  entry->setField(QStringLiteral("date"), QStringLiteral("2011-1-25"));
  QVERIFY(filter.matches(entry));
  entry->setField(QStringLiteral("date"), QStringLiteral("2011-01-25"));
  QVERIFY(filter.matches(entry));

  rule6->setFunction(Tellico::FilterRule::FuncBefore);
  QVERIFY(!filter.matches(entry));

  // check that a date match is neither before or after
  entry->setField(QStringLiteral("date"), rule6->pattern());
  rule6->setFunction(Tellico::FilterRule::FuncAfter);
  QVERIFY(!filter.matches(entry));
  rule6->setFunction(Tellico::FilterRule::FuncBefore);
  QVERIFY(!filter.matches(entry));

  // check that an invalid date never matches
  entry->setField(QStringLiteral("date"), QStringLiteral("test"));
  rule6->setFunction(Tellico::FilterRule::FuncAfter);
  QVERIFY(!filter.matches(entry));
  rule6->setFunction(Tellico::FilterRule::FuncBefore);
  QVERIFY(!filter.matches(entry));

  Tellico::Data::FieldPtr number(new Tellico::Data::Field(QStringLiteral("number"),
                                                          QStringLiteral("Number"),
                                                          Tellico::Data::Field::Number));
  coll->addField(number);
  entry->setField(QStringLiteral("number"), QStringLiteral("3"));

  Tellico::FilterRule* rule7 = new Tellico::FilterRule(QStringLiteral("number"),
                                                       QStringLiteral("5.0"),
                                                       Tellico::FilterRule::FuncLess);
  QCOMPARE(rule7->pattern(), QStringLiteral("5.0"));
  filter.clear();
  filter.append(rule7);
  QVERIFY(filter.matches(entry));

  rule7->setFunction(Tellico::FilterRule::FuncGreater);
  QVERIFY(!filter.matches(entry));

  entry->setField(QStringLiteral("number"), QStringLiteral("6"));
  QVERIFY(filter.matches(entry));

  // check that a rating can use greater than
  Tellico::Data::FieldPtr rating(new Tellico::Data::Field(QStringLiteral("rating"),
                                                          QStringLiteral("Rating"),
                                                          Tellico::Data::Field::Rating));
  coll->addField(rating);
  entry->setField(QStringLiteral("rating"), QStringLiteral("3"));

  Tellico::FilterRule* rule8 = new Tellico::FilterRule(QStringLiteral("rating"),
                                                       QStringLiteral("2.0"),
                                                       Tellico::FilterRule::FuncGreater);
  QCOMPARE(rule8->pattern(), QStringLiteral("2.0"));
  filter.clear();
  filter.append(rule8);
  QVERIFY(filter.matches(entry));

  rule8->setFunction(Tellico::FilterRule::FuncLess);
  QVERIFY(!filter.matches(entry));

  entry->setField(QStringLiteral("rating"), QStringLiteral("1"));
  QVERIFY(filter.matches(entry));
}

void FilterTest::testGroupViewFilter() {
  // ideally, I'd instantiate a GroupView object and test that, but it's tough with all the dependencies
  // so this code is identical to what is in Tellico::GroupView::slotFilterGroup()
  Tellico::Data::CollPtr coll(new Tellico::Data::BookCollection(true, QStringLiteral("TestCollection")));
  Tellico::Data::EntryPtr entry1(new Tellico::Data::Entry(coll));
  entry1->setField(QStringLiteral("author"), QStringLiteral("John Author"));
  Tellico::Data::EntryPtr entry2(new Tellico::Data::Entry(coll));
  entry2->setField(QStringLiteral("author"), QStringLiteral("John Q. Author"));
  Tellico::Data::EntryPtr entry3(new Tellico::Data::Entry(coll));
  entry3->setField(QStringLiteral("author"), QStringLiteral("John Author") +
                                            Tellico::FieldFormat::delimiterString() +
                                            QStringLiteral("James Author"));
  Tellico::Data::EntryPtr entry4(new Tellico::Data::Entry(coll));
  entry4->setField(QStringLiteral("author"), QStringLiteral("James Author") +
                                            Tellico::FieldFormat::delimiterString() +
                                            QStringLiteral("John Author"));
  Tellico::Data::EntryPtr entry5(new Tellico::Data::Entry(coll));
  entry5->setField(QStringLiteral("author"), QStringLiteral("James Author") +
                                            Tellico::FieldFormat::delimiterString() +
                                            QStringLiteral("John Q. Author"));

  QString pattern(entry1->formattedField(QStringLiteral("author")));
  // the filter should match all since it was the initial way the group view filter was constructed
  Tellico::Filter filter1(Tellico::Filter::MatchAny);
  filter1.append(new Tellico::FilterRule(QStringLiteral("author"), pattern, Tellico::FilterRule::FuncContains));
  QVERIFY(filter1.matches(entry1));
  QVERIFY(filter1.matches(entry2));
  QVERIFY(filter1.matches(entry3));
  QVERIFY(filter1.matches(entry4));
  QVERIFY(filter1.matches(entry5));

  QString rxPattern(QStringLiteral("(^|;\\s)") + pattern + QStringLiteral("($|;)"));
  // the filter should match entry1, entry3, and entry 4 but not entry2 or entry5
  Tellico::Filter filter2(Tellico::Filter::MatchAny);
  filter2.append(new Tellico::FilterRule(QStringLiteral("author"), rxPattern, Tellico::FilterRule::FuncRegExp));
  QVERIFY(filter2.matches(entry1));
  QVERIFY(!filter2.matches(entry2)); // does not match
  QVERIFY(filter2.matches(entry3));
  QVERIFY(filter2.matches(entry4));
  QVERIFY(!filter2.matches(entry5));
}
